/* ----------------------------------------------------------------------   
* Copyright (C) 2010 ARM Limited. All rights reserved.   
*   
* $Date:        15. July 2011  
* $Revision: 	V1.0.10  
*   
* Project: 	    CMSIS DSP Library   
* Title:	    arm_dct4_init_f32.c   
*   
* Description:	Initialization function of DCT-4 & IDCT4 F32   
*   
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
*  
* Version 1.0.10 2011/7/15 
*    Big Endian support added and Merged M0 and M3/M4 Source code.  
*   
* Version 1.0.3 2010/11/29  
*    Re-organized the CMSIS folders and updated documentation.   
*    
* Version 1.0.2 2010/11/11   
*    Documentation updated.    
*   
* Version 1.0.1 2010/10/05    
*    Production release and review comments incorporated.   
*   
* Version 1.0.0 2010/09/20    
*    Production release and review comments incorporated.   
* -------------------------------------------------------------------- */


#include "arm_math.h"

/**   
 * @ingroup groupTransforms   
 */

/**   
 * @addtogroup DCT4_IDCT4   
 * @{   
 */

/*   
* @brief  Weights Table   
*/

/**   
* \par   
* Weights tables are generated using the formula : <pre>weights[n] = e^(-j*n*pi/(2*N))</pre>   
* \par   
* C command to generate the table   
* <pre>   
* for(i = 0; i< N; i++)   
* {   
*    weights[2*i]= cos(i*c);   
*    weights[(2*i)+1]= -sin(i * c);   
* } </pre>   
* \par   
* Where <code>N</code> is the Number of weights to be calculated and <code>c</code> is <code>pi/(2*N)</code>   
* \par   
* In the tables below the real and imaginary values are placed alternatively, hence the   
* array length is <code>2*N</code>.   
*/

static const float32_t Weights_128[256] = {
  1.000000000000000000f, 0.000000000000000000f, 0.999924701839144500f,
  -0.012271538285719925f,
  0.999698818696204250f, -0.024541228522912288f, 0.999322384588349540f,
  -0.036807222941358832f,
  0.998795456205172410f, -0.049067674327418015f, 0.998118112900149180f,
  -0.061320736302208578f,
  0.997290456678690210f, -0.073564563599667426f, 0.996312612182778000f,
  -0.085797312344439894f,
  0.995184726672196930f, -0.098017140329560604f, 0.993906970002356060f,
  -0.110222207293883060f,
  0.992479534598709970f, -0.122410675199216200f, 0.990902635427780010f,
  -0.134580708507126170f,
  0.989176509964781010f, -0.146730474455361750f, 0.987301418157858430f,
  -0.158858143333861450f,
  0.985277642388941220f, -0.170961888760301220f, 0.983105487431216290f,
  -0.183039887955140950f,
  0.980785280403230430f, -0.195090322016128250f, 0.978317370719627650f,
  -0.207111376192218560f,
  0.975702130038528570f, -0.219101240156869800f, 0.972939952205560180f,
  -0.231058108280671110f,
  0.970031253194543970f, -0.242980179903263870f, 0.966976471044852070f,
  -0.254865659604514570f,
  0.963776065795439840f, -0.266712757474898370f, 0.960430519415565790f,
  -0.278519689385053060f,
  0.956940335732208820f, -0.290284677254462330f, 0.953306040354193860f,
  -0.302005949319228080f,
  0.949528180593036670f, -0.313681740398891520f, 0.945607325380521280f,
  -0.325310292162262930f,
  0.941544065183020810f, -0.336889853392220050f, 0.937339011912574960f,
  -0.348418680249434560f,
  0.932992798834738960f, -0.359895036534988110f, 0.928506080473215590f,
  -0.371317193951837540f,
  0.923879532511286740f, -0.382683432365089780f, 0.919113851690057770f,
  -0.393992040061048100f,
  0.914209755703530690f, -0.405241314004989860f, 0.909167983090522380f,
  -0.416429560097637150f,
  0.903989293123443340f, -0.427555093430282080f, 0.898674465693953820f,
  -0.438616238538527660f,
  0.893224301195515320f, -0.449611329654606540f, 0.887639620402853930f,
  -0.460538710958240010f,
  0.881921264348355050f, -0.471396736825997640f, 0.876070094195406600f,
  -0.482183772079122720f,
  0.870086991108711460f, -0.492898192229784040f, 0.863972856121586810f,
  -0.503538383725717580f,
  0.857728610000272120f, -0.514102744193221660f, 0.851355193105265200f,
  -0.524589682678468950f,
  0.844853565249707120f, -0.534997619887097150f, 0.838224705554838080f,
  -0.545324988422046460f,
  0.831469612302545240f, -0.555570233019602180f, 0.824589302785025290f,
  -0.565731810783613120f,
  0.817584813151583710f, -0.575808191417845340f, 0.810457198252594770f,
  -0.585797857456438860f,
  0.803207531480644940f, -0.595699304492433360f, 0.795836904608883570f,
  -0.605511041404325550f,
  0.788346427626606340f, -0.615231590580626820f, 0.780737228572094490f,
  -0.624859488142386340f,
  0.773010453362736990f, -0.634393284163645490f, 0.765167265622458960f,
  -0.643831542889791390f,
  0.757208846506484570f, -0.653172842953776760f, 0.749136394523459370f,
  -0.662415777590171780f,
  0.740951125354959110f, -0.671558954847018330f, 0.732654271672412820f,
  -0.680600997795453020f,
  0.724247082951467000f, -0.689540544737066830f, 0.715730825283818590f,
  -0.698376249408972920f,
  0.707106781186547570f, -0.707106781186547460f, 0.698376249408972920f,
  -0.715730825283818590f,
  0.689540544737066940f, -0.724247082951466890f, 0.680600997795453130f,
  -0.732654271672412820f,
  0.671558954847018330f, -0.740951125354959110f, 0.662415777590171780f,
  -0.749136394523459260f,
  0.653172842953776760f, -0.757208846506484460f, 0.643831542889791500f,
  -0.765167265622458960f,
  0.634393284163645490f, -0.773010453362736990f, 0.624859488142386450f,
  -0.780737228572094380f,
  0.615231590580626820f, -0.788346427626606230f, 0.605511041404325550f,
  -0.795836904608883460f,
  0.595699304492433470f, -0.803207531480644830f, 0.585797857456438860f,
  -0.810457198252594770f,
  0.575808191417845340f, -0.817584813151583710f, 0.565731810783613230f,
  -0.824589302785025290f,
  0.555570233019602290f, -0.831469612302545240f, 0.545324988422046460f,
  -0.838224705554837970f,
  0.534997619887097260f, -0.844853565249707010f, 0.524589682678468840f,
  -0.851355193105265200f,
  0.514102744193221660f, -0.857728610000272120f, 0.503538383725717580f,
  -0.863972856121586700f,
  0.492898192229784090f, -0.870086991108711350f, 0.482183772079122830f,
  -0.876070094195406600f,
  0.471396736825997810f, -0.881921264348354940f, 0.460538710958240010f,
  -0.887639620402853930f,
  0.449611329654606600f, -0.893224301195515320f, 0.438616238538527710f,
  -0.898674465693953820f,
  0.427555093430282200f, -0.903989293123443340f, 0.416429560097637320f,
  -0.909167983090522270f,
  0.405241314004989860f, -0.914209755703530690f, 0.393992040061048100f,
  -0.919113851690057770f,
  0.382683432365089840f, -0.923879532511286740f, 0.371317193951837600f,
  -0.928506080473215480f,
  0.359895036534988280f, -0.932992798834738850f, 0.348418680249434510f,
  -0.937339011912574960f,
  0.336889853392220050f, -0.941544065183020810f, 0.325310292162262980f,
  -0.945607325380521280f,
  0.313681740398891570f, -0.949528180593036670f, 0.302005949319228200f,
  -0.953306040354193750f,
  0.290284677254462330f, -0.956940335732208940f, 0.278519689385053060f,
  -0.960430519415565790f,
  0.266712757474898420f, -0.963776065795439840f, 0.254865659604514630f,
  -0.966976471044852070f,
  0.242980179903263980f, -0.970031253194543970f, 0.231058108280671280f,
  -0.972939952205560070f,
  0.219101240156869770f, -0.975702130038528570f, 0.207111376192218560f,
  -0.978317370719627650f,
  0.195090322016128330f, -0.980785280403230430f, 0.183039887955141060f,
  -0.983105487431216290f,
  0.170961888760301360f, -0.985277642388941220f, 0.158858143333861390f,
  -0.987301418157858430f,
  0.146730474455361750f, -0.989176509964781010f, 0.134580708507126220f,
  -0.990902635427780010f,
  0.122410675199216280f, -0.992479534598709970f, 0.110222207293883180f,
  -0.993906970002356060f,
  0.098017140329560770f, -0.995184726672196820f, 0.085797312344439880f,
  -0.996312612182778000f,
  0.073564563599667454f, -0.997290456678690210f, 0.061320736302208648f,
  -0.998118112900149180f,
  0.049067674327418126f, -0.998795456205172410f, 0.036807222941358991f,
  -0.999322384588349540f,
  0.024541228522912264f, -0.999698818696204250f, 0.012271538285719944f,
  -0.999924701839144500f
};

static const float32_t Weights_512[1024] = {
  1.000000000000000000f, 0.000000000000000000f, 0.999995293809576190f,
  -0.003067956762965976f,
  0.999981175282601110f, -0.006135884649154475f, 0.999957644551963900f,
  -0.009203754782059819f,
  0.999924701839144500f, -0.012271538285719925f, 0.999882347454212560f,
  -0.015339206284988100f,
  0.999830581795823400f, -0.018406729905804820f, 0.999769405351215280f,
  -0.021474080275469508f,
  0.999698818696204250f, -0.024541228522912288f, 0.999618822495178640f,
  -0.027608145778965740f,
  0.999529417501093140f, -0.030674803176636626f, 0.999430604555461730f,
  -0.033741171851377580f,
  0.999322384588349540f, -0.036807222941358832f, 0.999204758618363890f,
  -0.039872927587739811f,
  0.999077727752645360f, -0.042938256934940820f, 0.998941293186856870f,
  -0.046003182130914623f,
  0.998795456205172410f, -0.049067674327418015f, 0.998640218180265270f,
  -0.052131704680283324f,
  0.998475580573294770f, -0.055195244349689934f, 0.998301544933892890f,
  -0.058258264500435752f,
  0.998118112900149180f, -0.061320736302208578f, 0.997925286198596000f,
  -0.064382630929857465f,
  0.997723066644191640f, -0.067443919563664051f, 0.997511456140303450f,
  -0.070504573389613856f,
  0.997290456678690210f, -0.073564563599667426f, 0.997060070339482960f,
  -0.076623861392031492f,
  0.996820299291165670f, -0.079682437971430126f, 0.996571145790554840f,
  -0.082740264549375692f,
  0.996312612182778000f, -0.085797312344439894f, 0.996044700901251970f,
  -0.088853552582524600f,
  0.995767414467659820f, -0.091908956497132724f, 0.995480755491926940f,
  -0.094963495329638992f,
  0.995184726672196930f, -0.098017140329560604f, 0.994879330794805620f,
  -0.101069862754827820f,
  0.994564570734255420f, -0.104121633872054590f, 0.994240449453187900f,
  -0.107172424956808840f,
  0.993906970002356060f, -0.110222207293883060f, 0.993564135520595300f,
  -0.113270952177564350f,
  0.993211949234794500f, -0.116318630911904750f, 0.992850414459865100f,
  -0.119365214810991350f,
  0.992479534598709970f, -0.122410675199216200f, 0.992099313142191800f,
  -0.125454983411546230f,
  0.991709753669099530f, -0.128498110793793170f, 0.991310859846115440f,
  -0.131540028702883120f,
  0.990902635427780010f, -0.134580708507126170f, 0.990485084256457090f,
  -0.137620121586486040f,
  0.990058210262297120f, -0.140658239332849210f, 0.989622017463200890f,
  -0.143695033150294470f,
  0.989176509964781010f, -0.146730474455361750f, 0.988721691960323780f,
  -0.149764534677321510f,
  0.988257567730749460f, -0.152797185258443440f, 0.987784141644572180f,
  -0.155828397654265230f,
  0.987301418157858430f, -0.158858143333861450f, 0.986809401814185530f,
  -0.161886393780111830f,
  0.986308097244598670f, -0.164913120489969890f, 0.985797509167567480f,
  -0.167938294974731170f,
  0.985277642388941220f, -0.170961888760301220f, 0.984748501801904210f,
  -0.173983873387463820f,
  0.984210092386929030f, -0.177004220412148750f, 0.983662419211730250f,
  -0.180022901405699510f,
  0.983105487431216290f, -0.183039887955140950f, 0.982539302287441240f,
  -0.186055151663446630f,
  0.981963869109555240f, -0.189068664149806190f, 0.981379193313754560f,
  -0.192080397049892440f,
  0.980785280403230430f, -0.195090322016128250f, 0.980182135968117430f,
  -0.198098410717953560f,
  0.979569765685440520f, -0.201104634842091900f, 0.978948175319062200f,
  -0.204108966092816870f,
  0.978317370719627650f, -0.207111376192218560f, 0.977677357824509930f,
  -0.210111836880469610f,
  0.977028142657754390f, -0.213110319916091360f, 0.976369731330021140f,
  -0.216106797076219520f,
  0.975702130038528570f, -0.219101240156869800f, 0.975025345066994120f,
  -0.222093620973203510f,
  0.974339382785575860f, -0.225083911359792830f, 0.973644249650811980f,
  -0.228072083170885730f,
  0.972939952205560180f, -0.231058108280671110f, 0.972226497078936270f,
  -0.234041958583543430f,
  0.971503890986251780f, -0.237023605994367200f, 0.970772140728950350f,
  -0.240003022448741500f,
  0.970031253194543970f, -0.242980179903263870f, 0.969281235356548530f,
  -0.245955050335794590f,
  0.968522094274417380f, -0.248927605745720150f, 0.967753837093475510f,
  -0.251897818154216970f,
  0.966976471044852070f, -0.254865659604514570f, 0.966190003445412500f,
  -0.257831102162158990f,
  0.965394441697689400f, -0.260794117915275510f, 0.964589793289812760f,
  -0.263754678974831350f,
  0.963776065795439840f, -0.266712757474898370f, 0.962953266873683880f,
  -0.269668325572915090f,
  0.962121404269041580f, -0.272621355449948980f, 0.961280485811320640f,
  -0.275571819310958140f,
  0.960430519415565790f, -0.278519689385053060f, 0.959571513081984520f,
  -0.281464937925757940f,
  0.958703474895871600f, -0.284407537211271880f, 0.957826413027532910f,
  -0.287347459544729510f,
  0.956940335732208820f, -0.290284677254462330f, 0.956045251349996410f,
  -0.293219162694258630f,
  0.955141168305770780f, -0.296150888243623790f, 0.954228095109105670f,
  -0.299079826308040480f,
  0.953306040354193860f, -0.302005949319228080f, 0.952375012719765880f,
  -0.304929229735402370f,
  0.951435020969008340f, -0.307849640041534870f, 0.950486073949481700f,
  -0.310767152749611470f,
  0.949528180593036670f, -0.313681740398891520f, 0.948561349915730270f,
  -0.316593375556165850f,
  0.947585591017741090f, -0.319502030816015690f, 0.946600913083283530f,
  -0.322407678801069850f,
  0.945607325380521280f, -0.325310292162262930f, 0.944604837261480260f,
  -0.328209843579092500f,
  0.943593458161960390f, -0.331106305759876430f, 0.942573197601446870f,
  -0.333999651442009380f,
  0.941544065183020810f, -0.336889853392220050f, 0.940506070593268300f,
  -0.339776884406826850f,
  0.939459223602189920f, -0.342660717311994380f, 0.938403534063108060f,
  -0.345541324963989090f,
  0.937339011912574960f, -0.348418680249434560f, 0.936265667170278260f,
  -0.351292756085567090f,
  0.935183509938947610f, -0.354163525420490340f, 0.934092550404258980f,
  -0.357030961233429980f,
  0.932992798834738960f, -0.359895036534988110f, 0.931884265581668150f,
  -0.362755724367397230f,
  0.930766961078983710f, -0.365612997804773850f, 0.929640895843181330f,
  -0.368466829953372320f,
  0.928506080473215590f, -0.371317193951837540f, 0.927362525650401110f,
  -0.374164062971457930f,
  0.926210242138311380f, -0.377007410216418260f, 0.925049240782677580f,
  -0.379847208924051160f,
  0.923879532511286740f, -0.382683432365089780f, 0.922701128333878630f,
  -0.385516053843918850f,
  0.921514039342042010f, -0.388345046698826250f, 0.920318276709110590f,
  -0.391170384302253870f,
  0.919113851690057770f, -0.393992040061048100f, 0.917900775621390500f,
  -0.396809987416710310f,
  0.916679059921042700f, -0.399624199845646790f, 0.915448716088267830f,
  -0.402434650859418430f,
  0.914209755703530690f, -0.405241314004989860f, 0.912962190428398210f,
  -0.408044162864978690f,
  0.911706032005429880f, -0.410843171057903910f, 0.910441292258067250f,
  -0.413638312238434500f,
  0.909167983090522380f, -0.416429560097637150f, 0.907886116487666260f,
  -0.419216888363223910f,
  0.906595704514915330f, -0.422000270799799680f, 0.905296759318118820f,
  -0.424779681209108810f,
  0.903989293123443340f, -0.427555093430282080f, 0.902673318237258830f,
  -0.430326481340082610f,
  0.901348847046022030f, -0.433093818853151960f, 0.900015892016160280f,
  -0.435857079922255470f,
  0.898674465693953820f, -0.438616238538527660f, 0.897324580705418320f,
  -0.441371268731716670f,
  0.895966249756185220f, -0.444122144570429200f, 0.894599485631382700f,
  -0.446868840162374160f,
  0.893224301195515320f, -0.449611329654606540f, 0.891840709392342720f,
  -0.452349587233770890f,
  0.890448723244757880f, -0.455083587126343840f, 0.889048355854664570f,
  -0.457813303598877170f,
  0.887639620402853930f, -0.460538710958240010f, 0.886222530148880640f,
  -0.463259783551860150f,
  0.884797098430937790f, -0.465976495767966180f, 0.883363338665731580f,
  -0.468688822035827900f,
  0.881921264348355050f, -0.471396736825997640f, 0.880470889052160750f,
  -0.474100214650549970f,
  0.879012226428633530f, -0.476799230063322090f, 0.877545290207261350f,
  -0.479493757660153010f,
  0.876070094195406600f, -0.482183772079122720f, 0.874586652278176110f,
  -0.484869248000791060f,
  0.873094978418290090f, -0.487550160148436000f, 0.871595086655950980f,
  -0.490226483288291160f,
  0.870086991108711460f, -0.492898192229784040f, 0.868570705971340900f,
  -0.495565261825772540f,
  0.867046245515692650f, -0.498227666972781870f, 0.865513624090569090f,
  -0.500885382611240710f,
  0.863972856121586810f, -0.503538383725717580f, 0.862423956111040610f,
  -0.506186645345155230f,
  0.860866938637767310f, -0.508830142543106990f, 0.859301818357008470f,
  -0.511468850437970300f,
  0.857728610000272120f, -0.514102744193221660f, 0.856147328375194470f,
  -0.516731799017649870f,
  0.854557988365400530f, -0.519355990165589640f, 0.852960604930363630f,
  -0.521975292937154390f,
  0.851355193105265200f, -0.524589682678468950f, 0.849741768000852550f,
  -0.527199134781901280f,
  0.848120344803297230f, -0.529803624686294610f, 0.846490938774052130f,
  -0.532403127877197900f,
  0.844853565249707120f, -0.534997619887097150f, 0.843208239641845440f,
  -0.537587076295645390f,
  0.841554977436898440f, -0.540171472729892850f, 0.839893794195999520f,
  -0.542750784864515890f,
  0.838224705554838080f, -0.545324988422046460f, 0.836547727223512010f,
  -0.547894059173100190f,
  0.834862874986380010f, -0.550457972936604810f, 0.833170164701913190f,
  -0.553016705580027470f,
  0.831469612302545240f, -0.555570233019602180f, 0.829761233794523050f,
  -0.558118531220556100f,
  0.828045045257755800f, -0.560661576197336030f, 0.826321062845663530f,
  -0.563199344013834090f,
  0.824589302785025290f, -0.565731810783613120f, 0.822849781375826430f,
  -0.568258952670131490f,
  0.821102514991104650f, -0.570780745886967260f, 0.819347520076796900f,
  -0.573297166698042200f,
  0.817584813151583710f, -0.575808191417845340f, 0.815814410806733780f,
  -0.578313796411655590f,
  0.814036329705948410f, -0.580813958095764530f, 0.812250586585203880f,
  -0.583308652937698290f,
  0.810457198252594770f, -0.585797857456438860f, 0.808656181588174980f,
  -0.588281548222645220f,
  0.806847553543799330f, -0.590759701858874160f, 0.805031331142963660f,
  -0.593232295039799800f,
  0.803207531480644940f, -0.595699304492433360f, 0.801376171723140240f,
  -0.598160706996342270f,
  0.799537269107905010f, -0.600616479383868970f, 0.797690840943391160f,
  -0.603066598540348160f,
  0.795836904608883570f, -0.605511041404325550f, 0.793975477554337170f,
  -0.607949784967773630f,
  0.792106577300212390f, -0.610382806276309480f, 0.790230221437310030f,
  -0.612810082429409710f,
  0.788346427626606340f, -0.615231590580626820f, 0.786455213599085770f,
  -0.617647307937803870f,
  0.784556597155575240f, -0.620057211763289100f, 0.782650596166575730f,
  -0.622461279374149970f,
  0.780737228572094490f, -0.624859488142386340f, 0.778816512381475980f,
  -0.627251815495144080f,
  0.776888465673232440f, -0.629638238914926980f, 0.774953106594873930f,
  -0.632018735939809060f,
  0.773010453362736990f, -0.634393284163645490f, 0.771060524261813820f,
  -0.636761861236284200f,
  0.769103337645579700f, -0.639124444863775730f, 0.767138911935820400f,
  -0.641481012808583160f,
  0.765167265622458960f, -0.643831542889791390f, 0.763188417263381270f,
  -0.646176012983316280f,
  0.761202385484261780f, -0.648514401022112440f, 0.759209188978388070f,
  -0.650846684996380880f,
  0.757208846506484570f, -0.653172842953776760f, 0.755201376896536550f,
  -0.655492852999615350f,
  0.753186799043612520f, -0.657806693297078640f, 0.751165131909686480f,
  -0.660114342067420480f,
  0.749136394523459370f, -0.662415777590171780f, 0.747100605980180130f,
  -0.664710978203344790f,
  0.745057785441466060f, -0.666999922303637470f, 0.743007952135121720f,
  -0.669282588346636010f,
  0.740951125354959110f, -0.671558954847018330f, 0.738887324460615110f,
  -0.673829000378756040f,
  0.736816568877369900f, -0.676092703575315920f, 0.734738878095963500f,
  -0.678350043129861470f,
  0.732654271672412820f, -0.680600997795453020f, 0.730562769227827590f,
  -0.682845546385248080f,
  0.728464390448225200f, -0.685083667772700360f, 0.726359155084346010f,
  -0.687315340891759050f,
  0.724247082951467000f, -0.689540544737066830f, 0.722128193929215350f,
  -0.691759258364157750f,
  0.720002507961381650f, -0.693971460889654000f, 0.717870045055731710f,
  -0.696177131491462990f,
  0.715730825283818590f, -0.698376249408972920f, 0.713584868780793640f,
  -0.700568793943248340f,
  0.711432195745216430f, -0.702754744457225300f, 0.709272826438865690f,
  -0.704934080375904880f,
  0.707106781186547570f, -0.707106781186547460f, 0.704934080375904990f,
  -0.709272826438865580f,
  0.702754744457225300f, -0.711432195745216430f, 0.700568793943248450f,
  -0.713584868780793520f,
  0.698376249408972920f, -0.715730825283818590f, 0.696177131491462990f,
  -0.717870045055731710f,
  0.693971460889654000f, -0.720002507961381650f, 0.691759258364157750f,
  -0.722128193929215350f,
  0.689540544737066940f, -0.724247082951466890f, 0.687315340891759160f,
  -0.726359155084346010f,
  0.685083667772700360f, -0.728464390448225200f, 0.682845546385248080f,
  -0.730562769227827590f,
  0.680600997795453130f, -0.732654271672412820f, 0.678350043129861580f,
  -0.734738878095963390f,
  0.676092703575316030f, -0.736816568877369790f, 0.673829000378756150f,
  -0.738887324460615110f,
  0.671558954847018330f, -0.740951125354959110f, 0.669282588346636010f,
  -0.743007952135121720f,
  0.666999922303637470f, -0.745057785441465950f, 0.664710978203344900f,
  -0.747100605980180130f,
  0.662415777590171780f, -0.749136394523459260f, 0.660114342067420480f,
  -0.751165131909686370f,
  0.657806693297078640f, -0.753186799043612410f, 0.655492852999615460f,
  -0.755201376896536550f,
  0.653172842953776760f, -0.757208846506484460f, 0.650846684996380990f,
  -0.759209188978387960f,
  0.648514401022112550f, -0.761202385484261780f, 0.646176012983316390f,
  -0.763188417263381270f,
  0.643831542889791500f, -0.765167265622458960f, 0.641481012808583160f,
  -0.767138911935820400f,
  0.639124444863775730f, -0.769103337645579590f, 0.636761861236284200f,
  -0.771060524261813710f,
  0.634393284163645490f, -0.773010453362736990f, 0.632018735939809060f,
  -0.774953106594873820f,
  0.629638238914927100f, -0.776888465673232440f, 0.627251815495144190f,
  -0.778816512381475870f,
  0.624859488142386450f, -0.780737228572094380f, 0.622461279374150080f,
  -0.782650596166575730f,
  0.620057211763289210f, -0.784556597155575240f, 0.617647307937803980f,
  -0.786455213599085770f,
  0.615231590580626820f, -0.788346427626606230f, 0.612810082429409710f,
  -0.790230221437310030f,
  0.610382806276309480f, -0.792106577300212390f, 0.607949784967773740f,
  -0.793975477554337170f,
  0.605511041404325550f, -0.795836904608883460f, 0.603066598540348280f,
  -0.797690840943391040f,
  0.600616479383868970f, -0.799537269107905010f, 0.598160706996342380f,
  -0.801376171723140130f,
  0.595699304492433470f, -0.803207531480644830f, 0.593232295039799800f,
  -0.805031331142963660f,
  0.590759701858874280f, -0.806847553543799220f, 0.588281548222645330f,
  -0.808656181588174980f,
  0.585797857456438860f, -0.810457198252594770f, 0.583308652937698290f,
  -0.812250586585203880f,
  0.580813958095764530f, -0.814036329705948300f, 0.578313796411655590f,
  -0.815814410806733780f,
  0.575808191417845340f, -0.817584813151583710f, 0.573297166698042320f,
  -0.819347520076796900f,
  0.570780745886967370f, -0.821102514991104650f, 0.568258952670131490f,
  -0.822849781375826320f,
  0.565731810783613230f, -0.824589302785025290f, 0.563199344013834090f,
  -0.826321062845663420f,
  0.560661576197336030f, -0.828045045257755800f, 0.558118531220556100f,
  -0.829761233794523050f,
  0.555570233019602290f, -0.831469612302545240f, 0.553016705580027580f,
  -0.833170164701913190f,
  0.550457972936604810f, -0.834862874986380010f, 0.547894059173100190f,
  -0.836547727223511890f,
  0.545324988422046460f, -0.838224705554837970f, 0.542750784864516000f,
  -0.839893794195999410f,
  0.540171472729892970f, -0.841554977436898330f, 0.537587076295645510f,
  -0.843208239641845440f,
  0.534997619887097260f, -0.844853565249707010f, 0.532403127877198010f,
  -0.846490938774052020f,
  0.529803624686294830f, -0.848120344803297120f, 0.527199134781901390f,
  -0.849741768000852440f,
  0.524589682678468840f, -0.851355193105265200f, 0.521975292937154390f,
  -0.852960604930363630f,
  0.519355990165589530f, -0.854557988365400530f, 0.516731799017649980f,
  -0.856147328375194470f,
  0.514102744193221660f, -0.857728610000272120f, 0.511468850437970520f,
  -0.859301818357008360f,
  0.508830142543106990f, -0.860866938637767310f, 0.506186645345155450f,
  -0.862423956111040500f,
  0.503538383725717580f, -0.863972856121586700f, 0.500885382611240940f,
  -0.865513624090568980f,
  0.498227666972781870f, -0.867046245515692650f, 0.495565261825772490f,
  -0.868570705971340900f,
  0.492898192229784090f, -0.870086991108711350f, 0.490226483288291100f,
  -0.871595086655951090f,
  0.487550160148436050f, -0.873094978418290090f, 0.484869248000791120f,
  -0.874586652278176110f,
  0.482183772079122830f, -0.876070094195406600f, 0.479493757660153010f,
  -0.877545290207261240f,
  0.476799230063322250f, -0.879012226428633410f, 0.474100214650550020f,
  -0.880470889052160750f,
  0.471396736825997810f, -0.881921264348354940f, 0.468688822035827960f,
  -0.883363338665731580f,
  0.465976495767966130f, -0.884797098430937790f, 0.463259783551860260f,
  -0.886222530148880640f,
  0.460538710958240010f, -0.887639620402853930f, 0.457813303598877290f,
  -0.889048355854664570f,
  0.455083587126343840f, -0.890448723244757880f, 0.452349587233771000f,
  -0.891840709392342720f,
  0.449611329654606600f, -0.893224301195515320f, 0.446868840162374330f,
  -0.894599485631382580f,
  0.444122144570429260f, -0.895966249756185110f, 0.441371268731716620f,
  -0.897324580705418320f,
  0.438616238538527710f, -0.898674465693953820f, 0.435857079922255470f,
  -0.900015892016160280f,
  0.433093818853152010f, -0.901348847046022030f, 0.430326481340082610f,
  -0.902673318237258830f,
  0.427555093430282200f, -0.903989293123443340f, 0.424779681209108810f,
  -0.905296759318118820f,
  0.422000270799799790f, -0.906595704514915330f, 0.419216888363223960f,
  -0.907886116487666150f,
  0.416429560097637320f, -0.909167983090522270f, 0.413638312238434560f,
  -0.910441292258067140f,
  0.410843171057903910f, -0.911706032005429880f, 0.408044162864978740f,
  -0.912962190428398100f,
  0.405241314004989860f, -0.914209755703530690f, 0.402434650859418540f,
  -0.915448716088267830f,
  0.399624199845646790f, -0.916679059921042700f, 0.396809987416710420f,
  -0.917900775621390390f,
  0.393992040061048100f, -0.919113851690057770f, 0.391170384302253980f,
  -0.920318276709110480f,
  0.388345046698826300f, -0.921514039342041900f, 0.385516053843919020f,
  -0.922701128333878520f,
  0.382683432365089840f, -0.923879532511286740f, 0.379847208924051110f,
  -0.925049240782677580f,
  0.377007410216418310f, -0.926210242138311270f, 0.374164062971457990f,
  -0.927362525650401110f,
  0.371317193951837600f, -0.928506080473215480f, 0.368466829953372320f,
  -0.929640895843181330f,
  0.365612997804773960f, -0.930766961078983710f, 0.362755724367397230f,
  -0.931884265581668150f,
  0.359895036534988280f, -0.932992798834738850f, 0.357030961233430030f,
  -0.934092550404258870f,
  0.354163525420490510f, -0.935183509938947500f, 0.351292756085567150f,
  -0.936265667170278260f,
  0.348418680249434510f, -0.937339011912574960f, 0.345541324963989150f,
  -0.938403534063108060f,
  0.342660717311994380f, -0.939459223602189920f, 0.339776884406826960f,
  -0.940506070593268300f,
  0.336889853392220050f, -0.941544065183020810f, 0.333999651442009490f,
  -0.942573197601446870f,
  0.331106305759876430f, -0.943593458161960390f, 0.328209843579092660f,
  -0.944604837261480260f,
  0.325310292162262980f, -0.945607325380521280f, 0.322407678801070020f,
  -0.946600913083283530f,
  0.319502030816015750f, -0.947585591017741090f, 0.316593375556165850f,
  -0.948561349915730270f,
  0.313681740398891570f, -0.949528180593036670f, 0.310767152749611470f,
  -0.950486073949481700f,
  0.307849640041534980f, -0.951435020969008340f, 0.304929229735402430f,
  -0.952375012719765880f,
  0.302005949319228200f, -0.953306040354193750f, 0.299079826308040480f,
  -0.954228095109105670f,
  0.296150888243623960f, -0.955141168305770670f, 0.293219162694258680f,
  -0.956045251349996410f,
  0.290284677254462330f, -0.956940335732208940f, 0.287347459544729570f,
  -0.957826413027532910f,
  0.284407537211271820f, -0.958703474895871600f, 0.281464937925758050f,
  -0.959571513081984520f,
  0.278519689385053060f, -0.960430519415565790f, 0.275571819310958250f,
  -0.961280485811320640f,
  0.272621355449948980f, -0.962121404269041580f, 0.269668325572915200f,
  -0.962953266873683880f,
  0.266712757474898420f, -0.963776065795439840f, 0.263754678974831510f,
  -0.964589793289812650f,
  0.260794117915275570f, -0.965394441697689400f, 0.257831102162158930f,
  -0.966190003445412620f,
  0.254865659604514630f, -0.966976471044852070f, 0.251897818154216910f,
  -0.967753837093475510f,
  0.248927605745720260f, -0.968522094274417270f, 0.245955050335794590f,
  -0.969281235356548530f,
  0.242980179903263980f, -0.970031253194543970f, 0.240003022448741500f,
  -0.970772140728950350f,
  0.237023605994367340f, -0.971503890986251780f, 0.234041958583543460f,
  -0.972226497078936270f,
  0.231058108280671280f, -0.972939952205560070f, 0.228072083170885790f,
  -0.973644249650811870f,
  0.225083911359792780f, -0.974339382785575860f, 0.222093620973203590f,
  -0.975025345066994120f,
  0.219101240156869770f, -0.975702130038528570f, 0.216106797076219600f,
  -0.976369731330021140f,
  0.213110319916091360f, -0.977028142657754390f, 0.210111836880469720f,
  -0.977677357824509930f,
  0.207111376192218560f, -0.978317370719627650f, 0.204108966092817010f,
  -0.978948175319062200f,
  0.201104634842091960f, -0.979569765685440520f, 0.198098410717953730f,
  -0.980182135968117320f,
  0.195090322016128330f, -0.980785280403230430f, 0.192080397049892380f,
  -0.981379193313754560f,
  0.189068664149806280f, -0.981963869109555240f, 0.186055151663446630f,
  -0.982539302287441240f,
  0.183039887955141060f, -0.983105487431216290f, 0.180022901405699510f,
  -0.983662419211730250f,
  0.177004220412148860f, -0.984210092386929030f, 0.173983873387463850f,
  -0.984748501801904210f,
  0.170961888760301360f, -0.985277642388941220f, 0.167938294974731230f,
  -0.985797509167567370f,
  0.164913120489970090f, -0.986308097244598670f, 0.161886393780111910f,
  -0.986809401814185420f,
  0.158858143333861390f, -0.987301418157858430f, 0.155828397654265320f,
  -0.987784141644572180f,
  0.152797185258443410f, -0.988257567730749460f, 0.149764534677321620f,
  -0.988721691960323780f,
  0.146730474455361750f, -0.989176509964781010f, 0.143695033150294580f,
  -0.989622017463200780f,
  0.140658239332849240f, -0.990058210262297120f, 0.137620121586486180f,
  -0.990485084256456980f,
  0.134580708507126220f, -0.990902635427780010f, 0.131540028702883280f,
  -0.991310859846115440f,
  0.128498110793793220f, -0.991709753669099530f, 0.125454983411546210f,
  -0.992099313142191800f,
  0.122410675199216280f, -0.992479534598709970f, 0.119365214810991350f,
  -0.992850414459865100f,
  0.116318630911904880f, -0.993211949234794500f, 0.113270952177564360f,
  -0.993564135520595300f,
  0.110222207293883180f, -0.993906970002356060f, 0.107172424956808870f,
  -0.994240449453187900f,
  0.104121633872054730f, -0.994564570734255420f, 0.101069862754827880f,
  -0.994879330794805620f,
  0.098017140329560770f, -0.995184726672196820f, 0.094963495329639061f,
  -0.995480755491926940f,
  0.091908956497132696f, -0.995767414467659820f, 0.088853552582524684f,
  -0.996044700901251970f,
  0.085797312344439880f, -0.996312612182778000f, 0.082740264549375803f,
  -0.996571145790554840f,
  0.079682437971430126f, -0.996820299291165670f, 0.076623861392031617f,
  -0.997060070339482960f,
  0.073564563599667454f, -0.997290456678690210f, 0.070504573389614009f,
  -0.997511456140303450f,
  0.067443919563664106f, -0.997723066644191640f, 0.064382630929857410f,
  -0.997925286198596000f,
  0.061320736302208648f, -0.998118112900149180f, 0.058258264500435732f,
  -0.998301544933892890f,
  0.055195244349690031f, -0.998475580573294770f, 0.052131704680283317f,
  -0.998640218180265270f,
  0.049067674327418126f, -0.998795456205172410f, 0.046003182130914644f,
  -0.998941293186856870f,
  0.042938256934940959f, -0.999077727752645360f, 0.039872927587739845f,
  -0.999204758618363890f,
  0.036807222941358991f, -0.999322384588349540f, 0.033741171851377642f,
  -0.999430604555461730f,
  0.030674803176636581f, -0.999529417501093140f, 0.027608145778965820f,
  -0.999618822495178640f,
  0.024541228522912264f, -0.999698818696204250f, 0.021474080275469605f,
  -0.999769405351215280f,
  0.018406729905804820f, -0.999830581795823400f, 0.015339206284988220f,
  -0.999882347454212560f,
  0.012271538285719944f, -0.999924701839144500f, 0.009203754782059960f,
  -0.999957644551963900f,
  0.006135884649154515f, -0.999981175282601110f, 0.003067956762966138f,
  -0.999995293809576190f
};

static const float32_t Weights_2048[4096] = {
  1.000000000000000000f, 0.000000000000000000f, 0.999999705862882230f,
  -0.000766990318742704f,
  0.999998823451701880f, -0.001533980186284766f, 0.999997352766978210f,
  -0.002300969151425805f,
  0.999995293809576190f, -0.003067956762965976f, 0.999992646580707190f,
  -0.003834942569706228f,
  0.999989411081928400f, -0.004601926120448571f, 0.999985587315143200f,
  -0.005368906963996343f,
  0.999981175282601110f, -0.006135884649154475f, 0.999976174986897610f,
  -0.006902858724729756f,
  0.999970586430974140f, -0.007669828739531097f, 0.999964409618118280f,
  -0.008436794242369799f,
  0.999957644551963900f, -0.009203754782059819f, 0.999950291236490480f,
  -0.009970709907418031f,
  0.999942349676023910f, -0.010737659167264491f, 0.999933819875236000f,
  -0.011504602110422714f,
  0.999924701839144500f, -0.012271538285719925f, 0.999914995573113470f,
  -0.013038467241987334f,
  0.999904701082852900f, -0.013805388528060391f, 0.999893818374418490f,
  -0.014572301692779064f,
  0.999882347454212560f, -0.015339206284988100f, 0.999870288328982950f,
  -0.016106101853537287f,
  0.999857641005823860f, -0.016872987947281710f, 0.999844405492175240f,
  -0.017639864115082053f,
  0.999830581795823400f, -0.018406729905804820f, 0.999816169924900410f,
  -0.019173584868322623f,
  0.999801169887884260f, -0.019940428551514441f, 0.999785581693599210f,
  -0.020707260504265895f,
  0.999769405351215280f, -0.021474080275469508f, 0.999752640870248840f,
  -0.022240887414024961f,
  0.999735288260561680f, -0.023007681468839369f, 0.999717347532362190f,
  -0.023774461988827555f,
  0.999698818696204250f, -0.024541228522912288f, 0.999679701762987930f,
  -0.025307980620024571f,
  0.999659996743959220f, -0.026074717829103901f, 0.999639703650710200f,
  -0.026841439699098531f,
  0.999618822495178640f, -0.027608145778965740f, 0.999597353289648380f,
  -0.028374835617672099f,
  0.999575296046749220f, -0.029141508764193722f, 0.999552650779456990f,
  -0.029908164767516555f,
  0.999529417501093140f, -0.030674803176636626f, 0.999505596225325310f,
  -0.031441423540560301f,
  0.999481186966166950f, -0.032208025408304586f, 0.999456189737977340f,
  -0.032974608328897335f,
  0.999430604555461730f, -0.033741171851377580f, 0.999404431433671300f,
  -0.034507715524795750f,
  0.999377670388002850f, -0.035274238898213947f, 0.999350321434199440f,
  -0.036040741520706229f,
  0.999322384588349540f, -0.036807222941358832f, 0.999293859866887790f,
  -0.037573682709270494f,
  0.999264747286594420f, -0.038340120373552694f, 0.999235046864595850f,
  -0.039106535483329888f,
  0.999204758618363890f, -0.039872927587739811f, 0.999173882565716380f,
  -0.040639296235933736f,
  0.999142418724816910f, -0.041405640977076739f, 0.999110367114174890f,
  -0.042171961360347947f,
  0.999077727752645360f, -0.042938256934940820f, 0.999044500659429290f,
  -0.043704527250063421f,
  0.999010685854073380f, -0.044470771854938668f, 0.998976283356469820f,
  -0.045236990298804590f,
  0.998941293186856870f, -0.046003182130914623f, 0.998905715365818290f,
  -0.046769346900537863f,
  0.998869549914283560f, -0.047535484156959303f, 0.998832796853527990f,
  -0.048301593449480144f,
  0.998795456205172410f, -0.049067674327418015f, 0.998757527991183340f,
  -0.049833726340107277f,
  0.998719012233872940f, -0.050599749036899282f, 0.998679908955899090f,
  -0.051365741967162593f,
  0.998640218180265270f, -0.052131704680283324f, 0.998599939930320370f,
  -0.052897636725665324f,
  0.998559074229759310f, -0.053663537652730520f, 0.998517621102622210f,
  -0.054429407010919133f,
  0.998475580573294770f, -0.055195244349689934f, 0.998432952666508440f,
  -0.055961049218520569f,
  0.998389737407340160f, -0.056726821166907748f, 0.998345934821212370f,
  -0.057492559744367566f,
  0.998301544933892890f, -0.058258264500435752f, 0.998256567771495180f,
  -0.059023934984667931f,
  0.998211003360478190f, -0.059789570746639868f, 0.998164851727646240f,
  -0.060555171335947788f,
  0.998118112900149180f, -0.061320736302208578f, 0.998070786905482340f,
  -0.062086265195060088f,
  0.998022873771486240f, -0.062851757564161406f, 0.997974373526346990f,
  -0.063617212959193106f,
  0.997925286198596000f, -0.064382630929857465f, 0.997875611817110150f,
  -0.065148011025878833f,
  0.997825350411111640f, -0.065913352797003805f, 0.997774502010167820f,
  -0.066678655793001557f,
  0.997723066644191640f, -0.067443919563664051f, 0.997671044343441000f,
  -0.068209143658806329f,
  0.997618435138519550f, -0.068974327628266746f, 0.997565239060375750f,
  -0.069739471021907307f,
  0.997511456140303450f, -0.070504573389613856f, 0.997457086409941910f,
  -0.071269634281296401f,
  0.997402129901275300f, -0.072034653246889332f, 0.997346586646633230f,
  -0.072799629836351673f,
  0.997290456678690210f, -0.073564563599667426f, 0.997233740030466280f,
  -0.074329454086845756f,
  0.997176436735326190f, -0.075094300847921305f, 0.997118546826979980f,
  -0.075859103432954447f,
  0.997060070339482960f, -0.076623861392031492f, 0.997001007307235290f,
  -0.077388574275265049f,
  0.996941357764982160f, -0.078153241632794232f, 0.996881121747813850f,
  -0.078917863014784942f,
  0.996820299291165670f, -0.079682437971430126f, 0.996758890430818000f,
  -0.080446966052950014f,
  0.996696895202896060f, -0.081211446809592441f, 0.996634313643869900f,
  -0.081975879791633066f,
  0.996571145790554840f, -0.082740264549375692f, 0.996507391680110820f,
  -0.083504600633152432f,
  0.996443051350042630f, -0.084268887593324071f, 0.996378124838200210f,
  -0.085033124980280275f,
  0.996312612182778000f, -0.085797312344439894f, 0.996246513422315520f,
  -0.086561449236251170f,
  0.996179828595696980f, -0.087325535206192059f, 0.996112557742151130f,
  -0.088089569804770507f,
  0.996044700901251970f, -0.088853552582524600f, 0.995976258112917790f,
  -0.089617483090022959f,
  0.995907229417411720f, -0.090381360877864983f, 0.995837614855341610f,
  -0.091145185496681005f,
  0.995767414467659820f, -0.091908956497132724f, 0.995696628295663520f,
  -0.092672673429913310f,
  0.995625256380994310f, -0.093436335845747787f, 0.995553298765638470f,
  -0.094199943295393204f,
  0.995480755491926940f, -0.094963495329638992f, 0.995407626602534900f,
  -0.095726991499307162f,
  0.995333912140482280f, -0.096490431355252593f, 0.995259612149133390f,
  -0.097253814448363271f,
  0.995184726672196930f, -0.098017140329560604f, 0.995109255753726110f,
  -0.098780408549799623f,
  0.995033199438118630f, -0.099543618660069319f, 0.994956557770116380f,
  -0.100306770211392860f,
  0.994879330794805620f, -0.101069862754827820f, 0.994801518557617110f,
  -0.101832895841466530f,
  0.994723121104325700f, -0.102595869022436280f, 0.994644138481050710f,
  -0.103358781848899610f,
  0.994564570734255420f, -0.104121633872054590f, 0.994484417910747600f,
  -0.104884424643134970f,
  0.994403680057679100f, -0.105647153713410620f, 0.994322357222545810f,
  -0.106409820634187680f,
  0.994240449453187900f, -0.107172424956808840f, 0.994157956797789730f,
  -0.107934966232653650f,
  0.994074879304879370f, -0.108697444013138720f, 0.993991217023329380f,
  -0.109459857849717980f,
  0.993906970002356060f, -0.110222207293883060f, 0.993822138291519660f,
  -0.110984491897163390f,
  0.993736721940724600f, -0.111746711211126590f, 0.993650721000219120f,
  -0.112508864787378690f,
  0.993564135520595300f, -0.113270952177564350f, 0.993476965552789190f,
  -0.114032972933367200f,
  0.993389211148080650f, -0.114794926606510080f, 0.993300872358093280f,
  -0.115556812748755260f,
  0.993211949234794500f, -0.116318630911904750f, 0.993122441830495580f,
  -0.117080380647800590f,
  0.993032350197851410f, -0.117842061508324980f, 0.992941674389860470f,
  -0.118603673045400720f,
  0.992850414459865100f, -0.119365214810991350f, 0.992758570461551140f,
  -0.120126686357101500f,
  0.992666142448948020f, -0.120888087235777080f, 0.992573130476428810f,
  -0.121649416999105530f,
  0.992479534598709970f, -0.122410675199216200f, 0.992385354870851670f,
  -0.123171861388280480f,
  0.992290591348257370f, -0.123932975118512160f, 0.992195244086673920f,
  -0.124694015942167640f,
  0.992099313142191800f, -0.125454983411546230f, 0.992002798571244520f,
  -0.126215877078990350f,
  0.991905700430609330f, -0.126976696496885870f, 0.991808018777406430f,
  -0.127737441217662310f,
  0.991709753669099530f, -0.128498110793793170f, 0.991610905163495370f,
  -0.129258704777796140f,
  0.991511473318743900f, -0.130019222722233350f, 0.991411458193338540f,
  -0.130779664179711710f,
  0.991310859846115440f, -0.131540028702883120f, 0.991209678336254060f,
  -0.132300315844444650f,
  0.991107913723276890f, -0.133060525157139060f, 0.991005566067049370f,
  -0.133820656193754720f,
  0.990902635427780010f, -0.134580708507126170f, 0.990799121866020370f,
  -0.135340681650134210f,
  0.990695025442664630f, -0.136100575175706200f, 0.990590346218950150f,
  -0.136860388636816380f,
  0.990485084256457090f, -0.137620121586486040f, 0.990379239617108160f,
  -0.138379773577783890f,
  0.990272812363169110f, -0.139139344163826200f, 0.990165802557248400f,
  -0.139898832897777210f,
  0.990058210262297120f, -0.140658239332849210f, 0.989950035541608990f,
  -0.141417563022303020f,
  0.989841278458820530f, -0.142176803519448030f, 0.989731939077910570f,
  -0.142935960377642670f,
  0.989622017463200890f, -0.143695033150294470f, 0.989511513679355190f,
  -0.144454021390860470f,
  0.989400427791380380f, -0.145212924652847460f, 0.989288759864625170f,
  -0.145971742489812210f,
  0.989176509964781010f, -0.146730474455361750f, 0.989063678157881540f,
  -0.147489120103153570f,
  0.988950264510302990f, -0.148247678986896030f, 0.988836269088763540f,
  -0.149006150660348450f,
  0.988721691960323780f, -0.149764534677321510f, 0.988606533192386450f,
  -0.150522830591677400f,
  0.988490792852696590f, -0.151281037957330220f, 0.988374471009341280f,
  -0.152039156328246050f,
  0.988257567730749460f, -0.152797185258443440f, 0.988140083085692570f,
  -0.153555124301993450f,
  0.988022017143283530f, -0.154312973013020100f, 0.987903369972977790f,
  -0.155070730945700510f,
  0.987784141644572180f, -0.155828397654265230f, 0.987664332228205710f,
  -0.156585972692998430f,
  0.987543941794359230f, -0.157343455616238250f, 0.987422970413855410f,
  -0.158100845978376980f,
  0.987301418157858430f, -0.158858143333861450f, 0.987179285097874340f,
  -0.159615347237193060f,
  0.987056571305750970f, -0.160372457242928280f, 0.986933276853677710f,
  -0.161129472905678810f,
  0.986809401814185530f, -0.161886393780111830f, 0.986684946260146690f,
  -0.162643219420950310f,
  0.986559910264775410f, -0.163399949382973230f, 0.986434293901627180f,
  -0.164156583221015810f,
  0.986308097244598670f, -0.164913120489969890f, 0.986181320367928270f,
  -0.165669560744784120f,
  0.986053963346195440f, -0.166425903540464100f, 0.985926026254321130f,
  -0.167182148432072940f,
  0.985797509167567480f, -0.167938294974731170f, 0.985668412161537550f,
  -0.168694342723617330f,
  0.985538735312176060f, -0.169450291233967960f, 0.985408478695768420f,
  -0.170206140061078070f,
  0.985277642388941220f, -0.170961888760301220f, 0.985146226468662230f,
  -0.171717536887049970f,
  0.985014231012239840f, -0.172473083996795950f, 0.984881656097323700f,
  -0.173228529645070320f,
  0.984748501801904210f, -0.173983873387463820f, 0.984614768204312600f,
  -0.174739114779627200f,
  0.984480455383220930f, -0.175494253377271430f, 0.984345563417641900f,
  -0.176249288736167880f,
  0.984210092386929030f, -0.177004220412148750f, 0.984074042370776450f,
  -0.177759047961107170f,
  0.983937413449218920f, -0.178513770938997510f, 0.983800205702631600f,
  -0.179268388901835750f,
  0.983662419211730250f, -0.180022901405699510f, 0.983524054057571260f,
  -0.180777308006728590f,
  0.983385110321551180f, -0.181531608261124970f, 0.983245588085407070f,
  -0.182285801725153300f,
  0.983105487431216290f, -0.183039887955140950f, 0.982964808441396440f,
  -0.183793866507478450f,
  0.982823551198705240f, -0.184547736938619620f, 0.982681715786240860f,
  -0.185301498805081900f,
  0.982539302287441240f, -0.186055151663446630f, 0.982396310786084690f,
  -0.186808695070359270f,
  0.982252741366289370f, -0.187562128582529600f, 0.982108594112513610f,
  -0.188315451756732120f,
  0.981963869109555240f, -0.189068664149806190f, 0.981818566442552500f,
  -0.189821765318656410f,
  0.981672686196983110f, -0.190574754820252740f, 0.981526228458664770f,
  -0.191327632211630900f,
  0.981379193313754560f, -0.192080397049892440f, 0.981231580848749730f,
  -0.192833048892205230f,
  0.981083391150486710f, -0.193585587295803610f, 0.980934624306141640f,
  -0.194338011817988600f,
  0.980785280403230430f, -0.195090322016128250f, 0.980635359529608120f,
  -0.195842517447657850f,
  0.980484861773469380f, -0.196594597670080220f, 0.980333787223347960f,
  -0.197346562240965920f,
  0.980182135968117430f, -0.198098410717953560f, 0.980029908096990090f,
  -0.198850142658750090f,
  0.979877103699517640f, -0.199601757621130970f, 0.979723722865591170f,
  -0.200353255162940450f,
  0.979569765685440520f, -0.201104634842091900f, 0.979415232249634780f,
  -0.201855896216568050f,
  0.979260122649082020f, -0.202607038844421130f, 0.979104436975029250f,
  -0.203358062283773320f,
  0.978948175319062200f, -0.204108966092816870f, 0.978791337773105670f,
  -0.204859749829814420f,
  0.978633924429423210f, -0.205610413053099240f, 0.978475935380616830f,
  -0.206360955321075510f,
  0.978317370719627650f, -0.207111376192218560f, 0.978158230539735050f,
  -0.207861675225075070f,
  0.977998514934557140f, -0.208611851978263490f, 0.977838223998050430f,
  -0.209361906010474160f,
  0.977677357824509930f, -0.210111836880469610f, 0.977515916508569280f,
  -0.210861644147084860f,
  0.977353900145199960f, -0.211611327369227550f, 0.977191308829712280f,
  -0.212360886105878420f,
  0.977028142657754390f, -0.213110319916091360f, 0.976864401725312640f,
  -0.213859628358993750f,
  0.976700086128711840f, -0.214608810993786760f, 0.976535195964614470f,
  -0.215357867379745550f,
  0.976369731330021140f, -0.216106797076219520f, 0.976203692322270560f,
  -0.216855599642632620f,
  0.976037079039039020f, -0.217604274638483640f, 0.975869891578341030f,
  -0.218352821623346320f,
  0.975702130038528570f, -0.219101240156869800f, 0.975533794518291360f,
  -0.219849529798778700f,
  0.975364885116656980f, -0.220597690108873510f, 0.975195401932990370f,
  -0.221345720647030810f,
  0.975025345066994120f, -0.222093620973203510f, 0.974854714618708430f,
  -0.222841390647421120f,
  0.974683510688510670f, -0.223589029229789990f, 0.974511733377115720f,
  -0.224336536280493600f,
  0.974339382785575860f, -0.225083911359792830f, 0.974166459015280320f,
  -0.225831154028026170f,
  0.973992962167955830f, -0.226578263845610000f, 0.973818892345666100f,
  -0.227325240373038860f,
  0.973644249650811980f, -0.228072083170885730f, 0.973469034186131070f,
  -0.228818791799802220f,
  0.973293246054698250f, -0.229565365820518870f, 0.973116885359925130f,
  -0.230311804793845440f,
  0.972939952205560180f, -0.231058108280671110f, 0.972762446695688570f,
  -0.231804275841964780f,
  0.972584368934732210f, -0.232550307038775240f, 0.972405719027449770f,
  -0.233296201432231590f,
  0.972226497078936270f, -0.234041958583543430f, 0.972046703194623500f,
  -0.234787578054000970f,
  0.971866337480279400f, -0.235533059404975490f, 0.971685400042008540f,
  -0.236278402197919570f,
  0.971503890986251780f, -0.237023605994367200f, 0.971321810419786160f,
  -0.237768670355934190f,
  0.971139158449725090f, -0.238513594844318420f, 0.970955935183517970f,
  -0.239258379021299980f,
  0.970772140728950350f, -0.240003022448741500f, 0.970587775194143630f,
  -0.240747524688588430f,
  0.970402838687555500f, -0.241491885302869330f, 0.970217331317979160f,
  -0.242236103853696010f,
  0.970031253194543970f, -0.242980179903263870f, 0.969844604426714830f,
  -0.243724113013852160f,
  0.969657385124292450f, -0.244467902747824150f, 0.969469595397413060f,
  -0.245211548667627540f,
  0.969281235356548530f, -0.245955050335794590f, 0.969092305112506210f,
  -0.246698407314942410f,
  0.968902804776428870f, -0.247441619167773270f, 0.968712734459794780f,
  -0.248184685457074780f,
  0.968522094274417380f, -0.248927605745720150f, 0.968330884332445190f,
  -0.249670379596668570f,
  0.968139104746362440f, -0.250413006572965220f, 0.967946755628987800f,
  -0.251155486237741920f,
  0.967753837093475510f, -0.251897818154216970f, 0.967560349253314360f,
  -0.252640001885695520f,
  0.967366292222328510f, -0.253382036995570160f, 0.967171666114676640f,
  -0.254123923047320620f,
  0.966976471044852070f, -0.254865659604514570f, 0.966780707127683270f,
  -0.255607246230807380f,
  0.966584374478333120f, -0.256348682489942910f, 0.966387473212298900f,
  -0.257089967945753120f,
  0.966190003445412500f, -0.257831102162158990f, 0.965991965293840570f,
  -0.258572084703170340f,
  0.965793358874083680f, -0.259312915132886230f, 0.965594184302976830f,
  -0.260053593015495190f,
  0.965394441697689400f, -0.260794117915275510f, 0.965194131175724720f,
  -0.261534489396595520f,
  0.964993252854920320f, -0.262274707023913590f, 0.964791806853447900f,
  -0.263014770361779000f,
  0.964589793289812760f, -0.263754678974831350f, 0.964387212282854290f,
  -0.264494432427801630f,
  0.964184063951745830f, -0.265234030285511790f, 0.963980348415994110f,
  -0.265973472112875590f,
  0.963776065795439840f, -0.266712757474898370f, 0.963571216210257320f,
  -0.267451885936677620f,
  0.963365799780954050f, -0.268190857063403180f, 0.963159816628371360f,
  -0.268929670420357260f,
  0.962953266873683880f, -0.269668325572915090f, 0.962746150638399410f,
  -0.270406822086544820f,
  0.962538468044359160f, -0.271145159526808010f, 0.962330219213737400f,
  -0.271883337459359720f,
  0.962121404269041580f, -0.272621355449948980f, 0.961912023333112210f,
  -0.273359213064418680f,
  0.961702076529122540f, -0.274096909868706380f, 0.961491563980579000f,
  -0.274834445428843940f,
  0.961280485811320640f, -0.275571819310958140f, 0.961068842145519350f,
  -0.276309031081271080f,
  0.960856633107679660f, -0.277046080306099900f, 0.960643858822638590f,
  -0.277782966551857690f,
  0.960430519415565790f, -0.278519689385053060f, 0.960216615011963430f,
  -0.279256248372291180f,
  0.960002145737665960f, -0.279992643080273220f, 0.959787111718839900f,
  -0.280728873075797190f,
  0.959571513081984520f, -0.281464937925757940f, 0.959355349953930790f,
  -0.282200837197147560f,
  0.959138622461841890f, -0.282936570457055390f, 0.958921330733213170f,
  -0.283672137272668430f,
  0.958703474895871600f, -0.284407537211271880f, 0.958485055077976100f,
  -0.285142769840248670f,
  0.958266071408017670f, -0.285877834727080620f, 0.958046524014818600f,
  -0.286612731439347790f,
  0.957826413027532910f, -0.287347459544729510f, 0.957605738575646350f,
  -0.288082018611004130f,
  0.957384500788975860f, -0.288816408206049480f, 0.957162699797670210f,
  -0.289550627897843030f,
  0.956940335732208820f, -0.290284677254462330f, 0.956717408723403050f,
  -0.291018555844085090f,
  0.956493918902395100f, -0.291752263234989260f, 0.956269866400658030f,
  -0.292485798995553880f,
  0.956045251349996410f, -0.293219162694258630f, 0.955820073882545420f,
  -0.293952353899684660f,
  0.955594334130771110f, -0.294685372180514330f, 0.955368032227470350f,
  -0.295418217105532010f,
  0.955141168305770780f, -0.296150888243623790f, 0.954913742499130520f,
  -0.296883385163778270f,
  0.954685754941338340f, -0.297615707435086200f, 0.954457205766513490f,
  -0.298347854626741400f,
  0.954228095109105670f, -0.299079826308040480f, 0.953998423103894490f,
  -0.299811622048383350f,
  0.953768189885990330f, -0.300543241417273450f, 0.953537395590833280f,
  -0.301274683984317950f,
  0.953306040354193860f, -0.302005949319228080f, 0.953074124312172200f,
  -0.302737036991819140f,
  0.952841647601198720f, -0.303467946572011320f, 0.952608610358033350f,
  -0.304198677629829110f,
  0.952375012719765880f, -0.304929229735402370f, 0.952140854823815830f,
  -0.305659602458966120f,
  0.951906136807932350f, -0.306389795370860920f, 0.951670858810193860f,
  -0.307119808041533100f,
  0.951435020969008340f, -0.307849640041534870f, 0.951198623423113230f,
  -0.308579290941525090f,
  0.950961666311575080f, -0.309308760312268730f, 0.950724149773789610f,
  -0.310038047724637890f,
  0.950486073949481700f, -0.310767152749611470f, 0.950247438978705230f,
  -0.311496074958275910f,
  0.950008245001843000f, -0.312224813921824880f, 0.949768492159606680f,
  -0.312953369211560200f,
  0.949528180593036670f, -0.313681740398891520f, 0.949287310443502120f,
  -0.314409927055336660f,
  0.949045881852700560f, -0.315137928752522440f, 0.948803894962658490f,
  -0.315865745062183960f,
  0.948561349915730270f, -0.316593375556165850f, 0.948318246854599090f,
  -0.317320819806421740f,
  0.948074585922276230f, -0.318048077385014950f, 0.947830367262101010f,
  -0.318775147864118480f,
  0.947585591017741090f, -0.319502030816015690f, 0.947340257333192050f,
  -0.320228725813099860f,
  0.947094366352777220f, -0.320955232427875210f, 0.946847918221148000f,
  -0.321681550232956580f,
  0.946600913083283530f, -0.322407678801069850f, 0.946353351084490590f,
  -0.323133617705052330f,
  0.946105232370403450f, -0.323859366517852850f, 0.945856557086983910f,
  -0.324584924812532150f,
  0.945607325380521280f, -0.325310292162262930f, 0.945357537397632290f,
  -0.326035468140330240f,
  0.945107193285260610f, -0.326760452320131730f, 0.944856293190677210f,
  -0.327485244275178000f,
  0.944604837261480260f, -0.328209843579092500f, 0.944352825645594750f,
  -0.328934249805612200f,
  0.944100258491272660f, -0.329658462528587490f, 0.943847135947092690f,
  -0.330382481321982780f,
  0.943593458161960390f, -0.331106305759876430f, 0.943339225285107720f,
  -0.331829935416461110f,
  0.943084437466093490f, -0.332553369866044220f, 0.942829094854802710f,
  -0.333276608683047930f,
  0.942573197601446870f, -0.333999651442009380f, 0.942316745856563780f,
  -0.334722497717581220f,
  0.942059739771017310f, -0.335445147084531600f, 0.941802179495997650f,
  -0.336167599117744520f,
  0.941544065183020810f, -0.336889853392220050f, 0.941285396983928660f,
  -0.337611909483074620f,
  0.941026175050889260f, -0.338333766965541130f, 0.940766399536396070f,
  -0.339055425414969640f,
  0.940506070593268300f, -0.339776884406826850f, 0.940245188374650880f,
  -0.340498143516697160f,
  0.939983753034014050f, -0.341219202320282360f, 0.939721764725153340f,
  -0.341940060393402190f,
  0.939459223602189920f, -0.342660717311994380f, 0.939196129819569900f,
  -0.343381172652115040f,
  0.938932483532064600f, -0.344101425989938810f, 0.938668284894770170f,
  -0.344821476901759290f,
  0.938403534063108060f, -0.345541324963989090f, 0.938138231192824360f,
  -0.346260969753160010f,
  0.937872376439989890f, -0.346980410845923680f, 0.937605969960999990f,
  -0.347699647819051380f,
  0.937339011912574960f, -0.348418680249434560f, 0.937071502451759190f,
  -0.349137507714084970f,
  0.936803441735921560f, -0.349856129790134920f, 0.936534829922755500f,
  -0.350574546054837510f,
  0.936265667170278260f, -0.351292756085567090f, 0.935995953636831410f,
  -0.352010759459819080f,
  0.935725689481080370f, -0.352728555755210730f, 0.935454874862014620f,
  -0.353446144549480810f,
  0.935183509938947610f, -0.354163525420490340f, 0.934911594871516090f,
  -0.354880697946222790f,
  0.934639129819680780f, -0.355597661704783850f, 0.934366114943725790f,
  -0.356314416274402410f,
  0.934092550404258980f, -0.357030961233429980f, 0.933818436362210960f,
  -0.357747296160341900f,
  0.933543772978836170f, -0.358463420633736540f, 0.933268560415712050f,
  -0.359179334232336500f,
  0.932992798834738960f, -0.359895036534988110f, 0.932716488398140250f,
  -0.360610527120662270f,
  0.932439629268462360f, -0.361325805568454280f, 0.932162221608574430f,
  -0.362040871457584180f,
  0.931884265581668150f, -0.362755724367397230f, 0.931605761351257830f,
  -0.363470363877363760f,
  0.931326709081180430f, -0.364184789567079890f, 0.931047108935595280f,
  -0.364899001016267320f,
  0.930766961078983710f, -0.365612997804773850f, 0.930486265676149780f,
  -0.366326779512573590f,
  0.930205022892219070f, -0.367040345719767180f, 0.929923232892639670f,
  -0.367753696006581980f,
  0.929640895843181330f, -0.368466829953372320f, 0.929358011909935500f,
  -0.369179747140620020f,
  0.929074581259315860f, -0.369892447148934100f, 0.928790604058057020f,
  -0.370604929559051670f,
  0.928506080473215590f, -0.371317193951837540f, 0.928221010672169440f,
  -0.372029239908285010f,
  0.927935394822617890f, -0.372741067009515760f, 0.927649233092581180f,
  -0.373452674836780300f,
  0.927362525650401110f, -0.374164062971457930f, 0.927075272664740100f,
  -0.374875230995057540f,
  0.926787474304581750f, -0.375586178489217220f, 0.926499130739230510f,
  -0.376296905035704790f,
  0.926210242138311380f, -0.377007410216418260f, 0.925920808671770070f,
  -0.377717693613385640f,
  0.925630830509872720f, -0.378427754808765560f, 0.925340307823206310f,
  -0.379137593384847320f,
  0.925049240782677580f, -0.379847208924051160f, 0.924757629559513910f,
  -0.380556601008928520f,
  0.924465474325262600f, -0.381265769222162380f, 0.924172775251791200f,
  -0.381974713146567220f,
  0.923879532511286740f, -0.382683432365089780f, 0.923585746276256670f,
  -0.383391926460808660f,
  0.923291416719527640f, -0.384100195016935040f, 0.922996544014246250f,
  -0.384808237616812880f,
  0.922701128333878630f, -0.385516053843918850f, 0.922405169852209880f,
  -0.386223643281862980f,
  0.922108668743345180f, -0.386931005514388580f, 0.921811625181708120f,
  -0.387638140125372730f,
  0.921514039342042010f, -0.388345046698826250f, 0.921215911399408730f,
  -0.389051724818894380f,
  0.920917241529189520f, -0.389758174069856410f, 0.920618029907083970f,
  -0.390464394036126590f,
  0.920318276709110590f, -0.391170384302253870f, 0.920017982111606570f,
  -0.391876144452922350f,
  0.919717146291227360f, -0.392581674072951470f, 0.919415769424947070f,
  -0.393286972747296400f,
  0.919113851690057770f, -0.393992040061048100f, 0.918811393264170050f,
  -0.394696875599433560f,
  0.918508394325212250f, -0.395401478947816350f, 0.918204855051430900f,
  -0.396105849691696270f,
  0.917900775621390500f, -0.396809987416710310f, 0.917596156213972950f,
  -0.397513891708632330f,
  0.917290997008377910f, -0.398217562153373560f, 0.916985298184123000f,
  -0.398920998336982910f,
  0.916679059921042700f, -0.399624199845646790f, 0.916372282399289140f,
  -0.400327166265690090f,
  0.916064965799331720f, -0.401029897183575620f, 0.915757110301956720f,
  -0.401732392185905010f,
  0.915448716088267830f, -0.402434650859418430f, 0.915139783339685260f,
  -0.403136672790995300f,
  0.914830312237946200f, -0.403838457567654070f, 0.914520302965104450f,
  -0.404540004776553000f,
  0.914209755703530690f, -0.405241314004989860f, 0.913898670635911680f,
  -0.405942384840402510f,
  0.913587047945250810f, -0.406643216870369030f, 0.913274887814867760f,
  -0.407343809682607970f,
  0.912962190428398210f, -0.408044162864978690f, 0.912648955969793900f,
  -0.408744276005481360f,
  0.912335184623322750f, -0.409444148692257590f, 0.912020876573568340f,
  -0.410143780513590240f,
  0.911706032005429880f, -0.410843171057903910f, 0.911390651104122430f,
  -0.411542319913765220f,
  0.911074734055176360f, -0.412241226669882890f, 0.910758281044437570f,
  -0.412939890915108080f,
  0.910441292258067250f, -0.413638312238434500f, 0.910123767882541680f,
  -0.414336490228999100f,
  0.909805708104652220f, -0.415034424476081630f, 0.909487113111505430f,
  -0.415732114569105360f,
  0.909167983090522380f, -0.416429560097637150f, 0.908848318229439120f,
  -0.417126760651387870f,
  0.908528118716306120f, -0.417823715820212270f, 0.908207384739488700f,
  -0.418520425194109700f,
  0.907886116487666260f, -0.419216888363223910f, 0.907564314149832630f,
  -0.419913104917843620f,
  0.907241977915295820f, -0.420609074448402510f, 0.906919107973678140f,
  -0.421304796545479640f,
  0.906595704514915330f, -0.422000270799799680f, 0.906271767729257660f,
  -0.422695496802232950f,
  0.905947297807268460f, -0.423390474143796050f, 0.905622294939825270f,
  -0.424085202415651560f,
  0.905296759318118820f, -0.424779681209108810f, 0.904970691133653250f,
  -0.425473910115623800f,
  0.904644090578246240f, -0.426167888726799620f, 0.904316957844028320f,
  -0.426861616634386430f,
  0.903989293123443340f, -0.427555093430282080f, 0.903661096609247980f,
  -0.428248318706531960f,
  0.903332368494511820f, -0.428941292055329490f, 0.903003108972617150f,
  -0.429634013069016380f,
  0.902673318237258830f, -0.430326481340082610f, 0.902342996482444200f,
  -0.431018696461167030f,
  0.902012143902493180f, -0.431710658025057260f, 0.901680760692037730f,
  -0.432402365624690140f,
  0.901348847046022030f, -0.433093818853151960f, 0.901016403159702330f,
  -0.433785017303678520f,
  0.900683429228646970f, -0.434475960569655650f, 0.900349925448735600f,
  -0.435166648244619260f,
  0.900015892016160280f, -0.435857079922255470f, 0.899681329127423930f,
  -0.436547255196401200f,
  0.899346236979341570f, -0.437237173661044090f, 0.899010615769039070f,
  -0.437926834910322860f,
  0.898674465693953820f, -0.438616238538527660f, 0.898337786951834310f,
  -0.439305384140099950f,
  0.898000579740739880f, -0.439994271309633260f, 0.897662844259040860f,
  -0.440682899641872900f,
  0.897324580705418320f, -0.441371268731716670f, 0.896985789278863970f,
  -0.442059378174214700f,
  0.896646470178680150f, -0.442747227564570020f, 0.896306623604479550f,
  -0.443434816498138480f,
  0.895966249756185220f, -0.444122144570429200f, 0.895625348834030110f,
  -0.444809211377104880f,
  0.895283921038557580f, -0.445496016513981740f, 0.894941966570620750f,
  -0.446182559577030070f,
  0.894599485631382700f, -0.446868840162374160f, 0.894256478422316040f,
  -0.447554857866293010f,
  0.893912945145203250f, -0.448240612285219890f, 0.893568886002135910f,
  -0.448926103015743260f,
  0.893224301195515320f, -0.449611329654606540f, 0.892879190928051680f,
  -0.450296291798708610f,
  0.892533555402764580f, -0.450980989045103860f, 0.892187394822982480f,
  -0.451665420991002490f,
  0.891840709392342720f, -0.452349587233770890f, 0.891493499314791380f,
  -0.453033487370931580f,
  0.891145764794583180f, -0.453717121000163870f, 0.890797506036281490f,
  -0.454400487719303580f,
  0.890448723244757880f, -0.455083587126343840f, 0.890099416625192320f,
  -0.455766418819434640f,
  0.889749586383072780f, -0.456448982396883920f, 0.889399232724195520f,
  -0.457131277457156980f,
  0.889048355854664570f, -0.457813303598877170f, 0.888696955980891600f,
  -0.458495060420826270f,
  0.888345033309596350f, -0.459176547521944090f, 0.887992588047805560f,
  -0.459857764501329540f,
  0.887639620402853930f, -0.460538710958240010f, 0.887286130582383150f,
  -0.461219386492092380f,
  0.886932118794342190f, -0.461899790702462730f, 0.886577585246987040f,
  -0.462579923189086810f,
  0.886222530148880640f, -0.463259783551860150f, 0.885866953708892790f,
  -0.463939371390838520f,
  0.885510856136199950f, -0.464618686306237820f, 0.885154237640285110f,
  -0.465297727898434600f,
  0.884797098430937790f, -0.465976495767966180f, 0.884439438718253810f,
  -0.466654989515530920f,
  0.884081258712634990f, -0.467333208741988420f, 0.883722558624789660f,
  -0.468011153048359830f,
  0.883363338665731580f, -0.468688822035827900f, 0.883003599046780830f,
  -0.469366215305737520f,
  0.882643339979562790f, -0.470043332459595620f, 0.882282561676008710f,
  -0.470720173099071600f,
  0.881921264348355050f, -0.471396736825997640f, 0.881559448209143780f,
  -0.472073023242368660f,
  0.881197113471222090f, -0.472749031950342790f, 0.880834260347742040f,
  -0.473424762552241530f,
  0.880470889052160750f, -0.474100214650549970f, 0.880106999798240360f,
  -0.474775387847917120f,
  0.879742592800047410f, -0.475450281747155870f, 0.879377668271953290f,
  -0.476124895951243580f,
  0.879012226428633530f, -0.476799230063322090f, 0.878646267485068130f,
  -0.477473283686698060f,
  0.878279791656541580f, -0.478147056424843010f, 0.877912799158641840f,
  -0.478820547881393890f,
  0.877545290207261350f, -0.479493757660153010f, 0.877177265018595940f,
  -0.480166685365088390f,
  0.876808723809145650f, -0.480839330600333960f, 0.876439666795713610f,
  -0.481511692970189860f,
  0.876070094195406600f, -0.482183772079122720f, 0.875700006225634600f,
  -0.482855567531765670f,
  0.875329403104110890f, -0.483527078932918740f, 0.874958285048851650f,
  -0.484198305887549030f,
  0.874586652278176110f, -0.484869248000791060f, 0.874214505010706300f,
  -0.485539904877946960f,
  0.873841843465366860f, -0.486210276124486420f, 0.873468667861384880f,
  -0.486880361346047340f,
  0.873094978418290090f, -0.487550160148436000f, 0.872720775355914300f,
  -0.488219672137626790f,
  0.872346058894391540f, -0.488888896919763170f, 0.871970829254157810f,
  -0.489557834101157440f,
  0.871595086655950980f, -0.490226483288291160f, 0.871218831320811020f,
  -0.490894844087815090f,
  0.870842063470078980f, -0.491562916106549900f, 0.870464783325397670f,
  -0.492230698951486020f,
  0.870086991108711460f, -0.492898192229784040f, 0.869708687042265670f,
  -0.493565395548774770f,
  0.869329871348606840f, -0.494232308515959670f, 0.868950544250582380f,
  -0.494898930739011260f,
  0.868570705971340900f, -0.495565261825772540f, 0.868190356734331310f,
  -0.496231301384258250f,
  0.867809496763303320f, -0.496897049022654470f, 0.867428126282306920f,
  -0.497562504349319150f,
  0.867046245515692650f, -0.498227666972781870f, 0.866663854688111130f,
  -0.498892536501744590f,
  0.866280954024512990f, -0.499557112545081840f, 0.865897543750148820f,
  -0.500221394711840680f,
  0.865513624090569090f, -0.500885382611240710f, 0.865129195271623800f,
  -0.501549075852675390f,
  0.864744257519462380f, -0.502212474045710790f, 0.864358811060534030f,
  -0.502875576800086990f,
  0.863972856121586810f, -0.503538383725717580f, 0.863586392929668100f,
  -0.504200894432690340f,
  0.863199421712124160f, -0.504863108531267590f, 0.862811942696600330f,
  -0.505525025631885390f,
  0.862423956111040610f, -0.506186645345155230f, 0.862035462183687210f,
  -0.506847967281863210f,
  0.861646461143081300f, -0.507508991052970870f, 0.861256953218062170f,
  -0.508169716269614600f,
  0.860866938637767310f, -0.508830142543106990f, 0.860476417631632070f,
  -0.509490269484936360f,
  0.860085390429390140f, -0.510150096706766810f, 0.859693857261072610f,
  -0.510809623820439040f,
  0.859301818357008470f, -0.511468850437970300f, 0.858909273947823900f,
  -0.512127776171554690f,
  0.858516224264442740f, -0.512786400633562960f, 0.858122669538086140f,
  -0.513444723436543460f,
  0.857728610000272120f, -0.514102744193221660f, 0.857334045882815590f,
  -0.514760462516501200f,
  0.856938977417828760f, -0.515417878019462930f, 0.856543404837719960f,
  -0.516074990315366630f,
  0.856147328375194470f, -0.516731799017649870f, 0.855750748263253920f,
  -0.517388303739929060f,
  0.855353664735196030f, -0.518044504095999340f, 0.854956078024614930f,
  -0.518700399699834950f,
  0.854557988365400530f, -0.519355990165589640f, 0.854159395991738850f,
  -0.520011275107596040f,
  0.853760301138111410f, -0.520666254140367160f, 0.853360704039295430f,
  -0.521320926878595660f,
  0.852960604930363630f, -0.521975292937154390f, 0.852560004046684080f,
  -0.522629351931096610f,
  0.852158901623919830f, -0.523283103475656430f, 0.851757297898029120f,
  -0.523936547186248600f,
  0.851355193105265200f, -0.524589682678468950f, 0.850952587482175730f,
  -0.525242509568094710f,
  0.850549481265603480f, -0.525895027471084630f, 0.850145874692685210f,
  -0.526547236003579440f,
  0.849741768000852550f, -0.527199134781901280f, 0.849337161427830780f,
  -0.527850723422555230f,
  0.848932055211639610f, -0.528502001542228480f, 0.848526449590592650f,
  -0.529152968757790610f,
  0.848120344803297230f, -0.529803624686294610f, 0.847713741088654380f,
  -0.530453968944976320f,
  0.847306638685858320f, -0.531104001151255000f, 0.846899037834397240f,
  -0.531753720922733320f,
  0.846490938774052130f, -0.532403127877197900f, 0.846082341744897050f,
  -0.533052221632619450f,
  0.845673246987299070f, -0.533701001807152960f, 0.845263654741918220f,
  -0.534349468019137520f,
  0.844853565249707120f, -0.534997619887097150f, 0.844442978751910660f,
  -0.535645457029741090f,
  0.844031895490066410f, -0.536292979065963180f, 0.843620315706004150f,
  -0.536940185614842910f,
  0.843208239641845440f, -0.537587076295645390f, 0.842795667540004120f,
  -0.538233650727821700f,
  0.842382599643185850f, -0.538879908531008420f, 0.841969036194387680f,
  -0.539525849325028890f,
  0.841554977436898440f, -0.540171472729892850f, 0.841140423614298080f,
  -0.540816778365796670f,
  0.840725374970458070f, -0.541461765853123440f, 0.840309831749540770f,
  -0.542106434812443920f,
  0.839893794195999520f, -0.542750784864515890f, 0.839477262554578550f,
  -0.543394815630284800f,
  0.839060237070312740f, -0.544038526730883820f, 0.838642717988527300f,
  -0.544681917787634530f,
  0.838224705554838080f, -0.545324988422046460f, 0.837806200015150940f,
  -0.545967738255817570f,
  0.837387201615661940f, -0.546610166910834860f, 0.836967710602857020f,
  -0.547252274009174090f,
  0.836547727223512010f, -0.547894059173100190f, 0.836127251724692270f,
  -0.548535522025067390f,
  0.835706284353752600f, -0.549176662187719660f, 0.835284825358337370f,
  -0.549817479283890910f,
  0.834862874986380010f, -0.550457972936604810f, 0.834440433486103190f,
  -0.551098142769075430f,
  0.834017501106018130f, -0.551737988404707340f, 0.833594078094925140f,
  -0.552377509467096070f,
  0.833170164701913190f, -0.553016705580027470f, 0.832745761176359460f,
  -0.553655576367479310f,
  0.832320867767929680f, -0.554294121453620000f, 0.831895484726577590f,
  -0.554932340462810370f,
  0.831469612302545240f, -0.555570233019602180f, 0.831043250746362320f,
  -0.556207798748739930f,
  0.830616400308846310f, -0.556845037275160100f, 0.830189061241102370f,
  -0.557481948223991550f,
  0.829761233794523050f, -0.558118531220556100f, 0.829332918220788250f,
  -0.558754785890368310f,
  0.828904114771864870f, -0.559390711859136140f, 0.828474823700007130f,
  -0.560026308752760380f,
  0.828045045257755800f, -0.560661576197336030f, 0.827614779697938400f,
  -0.561296513819151470f,
  0.827184027273669130f, -0.561931121244689470f, 0.826752788238348520f,
  -0.562565398100626560f,
  0.826321062845663530f, -0.563199344013834090f, 0.825888851349586780f,
  -0.563832958611378170f,
  0.825456154004377550f, -0.564466241520519500f, 0.825022971064580220f,
  -0.565099192368713980f,
  0.824589302785025290f, -0.565731810783613120f, 0.824155149420828570f,
  -0.566364096393063840f,
  0.823720511227391430f, -0.566996048825108680f, 0.823285388460400110f,
  -0.567627667707986230f,
  0.822849781375826430f, -0.568258952670131490f, 0.822413690229926390f,
  -0.568889903340175860f,
  0.821977115279241550f, -0.569520519346947140f, 0.821540056780597610f,
  -0.570150800319470300f,
  0.821102514991104650f, -0.570780745886967260f, 0.820664490168157460f,
  -0.571410355678857230f,
  0.820225982569434690f, -0.572039629324757050f, 0.819786992452898990f,
  -0.572668566454481160f,
  0.819347520076796900f, -0.573297166698042200f, 0.818907565699658950f,
  -0.573925429685650750f,
  0.818467129580298660f, -0.574553355047715760f, 0.818026211977813440f,
  -0.575180942414845080f,
  0.817584813151583710f, -0.575808191417845340f, 0.817142933361272970f,
  -0.576435101687721830f,
  0.816700572866827850f, -0.577061672855679440f, 0.816257731928477390f,
  -0.577687904553122800f,
  0.815814410806733780f, -0.578313796411655590f, 0.815370609762391290f,
  -0.578939348063081780f,
  0.814926329056526620f, -0.579564559139405630f, 0.814481568950498610f,
  -0.580189429272831680f,
  0.814036329705948410f, -0.580813958095764530f, 0.813590611584798510f,
  -0.581438145240810170f,
  0.813144414849253590f, -0.582061990340775440f, 0.812697739761799490f,
  -0.582685493028668460f,
  0.812250586585203880f, -0.583308652937698290f, 0.811802955582515470f,
  -0.583931469701276180f,
  0.811354847017063730f, -0.584553942953015330f, 0.810906261152459670f,
  -0.585176072326730410f,
  0.810457198252594770f, -0.585797857456438860f, 0.810007658581641140f,
  -0.586419297976360500f,
  0.809557642404051260f, -0.587040393520917970f, 0.809107149984558240f,
  -0.587661143724736660f,
  0.808656181588174980f, -0.588281548222645220f, 0.808204737480194720f,
  -0.588901606649675720f,
  0.807752817926190360f, -0.589521318641063940f, 0.807300423192014450f,
  -0.590140683832248820f,
  0.806847553543799330f, -0.590759701858874160f, 0.806394209247956240f,
  -0.591378372356787580f,
  0.805940390571176280f, -0.591996694962040990f, 0.805486097780429230f,
  -0.592614669310891130f,
  0.805031331142963660f, -0.593232295039799800f, 0.804576090926307110f,
  -0.593849571785433630f,
  0.804120377398265810f, -0.594466499184664430f, 0.803664190826924090f,
  -0.595083076874569960f,
  0.803207531480644940f, -0.595699304492433360f, 0.802750399628069160f,
  -0.596315181675743710f,
  0.802292795538115720f, -0.596930708062196500f, 0.801834719479981310f,
  -0.597545883289693160f,
  0.801376171723140240f, -0.598160706996342270f, 0.800917152537344300f,
  -0.598775178820458720f,
  0.800457662192622820f, -0.599389298400564540f, 0.799997700959281910f,
  -0.600003065375388940f,
  0.799537269107905010f, -0.600616479383868970f, 0.799076366909352350f,
  -0.601229540065148500f,
  0.798614994634760820f, -0.601842247058580030f, 0.798153152555543750f,
  -0.602454600003723750f,
  0.797690840943391160f, -0.603066598540348160f, 0.797228060070268810f,
  -0.603678242308430370f,
  0.796764810208418830f, -0.604289530948155960f, 0.796301091630359110f,
  -0.604900464099919820f,
  0.795836904608883570f, -0.605511041404325550f, 0.795372249417061310f,
  -0.606121262502186120f,
  0.794907126328237010f, -0.606731127034524480f, 0.794441535616030590f,
  -0.607340634642572930f,
  0.793975477554337170f, -0.607949784967773630f, 0.793508952417326660f,
  -0.608558577651779450f,
  0.793041960479443640f, -0.609167012336453210f, 0.792574502015407690f,
  -0.609775088663868430f,
  0.792106577300212390f, -0.610382806276309480f, 0.791638186609125880f,
  -0.610990164816271660f,
  0.791169330217690200f, -0.611597163926461910f, 0.790700008401721610f,
  -0.612203803249797950f,
  0.790230221437310030f, -0.612810082429409710f, 0.789759969600819070f,
  -0.613416001108638590f,
  0.789289253168885650f, -0.614021558931038380f, 0.788818072418420280f,
  -0.614626755540375050f,
  0.788346427626606340f, -0.615231590580626820f, 0.787874319070900220f,
  -0.615836063695985090f,
  0.787401747029031430f, -0.616440174530853650f, 0.786928711779001810f,
  -0.617043922729849760f,
  0.786455213599085770f, -0.617647307937803870f, 0.785981252767830150f,
  -0.618250329799760250f,
  0.785506829564053930f, -0.618852987960976320f, 0.785031944266848080f,
  -0.619455282066924020f,
  0.784556597155575240f, -0.620057211763289100f, 0.784080788509869950f,
  -0.620658776695972140f,
  0.783604518609638200f, -0.621259976511087550f, 0.783127787735057310f,
  -0.621860810854965360f,
  0.782650596166575730f, -0.622461279374149970f, 0.782172944184913010f,
  -0.623061381715401260f,
  0.781694832071059390f, -0.623661117525694530f, 0.781216260106276090f,
  -0.624260486452220650f,
  0.780737228572094490f, -0.624859488142386340f, 0.780257737750316590f,
  -0.625458122243814360f,
  0.779777787923014550f, -0.626056388404343520f, 0.779297379372530300f,
  -0.626654286272029350f,
  0.778816512381475980f, -0.627251815495144080f, 0.778335187232733210f,
  -0.627848975722176460f,
  0.777853404209453150f, -0.628445766601832710f, 0.777371163595056310f,
  -0.629042187783036000f,
  0.776888465673232440f, -0.629638238914926980f, 0.776405310727940390f,
  -0.630233919646864370f,
  0.775921699043407690f, -0.630829229628424470f, 0.775437630904130540f,
  -0.631424168509401860f,
  0.774953106594873930f, -0.632018735939809060f, 0.774468126400670860f,
  -0.632612931569877410f,
  0.773982690606822900f, -0.633206755050057190f, 0.773496799498899050f,
  -0.633800206031017280f,
  0.773010453362736990f, -0.634393284163645490f, 0.772523652484441330f,
  -0.634985989099049460f,
  0.772036397150384520f, -0.635578320488556110f, 0.771548687647206300f,
  -0.636170277983712170f,
  0.771060524261813820f, -0.636761861236284200f, 0.770571907281380810f,
  -0.637353069898259130f,
  0.770082836993347900f, -0.637943903621844060f, 0.769593313685422940f,
  -0.638534362059466790f,
  0.769103337645579700f, -0.639124444863775730f, 0.768612909162058380f,
  -0.639714151687640450f,
  0.768122028523365420f, -0.640303482184151670f, 0.767630696018273380f,
  -0.640892436006621380f,
  0.767138911935820400f, -0.641481012808583160f, 0.766646676565310380f,
  -0.642069212243792540f,
  0.766153990196312920f, -0.642657033966226860f, 0.765660853118662500f,
  -0.643244477630085850f,
  0.765167265622458960f, -0.643831542889791390f, 0.764673227998067140f,
  -0.644418229399988380f,
  0.764178740536116670f, -0.645004536815543930f, 0.763683803527501870f,
  -0.645590464791548690f,
  0.763188417263381270f, -0.646176012983316280f, 0.762692582035177980f,
  -0.646761181046383920f,
  0.762196298134578900f, -0.647345968636512060f, 0.761699565853535380f,
  -0.647930375409685340f,
  0.761202385484261780f, -0.648514401022112440f, 0.760704757319236920f,
  -0.649098045130225950f,
  0.760206681651202420f, -0.649681307390683190f, 0.759708158773163440f,
  -0.650264187460365850f,
  0.759209188978388070f, -0.650846684996380880f, 0.758709772560407390f,
  -0.651428799656059820f,
  0.758209909813015280f, -0.652010531096959500f, 0.757709601030268080f,
  -0.652591878976862440f,
  0.757208846506484570f, -0.653172842953776760f, 0.756707646536245670f,
  -0.653753422685936060f,
  0.756206001414394540f, -0.654333617831800440f, 0.755703911436035880f,
  -0.654913428050056030f,
  0.755201376896536550f, -0.655492852999615350f, 0.754698398091524500f,
  -0.656071892339617600f,
  0.754194975316889170f, -0.656650545729428940f, 0.753691108868781210f,
  -0.657228812828642540f,
  0.753186799043612520f, -0.657806693297078640f, 0.752682046138055340f,
  -0.658384186794785050f,
  0.752176850449042810f, -0.658961292982037320f, 0.751671212273768430f,
  -0.659538011519338660f,
  0.751165131909686480f, -0.660114342067420480f, 0.750658609654510700f,
  -0.660690284287242300f,
  0.750151645806215070f, -0.661265837839992270f, 0.749644240663033480f,
  -0.661841002387086870f,
  0.749136394523459370f, -0.662415777590171780f, 0.748628107686245440f,
  -0.662990163111121470f,
  0.748119380450403600f, -0.663564158612039770f, 0.747610213115205150f,
  -0.664137763755260010f,
  0.747100605980180130f, -0.664710978203344790f, 0.746590559345117310f,
  -0.665283801619087180f,
  0.746080073510063780f, -0.665856233665509720f, 0.745569148775325430f,
  -0.666428274005865240f,
  0.745057785441466060f, -0.666999922303637470f, 0.744545983809307370f,
  -0.667571178222540310f,
  0.744033744179929290f, -0.668142041426518450f, 0.743521066854669120f,
  -0.668712511579747980f,
  0.743007952135121720f, -0.669282588346636010f, 0.742494400323139180f,
  -0.669852271391821020f,
  0.741980411720831070f, -0.670421560380173090f, 0.741465986630563290f,
  -0.670990454976794220f,
  0.740951125354959110f, -0.671558954847018330f, 0.740435828196898020f,
  -0.672127059656411730f,
  0.739920095459516200f, -0.672694769070772860f, 0.739403927446205760f,
  -0.673262082756132970f,
  0.738887324460615110f, -0.673829000378756040f, 0.738370286806648620f,
  -0.674395521605139050f,
  0.737852814788465980f, -0.674961646102011930f, 0.737334908710482910f,
  -0.675527373536338520f,
  0.736816568877369900f, -0.676092703575315920f, 0.736297795594053170f,
  -0.676657635886374950f,
  0.735778589165713590f, -0.677222170137180330f, 0.735258949897786840f,
  -0.677786305995631500f,
  0.734738878095963500f, -0.678350043129861470f, 0.734218374066188280f,
  -0.678913381208238410f,
  0.733697438114660370f, -0.679476319899364970f, 0.733176070547832740f,
  -0.680038858872078930f,
  0.732654271672412820f, -0.680600997795453020f, 0.732132041795361290f,
  -0.681162736338795430f,
  0.731609381223892630f, -0.681724074171649710f, 0.731086290265474340f,
  -0.682285010963795570f,
  0.730562769227827590f, -0.682845546385248080f, 0.730038818418926260f,
  -0.683405680106258680f,
  0.729514438146997010f, -0.683965411797315400f, 0.728989628720519420f,
  -0.684524741129142300f,
  0.728464390448225200f, -0.685083667772700360f, 0.727938723639098620f,
  -0.685642191399187470f,
  0.727412628602375770f, -0.686200311680038590f, 0.726886105647544970f,
  -0.686758028286925890f,
  0.726359155084346010f, -0.687315340891759050f, 0.725831777222770370f,
  -0.687872249166685550f,
  0.725303972373060770f, -0.688428752784090440f, 0.724775740845711280f,
  -0.688984851416597040f,
  0.724247082951467000f, -0.689540544737066830f, 0.723717999001323500f,
  -0.690095832418599950f,
  0.723188489306527460f, -0.690650714134534600f, 0.722658554178575610f,
  -0.691205189558448450f,
  0.722128193929215350f, -0.691759258364157750f, 0.721597408870443770f,
  -0.692312920225718220f,
  0.721066199314508110f, -0.692866174817424630f, 0.720534565573905270f,
  -0.693419021813811760f,
  0.720002507961381650f, -0.693971460889654000f, 0.719470026789932990f,
  -0.694523491719965520f,
  0.718937122372804490f, -0.695075113980000880f, 0.718403795023489830f,
  -0.695626327345254870f,
  0.717870045055731710f, -0.696177131491462990f, 0.717335872783521730f,
  -0.696727526094601200f,
  0.716801278521099540f, -0.697277510830886520f, 0.716266262582953120f,
  -0.697827085376777290f,
  0.715730825283818590f, -0.698376249408972920f, 0.715194966938680120f,
  -0.698925002604414150f,
  0.714658687862769090f, -0.699473344640283770f, 0.714121988371564820f,
  -0.700021275194006250f,
  0.713584868780793640f, -0.700568793943248340f, 0.713047329406429340f,
  -0.701115900565918660f,
  0.712509370564692320f, -0.701662594740168450f, 0.711970992572050100f,
  -0.702208876144391870f,
  0.711432195745216430f, -0.702754744457225300f, 0.710892980401151680f,
  -0.703300199357548730f,
  0.710353346857062420f, -0.703845240524484940f, 0.709813295430400840f,
  -0.704389867637400410f,
  0.709272826438865690f, -0.704934080375904880f, 0.708731940200400650f,
  -0.705477878419852100f,
  0.708190637033195400f, -0.706021261449339740f, 0.707648917255684350f,
  -0.706564229144709510f,
  0.707106781186547570f, -0.707106781186547460f, 0.706564229144709620f,
  -0.707648917255684350f,
  0.706021261449339740f, -0.708190637033195290f, 0.705477878419852210f,
  -0.708731940200400650f,
  0.704934080375904990f, -0.709272826438865580f, 0.704389867637400410f,
  -0.709813295430400840f,
  0.703845240524484940f, -0.710353346857062310f, 0.703300199357548730f,
  -0.710892980401151680f,
  0.702754744457225300f, -0.711432195745216430f, 0.702208876144391870f,
  -0.711970992572049990f,
  0.701662594740168570f, -0.712509370564692320f, 0.701115900565918660f,
  -0.713047329406429230f,
  0.700568793943248450f, -0.713584868780793520f, 0.700021275194006360f,
  -0.714121988371564710f,
  0.699473344640283770f, -0.714658687862768980f, 0.698925002604414150f,
  -0.715194966938680010f,
  0.698376249408972920f, -0.715730825283818590f, 0.697827085376777290f,
  -0.716266262582953120f,
  0.697277510830886630f, -0.716801278521099540f, 0.696727526094601200f,
  -0.717335872783521730f,
  0.696177131491462990f, -0.717870045055731710f, 0.695626327345254870f,
  -0.718403795023489720f,
  0.695075113980000880f, -0.718937122372804380f, 0.694523491719965520f,
  -0.719470026789932990f,
  0.693971460889654000f, -0.720002507961381650f, 0.693419021813811880f,
  -0.720534565573905270f,
  0.692866174817424740f, -0.721066199314508110f, 0.692312920225718220f,
  -0.721597408870443660f,
  0.691759258364157750f, -0.722128193929215350f, 0.691205189558448450f,
  -0.722658554178575610f,
  0.690650714134534720f, -0.723188489306527350f, 0.690095832418599950f,
  -0.723717999001323390f,
  0.689540544737066940f, -0.724247082951466890f, 0.688984851416597150f,
  -0.724775740845711280f,
  0.688428752784090550f, -0.725303972373060660f, 0.687872249166685550f,
  -0.725831777222770370f,
  0.687315340891759160f, -0.726359155084346010f, 0.686758028286925890f,
  -0.726886105647544970f,
  0.686200311680038700f, -0.727412628602375770f, 0.685642191399187470f,
  -0.727938723639098620f,
  0.685083667772700360f, -0.728464390448225200f, 0.684524741129142300f,
  -0.728989628720519310f,
  0.683965411797315510f, -0.729514438146996900f, 0.683405680106258790f,
  -0.730038818418926150f,
  0.682845546385248080f, -0.730562769227827590f, 0.682285010963795570f,
  -0.731086290265474230f,
  0.681724074171649820f, -0.731609381223892520f, 0.681162736338795430f,
  -0.732132041795361290f,
  0.680600997795453130f, -0.732654271672412820f, 0.680038858872079040f,
  -0.733176070547832740f,
  0.679476319899365080f, -0.733697438114660260f, 0.678913381208238410f,
  -0.734218374066188170f,
  0.678350043129861580f, -0.734738878095963390f, 0.677786305995631500f,
  -0.735258949897786730f,
  0.677222170137180450f, -0.735778589165713480f, 0.676657635886374950f,
  -0.736297795594053060f,
  0.676092703575316030f, -0.736816568877369790f, 0.675527373536338630f,
  -0.737334908710482790f,
  0.674961646102012040f, -0.737852814788465980f, 0.674395521605139050f,
  -0.738370286806648510f,
  0.673829000378756150f, -0.738887324460615110f, 0.673262082756132970f,
  -0.739403927446205760f,
  0.672694769070772970f, -0.739920095459516090f, 0.672127059656411840f,
  -0.740435828196898020f,
  0.671558954847018330f, -0.740951125354959110f, 0.670990454976794220f,
  -0.741465986630563290f,
  0.670421560380173090f, -0.741980411720830960f, 0.669852271391821130f,
  -0.742494400323139180f,
  0.669282588346636010f, -0.743007952135121720f, 0.668712511579748090f,
  -0.743521066854669120f,
  0.668142041426518560f, -0.744033744179929180f, 0.667571178222540310f,
  -0.744545983809307250f,
  0.666999922303637470f, -0.745057785441465950f, 0.666428274005865350f,
  -0.745569148775325430f,
  0.665856233665509720f, -0.746080073510063780f, 0.665283801619087180f,
  -0.746590559345117310f,
  0.664710978203344900f, -0.747100605980180130f, 0.664137763755260010f,
  -0.747610213115205150f,
  0.663564158612039880f, -0.748119380450403490f, 0.662990163111121470f,
  -0.748628107686245330f,
  0.662415777590171780f, -0.749136394523459260f, 0.661841002387086870f,
  -0.749644240663033480f,
  0.661265837839992270f, -0.750151645806214960f, 0.660690284287242300f,
  -0.750658609654510590f,
  0.660114342067420480f, -0.751165131909686370f, 0.659538011519338770f,
  -0.751671212273768430f,
  0.658961292982037320f, -0.752176850449042700f, 0.658384186794785050f,
  -0.752682046138055230f,
  0.657806693297078640f, -0.753186799043612410f, 0.657228812828642650f,
  -0.753691108868781210f,
  0.656650545729429050f, -0.754194975316889170f, 0.656071892339617710f,
  -0.754698398091524390f,
  0.655492852999615460f, -0.755201376896536550f, 0.654913428050056150f,
  -0.755703911436035880f,
  0.654333617831800550f, -0.756206001414394540f, 0.653753422685936170f,
  -0.756707646536245670f,
  0.653172842953776760f, -0.757208846506484460f, 0.652591878976862550f,
  -0.757709601030268080f,
  0.652010531096959500f, -0.758209909813015280f, 0.651428799656059820f,
  -0.758709772560407390f,
  0.650846684996380990f, -0.759209188978387960f, 0.650264187460365960f,
  -0.759708158773163440f,
  0.649681307390683190f, -0.760206681651202420f, 0.649098045130226060f,
  -0.760704757319236920f,
  0.648514401022112550f, -0.761202385484261780f, 0.647930375409685460f,
  -0.761699565853535270f,
  0.647345968636512060f, -0.762196298134578900f, 0.646761181046383920f,
  -0.762692582035177870f,
  0.646176012983316390f, -0.763188417263381270f, 0.645590464791548800f,
  -0.763683803527501870f,
  0.645004536815544040f, -0.764178740536116670f, 0.644418229399988380f,
  -0.764673227998067140f,
  0.643831542889791500f, -0.765167265622458960f, 0.643244477630085850f,
  -0.765660853118662390f,
  0.642657033966226860f, -0.766153990196312810f, 0.642069212243792540f,
  -0.766646676565310380f,
  0.641481012808583160f, -0.767138911935820400f, 0.640892436006621380f,
  -0.767630696018273270f,
  0.640303482184151670f, -0.768122028523365310f, 0.639714151687640450f,
  -0.768612909162058270f,
  0.639124444863775730f, -0.769103337645579590f, 0.638534362059466790f,
  -0.769593313685422940f,
  0.637943903621844170f, -0.770082836993347900f, 0.637353069898259130f,
  -0.770571907281380700f,
  0.636761861236284200f, -0.771060524261813710f, 0.636170277983712170f,
  -0.771548687647206300f,
  0.635578320488556230f, -0.772036397150384410f, 0.634985989099049460f,
  -0.772523652484441330f,
  0.634393284163645490f, -0.773010453362736990f, 0.633800206031017280f,
  -0.773496799498899050f,
  0.633206755050057190f, -0.773982690606822790f, 0.632612931569877520f,
  -0.774468126400670860f,
  0.632018735939809060f, -0.774953106594873820f, 0.631424168509401860f,
  -0.775437630904130430f,
  0.630829229628424470f, -0.775921699043407580f, 0.630233919646864480f,
  -0.776405310727940390f,
  0.629638238914927100f, -0.776888465673232440f, 0.629042187783036000f,
  -0.777371163595056200f,
  0.628445766601832710f, -0.777853404209453040f, 0.627848975722176570f,
  -0.778335187232733090f,
  0.627251815495144190f, -0.778816512381475870f, 0.626654286272029460f,
  -0.779297379372530300f,
  0.626056388404343520f, -0.779777787923014440f, 0.625458122243814360f,
  -0.780257737750316590f,
  0.624859488142386450f, -0.780737228572094380f, 0.624260486452220650f,
  -0.781216260106276090f,
  0.623661117525694640f, -0.781694832071059390f, 0.623061381715401370f,
  -0.782172944184912900f,
  0.622461279374150080f, -0.782650596166575730f, 0.621860810854965360f,
  -0.783127787735057310f,
  0.621259976511087660f, -0.783604518609638200f, 0.620658776695972140f,
  -0.784080788509869950f,
  0.620057211763289210f, -0.784556597155575240f, 0.619455282066924020f,
  -0.785031944266848080f,
  0.618852987960976320f, -0.785506829564053930f, 0.618250329799760250f,
  -0.785981252767830150f,
  0.617647307937803980f, -0.786455213599085770f, 0.617043922729849760f,
  -0.786928711779001700f,
  0.616440174530853650f, -0.787401747029031320f, 0.615836063695985090f,
  -0.787874319070900110f,
  0.615231590580626820f, -0.788346427626606230f, 0.614626755540375050f,
  -0.788818072418420170f,
  0.614021558931038490f, -0.789289253168885650f, 0.613416001108638590f,
  -0.789759969600819070f,
  0.612810082429409710f, -0.790230221437310030f, 0.612203803249798060f,
  -0.790700008401721610f,
  0.611597163926462020f, -0.791169330217690090f, 0.610990164816271770f,
  -0.791638186609125770f,
  0.610382806276309480f, -0.792106577300212390f, 0.609775088663868430f,
  -0.792574502015407580f,
  0.609167012336453210f, -0.793041960479443640f, 0.608558577651779450f,
  -0.793508952417326660f,
  0.607949784967773740f, -0.793975477554337170f, 0.607340634642572930f,
  -0.794441535616030590f,
  0.606731127034524480f, -0.794907126328237010f, 0.606121262502186230f,
  -0.795372249417061190f,
  0.605511041404325550f, -0.795836904608883460f, 0.604900464099919930f,
  -0.796301091630359110f,
  0.604289530948156070f, -0.796764810208418720f, 0.603678242308430370f,
  -0.797228060070268700f,
  0.603066598540348280f, -0.797690840943391040f, 0.602454600003723860f,
  -0.798153152555543750f,
  0.601842247058580030f, -0.798614994634760820f, 0.601229540065148620f,
  -0.799076366909352350f,
  0.600616479383868970f, -0.799537269107905010f, 0.600003065375389060f,
  -0.799997700959281910f,
  0.599389298400564540f, -0.800457662192622710f, 0.598775178820458720f,
  -0.800917152537344300f,
  0.598160706996342380f, -0.801376171723140130f, 0.597545883289693270f,
  -0.801834719479981310f,
  0.596930708062196500f, -0.802292795538115720f, 0.596315181675743820f,
  -0.802750399628069160f,
  0.595699304492433470f, -0.803207531480644830f, 0.595083076874569960f,
  -0.803664190826924090f,
  0.594466499184664540f, -0.804120377398265700f, 0.593849571785433630f,
  -0.804576090926307000f,
  0.593232295039799800f, -0.805031331142963660f, 0.592614669310891130f,
  -0.805486097780429120f,
  0.591996694962040990f, -0.805940390571176280f, 0.591378372356787580f,
  -0.806394209247956240f,
  0.590759701858874280f, -0.806847553543799220f, 0.590140683832248940f,
  -0.807300423192014450f,
  0.589521318641063940f, -0.807752817926190360f, 0.588901606649675840f,
  -0.808204737480194720f,
  0.588281548222645330f, -0.808656181588174980f, 0.587661143724736770f,
  -0.809107149984558130f,
  0.587040393520918080f, -0.809557642404051260f, 0.586419297976360500f,
  -0.810007658581641140f,
  0.585797857456438860f, -0.810457198252594770f, 0.585176072326730410f,
  -0.810906261152459670f,
  0.584553942953015330f, -0.811354847017063730f, 0.583931469701276300f,
  -0.811802955582515360f,
  0.583308652937698290f, -0.812250586585203880f, 0.582685493028668460f,
  -0.812697739761799490f,
  0.582061990340775550f, -0.813144414849253590f, 0.581438145240810280f,
  -0.813590611584798510f,
  0.580813958095764530f, -0.814036329705948300f, 0.580189429272831680f,
  -0.814481568950498610f,
  0.579564559139405740f, -0.814926329056526620f, 0.578939348063081890f,
  -0.815370609762391290f,
  0.578313796411655590f, -0.815814410806733780f, 0.577687904553122800f,
  -0.816257731928477390f,
  0.577061672855679550f, -0.816700572866827850f, 0.576435101687721830f,
  -0.817142933361272970f,
  0.575808191417845340f, -0.817584813151583710f, 0.575180942414845190f,
  -0.818026211977813440f,
  0.574553355047715760f, -0.818467129580298660f, 0.573925429685650750f,
  -0.818907565699658950f,
  0.573297166698042320f, -0.819347520076796900f, 0.572668566454481160f,
  -0.819786992452898990f,
  0.572039629324757050f, -0.820225982569434690f, 0.571410355678857340f,
  -0.820664490168157460f,
  0.570780745886967370f, -0.821102514991104650f, 0.570150800319470300f,
  -0.821540056780597610f,
  0.569520519346947250f, -0.821977115279241550f, 0.568889903340175970f,
  -0.822413690229926390f,
  0.568258952670131490f, -0.822849781375826320f, 0.567627667707986230f,
  -0.823285388460400110f,
  0.566996048825108680f, -0.823720511227391320f, 0.566364096393063950f,
  -0.824155149420828570f,
  0.565731810783613230f, -0.824589302785025290f, 0.565099192368714090f,
  -0.825022971064580220f,
  0.564466241520519500f, -0.825456154004377440f, 0.563832958611378170f,
  -0.825888851349586780f,
  0.563199344013834090f, -0.826321062845663420f, 0.562565398100626560f,
  -0.826752788238348520f,
  0.561931121244689470f, -0.827184027273669020f, 0.561296513819151470f,
  -0.827614779697938400f,
  0.560661576197336030f, -0.828045045257755800f, 0.560026308752760380f,
  -0.828474823700007130f,
  0.559390711859136140f, -0.828904114771864870f, 0.558754785890368310f,
  -0.829332918220788250f,
  0.558118531220556100f, -0.829761233794523050f, 0.557481948223991660f,
  -0.830189061241102370f,
  0.556845037275160100f, -0.830616400308846200f, 0.556207798748739930f,
  -0.831043250746362320f,
  0.555570233019602290f, -0.831469612302545240f, 0.554932340462810370f,
  -0.831895484726577590f,
  0.554294121453620110f, -0.832320867767929680f, 0.553655576367479310f,
  -0.832745761176359460f,
  0.553016705580027580f, -0.833170164701913190f, 0.552377509467096070f,
  -0.833594078094925140f,
  0.551737988404707450f, -0.834017501106018130f, 0.551098142769075430f,
  -0.834440433486103190f,
  0.550457972936604810f, -0.834862874986380010f, 0.549817479283891020f,
  -0.835284825358337370f,
  0.549176662187719770f, -0.835706284353752600f, 0.548535522025067390f,
  -0.836127251724692160f,
  0.547894059173100190f, -0.836547727223511890f, 0.547252274009174090f,
  -0.836967710602857020f,
  0.546610166910834860f, -0.837387201615661940f, 0.545967738255817680f,
  -0.837806200015150940f,
  0.545324988422046460f, -0.838224705554837970f, 0.544681917787634530f,
  -0.838642717988527300f,
  0.544038526730883930f, -0.839060237070312630f, 0.543394815630284800f,
  -0.839477262554578550f,
  0.542750784864516000f, -0.839893794195999410f, 0.542106434812444030f,
  -0.840309831749540770f,
  0.541461765853123560f, -0.840725374970458070f, 0.540816778365796670f,
  -0.841140423614298080f,
  0.540171472729892970f, -0.841554977436898330f, 0.539525849325029010f,
  -0.841969036194387680f,
  0.538879908531008420f, -0.842382599643185960f, 0.538233650727821700f,
  -0.842795667540004120f,
  0.537587076295645510f, -0.843208239641845440f, 0.536940185614843020f,
  -0.843620315706004040f,
  0.536292979065963180f, -0.844031895490066410f, 0.535645457029741090f,
  -0.844442978751910660f,
  0.534997619887097260f, -0.844853565249707010f, 0.534349468019137520f,
  -0.845263654741918220f,
  0.533701001807152960f, -0.845673246987299070f, 0.533052221632619670f,
  -0.846082341744896940f,
  0.532403127877198010f, -0.846490938774052020f, 0.531753720922733320f,
  -0.846899037834397350f,
  0.531104001151255000f, -0.847306638685858320f, 0.530453968944976320f,
  -0.847713741088654270f,
  0.529803624686294830f, -0.848120344803297120f, 0.529152968757790720f,
  -0.848526449590592650f,
  0.528502001542228480f, -0.848932055211639610f, 0.527850723422555460f,
  -0.849337161427830670f,
  0.527199134781901390f, -0.849741768000852440f, 0.526547236003579330f,
  -0.850145874692685210f,
  0.525895027471084740f, -0.850549481265603370f, 0.525242509568094710f,
  -0.850952587482175730f,
  0.524589682678468840f, -0.851355193105265200f, 0.523936547186248600f,
  -0.851757297898029120f,
  0.523283103475656430f, -0.852158901623919830f, 0.522629351931096720f,
  -0.852560004046683970f,
  0.521975292937154390f, -0.852960604930363630f, 0.521320926878595550f,
  -0.853360704039295430f,
  0.520666254140367270f, -0.853760301138111300f, 0.520011275107596040f,
  -0.854159395991738730f,
  0.519355990165589530f, -0.854557988365400530f, 0.518700399699835170f,
  -0.854956078024614820f,
  0.518044504095999340f, -0.855353664735196030f, 0.517388303739929060f,
  -0.855750748263253920f,
  0.516731799017649980f, -0.856147328375194470f, 0.516074990315366630f,
  -0.856543404837719960f,
  0.515417878019463150f, -0.856938977417828650f, 0.514760462516501200f,
  -0.857334045882815590f,
  0.514102744193221660f, -0.857728610000272120f, 0.513444723436543570f,
  -0.858122669538086020f,
  0.512786400633563070f, -0.858516224264442740f, 0.512127776171554690f,
  -0.858909273947823900f,
  0.511468850437970520f, -0.859301818357008360f, 0.510809623820439040f,
  -0.859693857261072610f,
  0.510150096706766700f, -0.860085390429390140f, 0.509490269484936360f,
  -0.860476417631632070f,
  0.508830142543106990f, -0.860866938637767310f, 0.508169716269614710f,
  -0.861256953218062060f,
  0.507508991052970870f, -0.861646461143081300f, 0.506847967281863320f,
  -0.862035462183687210f,
  0.506186645345155450f, -0.862423956111040500f, 0.505525025631885510f,
  -0.862811942696600330f,
  0.504863108531267480f, -0.863199421712124160f, 0.504200894432690560f,
  -0.863586392929667990f,
  0.503538383725717580f, -0.863972856121586700f, 0.502875576800086880f,
  -0.864358811060534030f,
  0.502212474045710900f, -0.864744257519462380f, 0.501549075852675390f,
  -0.865129195271623690f,
  0.500885382611240940f, -0.865513624090568980f, 0.500221394711840680f,
  -0.865897543750148820f,
  0.499557112545081890f, -0.866280954024512990f, 0.498892536501744750f,
  -0.866663854688111020f,
  0.498227666972781870f, -0.867046245515692650f, 0.497562504349319090f,
  -0.867428126282306920f,
  0.496897049022654640f, -0.867809496763303210f, 0.496231301384258310f,
  -0.868190356734331310f,
  0.495565261825772490f, -0.868570705971340900f, 0.494898930739011310f,
  -0.868950544250582380f,
  0.494232308515959730f, -0.869329871348606730f, 0.493565395548774880f,
  -0.869708687042265560f,
  0.492898192229784090f, -0.870086991108711350f, 0.492230698951486080f,
  -0.870464783325397670f,
  0.491562916106550060f, -0.870842063470078860f, 0.490894844087815140f,
  -0.871218831320810900f,
  0.490226483288291100f, -0.871595086655951090f, 0.489557834101157550f,
  -0.871970829254157700f,
  0.488888896919763230f, -0.872346058894391540f, 0.488219672137626740f,
  -0.872720775355914300f,
  0.487550160148436050f, -0.873094978418290090f, 0.486880361346047400f,
  -0.873468667861384880f,
  0.486210276124486530f, -0.873841843465366750f, 0.485539904877947020f,
  -0.874214505010706300f,
  0.484869248000791120f, -0.874586652278176110f, 0.484198305887549140f,
  -0.874958285048851540f,
  0.483527078932918740f, -0.875329403104110780f, 0.482855567531765670f,
  -0.875700006225634600f,
  0.482183772079122830f, -0.876070094195406600f, 0.481511692970189920f,
  -0.876439666795713610f,
  0.480839330600333900f, -0.876808723809145760f, 0.480166685365088440f,
  -0.877177265018595940f,
  0.479493757660153010f, -0.877545290207261240f, 0.478820547881394050f,
  -0.877912799158641730f,
  0.478147056424843120f, -0.878279791656541460f, 0.477473283686698060f,
  -0.878646267485068130f,
  0.476799230063322250f, -0.879012226428633410f, 0.476124895951243630f,
  -0.879377668271953180f,
  0.475450281747155870f, -0.879742592800047410f, 0.474775387847917230f,
  -0.880106999798240360f,
  0.474100214650550020f, -0.880470889052160750f, 0.473424762552241530f,
  -0.880834260347742040f,
  0.472749031950342900f, -0.881197113471221980f, 0.472073023242368660f,
  -0.881559448209143780f,
  0.471396736825997810f, -0.881921264348354940f, 0.470720173099071710f,
  -0.882282561676008600f,
  0.470043332459595620f, -0.882643339979562790f, 0.469366215305737630f,
  -0.883003599046780720f,
  0.468688822035827960f, -0.883363338665731580f, 0.468011153048359830f,
  -0.883722558624789660f,
  0.467333208741988530f, -0.884081258712634990f, 0.466654989515530970f,
  -0.884439438718253700f,
  0.465976495767966130f, -0.884797098430937790f, 0.465297727898434650f,
  -0.885154237640285110f,
  0.464618686306237820f, -0.885510856136199950f, 0.463939371390838460f,
  -0.885866953708892790f,
  0.463259783551860260f, -0.886222530148880640f, 0.462579923189086810f,
  -0.886577585246987040f,
  0.461899790702462840f, -0.886932118794342080f, 0.461219386492092430f,
  -0.887286130582383150f,
  0.460538710958240010f, -0.887639620402853930f, 0.459857764501329650f,
  -0.887992588047805560f,
  0.459176547521944150f, -0.888345033309596240f, 0.458495060420826220f,
  -0.888696955980891710f,
  0.457813303598877290f, -0.889048355854664570f, 0.457131277457156980f,
  -0.889399232724195520f,
  0.456448982396883860f, -0.889749586383072890f, 0.455766418819434750f,
  -0.890099416625192210f,
  0.455083587126343840f, -0.890448723244757880f, 0.454400487719303750f,
  -0.890797506036281490f,
  0.453717121000163930f, -0.891145764794583180f, 0.453033487370931580f,
  -0.891493499314791380f,
  0.452349587233771000f, -0.891840709392342720f, 0.451665420991002540f,
  -0.892187394822982480f,
  0.450980989045103810f, -0.892533555402764690f, 0.450296291798708730f,
  -0.892879190928051680f,
  0.449611329654606600f, -0.893224301195515320f, 0.448926103015743260f,
  -0.893568886002136020f,
  0.448240612285220000f, -0.893912945145203250f, 0.447554857866293010f,
  -0.894256478422316040f,
  0.446868840162374330f, -0.894599485631382580f, 0.446182559577030120f,
  -0.894941966570620750f,
  0.445496016513981740f, -0.895283921038557580f, 0.444809211377105000f,
  -0.895625348834030000f,
  0.444122144570429260f, -0.895966249756185110f, 0.443434816498138430f,
  -0.896306623604479660f,
  0.442747227564570130f, -0.896646470178680150f, 0.442059378174214760f,
  -0.896985789278863970f,
  0.441371268731716620f, -0.897324580705418320f, 0.440682899641873020f,
  -0.897662844259040750f,
  0.439994271309633260f, -0.898000579740739880f, 0.439305384140100060f,
  -0.898337786951834190f,
  0.438616238538527710f, -0.898674465693953820f, 0.437926834910322860f,
  -0.899010615769039070f,
  0.437237173661044200f, -0.899346236979341460f, 0.436547255196401250f,
  -0.899681329127423930f,
  0.435857079922255470f, -0.900015892016160280f, 0.435166648244619370f,
  -0.900349925448735600f,
  0.434475960569655710f, -0.900683429228646860f, 0.433785017303678520f,
  -0.901016403159702330f,
  0.433093818853152010f, -0.901348847046022030f, 0.432402365624690140f,
  -0.901680760692037730f,
  0.431710658025057370f, -0.902012143902493070f, 0.431018696461167080f,
  -0.902342996482444200f,
  0.430326481340082610f, -0.902673318237258830f, 0.429634013069016500f,
  -0.903003108972617040f,
  0.428941292055329550f, -0.903332368494511820f, 0.428248318706531910f,
  -0.903661096609247980f,
  0.427555093430282200f, -0.903989293123443340f, 0.426861616634386490f,
  -0.904316957844028320f,
  0.426167888726799620f, -0.904644090578246240f, 0.425473910115623910f,
  -0.904970691133653250f,
  0.424779681209108810f, -0.905296759318118820f, 0.424085202415651670f,
  -0.905622294939825160f,
  0.423390474143796100f, -0.905947297807268460f, 0.422695496802232950f,
  -0.906271767729257660f,
  0.422000270799799790f, -0.906595704514915330f, 0.421304796545479700f,
  -0.906919107973678030f,
  0.420609074448402510f, -0.907241977915295930f, 0.419913104917843730f,
  -0.907564314149832520f,
  0.419216888363223960f, -0.907886116487666150f, 0.418520425194109700f,
  -0.908207384739488700f,
  0.417823715820212380f, -0.908528118716306120f, 0.417126760651387870f,
  -0.908848318229439120f,
  0.416429560097637320f, -0.909167983090522270f, 0.415732114569105420f,
  -0.909487113111505430f,
  0.415034424476081630f, -0.909805708104652220f, 0.414336490228999210f,
  -0.910123767882541570f,
  0.413638312238434560f, -0.910441292258067140f, 0.412939890915108020f,
  -0.910758281044437570f,
  0.412241226669883000f, -0.911074734055176250f, 0.411542319913765280f,
  -0.911390651104122320f,
  0.410843171057903910f, -0.911706032005429880f, 0.410143780513590350f,
  -0.912020876573568230f,
  0.409444148692257590f, -0.912335184623322750f, 0.408744276005481520f,
  -0.912648955969793900f,
  0.408044162864978740f, -0.912962190428398100f, 0.407343809682607970f,
  -0.913274887814867760f,
  0.406643216870369140f, -0.913587047945250810f, 0.405942384840402570f,
  -0.913898670635911680f,
  0.405241314004989860f, -0.914209755703530690f, 0.404540004776553110f,
  -0.914520302965104450f,
  0.403838457567654130f, -0.914830312237946090f, 0.403136672790995240f,
  -0.915139783339685260f,
  0.402434650859418540f, -0.915448716088267830f, 0.401732392185905010f,
  -0.915757110301956720f,
  0.401029897183575790f, -0.916064965799331610f, 0.400327166265690150f,
  -0.916372282399289140f,
  0.399624199845646790f, -0.916679059921042700f, 0.398920998336983020f,
  -0.916985298184122890f,
  0.398217562153373620f, -0.917290997008377910f, 0.397513891708632330f,
  -0.917596156213972950f,
  0.396809987416710420f, -0.917900775621390390f, 0.396105849691696320f,
  -0.918204855051430900f,
  0.395401478947816300f, -0.918508394325212250f, 0.394696875599433670f,
  -0.918811393264169940f,
  0.393992040061048100f, -0.919113851690057770f, 0.393286972747296570f,
  -0.919415769424946960f,
  0.392581674072951530f, -0.919717146291227360f, 0.391876144452922350f,
  -0.920017982111606570f,
  0.391170384302253980f, -0.920318276709110480f, 0.390464394036126650f,
  -0.920618029907083860f,
  0.389758174069856410f, -0.920917241529189520f, 0.389051724818894500f,
  -0.921215911399408730f,
  0.388345046698826300f, -0.921514039342041900f, 0.387638140125372680f,
  -0.921811625181708120f,
  0.386931005514388690f, -0.922108668743345070f, 0.386223643281862980f,
  -0.922405169852209880f,
  0.385516053843919020f, -0.922701128333878520f, 0.384808237616812930f,
  -0.922996544014246250f,
  0.384100195016935040f, -0.923291416719527640f, 0.383391926460808770f,
  -0.923585746276256560f,
  0.382683432365089840f, -0.923879532511286740f, 0.381974713146567220f,
  -0.924172775251791200f,
  0.381265769222162490f, -0.924465474325262600f, 0.380556601008928570f,
  -0.924757629559513910f,
  0.379847208924051110f, -0.925049240782677580f, 0.379137593384847430f,
  -0.925340307823206200f,
  0.378427754808765620f, -0.925630830509872720f, 0.377717693613385810f,
  -0.925920808671769960f,
  0.377007410216418310f, -0.926210242138311270f, 0.376296905035704790f,
  -0.926499130739230510f,
  0.375586178489217330f, -0.926787474304581750f, 0.374875230995057600f,
  -0.927075272664740100f,
  0.374164062971457990f, -0.927362525650401110f, 0.373452674836780410f,
  -0.927649233092581180f,
  0.372741067009515810f, -0.927935394822617890f, 0.372029239908284960f,
  -0.928221010672169440f,
  0.371317193951837600f, -0.928506080473215480f, 0.370604929559051670f,
  -0.928790604058057020f,
  0.369892447148934270f, -0.929074581259315750f, 0.369179747140620070f,
  -0.929358011909935500f,
  0.368466829953372320f, -0.929640895843181330f, 0.367753696006582090f,
  -0.929923232892639560f,
  0.367040345719767240f, -0.930205022892219070f, 0.366326779512573590f,
  -0.930486265676149780f,
  0.365612997804773960f, -0.930766961078983710f, 0.364899001016267380f,
  -0.931047108935595170f,
  0.364184789567079840f, -0.931326709081180430f, 0.363470363877363870f,
  -0.931605761351257830f,
  0.362755724367397230f, -0.931884265581668150f, 0.362040871457584350f,
  -0.932162221608574320f,
  0.361325805568454340f, -0.932439629268462360f, 0.360610527120662270f,
  -0.932716488398140250f,
  0.359895036534988280f, -0.932992798834738850f, 0.359179334232336560f,
  -0.933268560415712050f,
  0.358463420633736540f, -0.933543772978836170f, 0.357747296160342010f,
  -0.933818436362210960f,
  0.357030961233430030f, -0.934092550404258870f, 0.356314416274402360f,
  -0.934366114943725900f,
  0.355597661704783960f, -0.934639129819680780f, 0.354880697946222790f,
  -0.934911594871516090f,
  0.354163525420490510f, -0.935183509938947500f, 0.353446144549480870f,
  -0.935454874862014620f,
  0.352728555755210730f, -0.935725689481080370f, 0.352010759459819240f,
  -0.935995953636831300f,
  0.351292756085567150f, -0.936265667170278260f, 0.350574546054837570f,
  -0.936534829922755500f,
  0.349856129790135030f, -0.936803441735921560f, 0.349137507714085030f,
  -0.937071502451759190f,
  0.348418680249434510f, -0.937339011912574960f, 0.347699647819051490f,
  -0.937605969960999990f,
  0.346980410845923680f, -0.937872376439989890f, 0.346260969753160170f,
  -0.938138231192824360f,
  0.345541324963989150f, -0.938403534063108060f, 0.344821476901759290f,
  -0.938668284894770170f,
  0.344101425989938980f, -0.938932483532064490f, 0.343381172652115100f,
  -0.939196129819569900f,
  0.342660717311994380f, -0.939459223602189920f, 0.341940060393402300f,
  -0.939721764725153340f,
  0.341219202320282410f, -0.939983753034013940f, 0.340498143516697100f,
  -0.940245188374650880f,
  0.339776884406826960f, -0.940506070593268300f, 0.339055425414969640f,
  -0.940766399536396070f,
  0.338333766965541290f, -0.941026175050889260f, 0.337611909483074680f,
  -0.941285396983928660f,
  0.336889853392220050f, -0.941544065183020810f, 0.336167599117744690f,
  -0.941802179495997650f,
  0.335445147084531660f, -0.942059739771017310f, 0.334722497717581220f,
  -0.942316745856563780f,
  0.333999651442009490f, -0.942573197601446870f, 0.333276608683047980f,
  -0.942829094854802710f,
  0.332553369866044220f, -0.943084437466093490f, 0.331829935416461220f,
  -0.943339225285107720f,
  0.331106305759876430f, -0.943593458161960390f, 0.330382481321982950f,
  -0.943847135947092690f,
  0.329658462528587550f, -0.944100258491272660f, 0.328934249805612200f,
  -0.944352825645594750f,
  0.328209843579092660f, -0.944604837261480260f, 0.327485244275178060f,
  -0.944856293190677210f,
  0.326760452320131790f, -0.945107193285260610f, 0.326035468140330350f,
  -0.945357537397632290f,
  0.325310292162262980f, -0.945607325380521280f, 0.324584924812532150f,
  -0.945856557086983910f,
  0.323859366517852960f, -0.946105232370403340f, 0.323133617705052330f,
  -0.946353351084490590f,
  0.322407678801070020f, -0.946600913083283530f, 0.321681550232956640f,
  -0.946847918221148000f,
  0.320955232427875210f, -0.947094366352777220f, 0.320228725813100020f,
  -0.947340257333191940f,
  0.319502030816015750f, -0.947585591017741090f, 0.318775147864118480f,
  -0.947830367262101010f,
  0.318048077385015060f, -0.948074585922276230f, 0.317320819806421790f,
  -0.948318246854599090f,
  0.316593375556165850f, -0.948561349915730270f, 0.315865745062184070f,
  -0.948803894962658380f,
  0.315137928752522440f, -0.949045881852700560f, 0.314409927055336820f,
  -0.949287310443502010f,
  0.313681740398891570f, -0.949528180593036670f, 0.312953369211560200f,
  -0.949768492159606680f,
  0.312224813921825050f, -0.950008245001843000f, 0.311496074958275970f,
  -0.950247438978705230f,
  0.310767152749611470f, -0.950486073949481700f, 0.310038047724638000f,
  -0.950724149773789610f,
  0.309308760312268780f, -0.950961666311575080f, 0.308579290941525030f,
  -0.951198623423113230f,
  0.307849640041534980f, -0.951435020969008340f, 0.307119808041533100f,
  -0.951670858810193860f,
  0.306389795370861080f, -0.951906136807932230f, 0.305659602458966230f,
  -0.952140854823815830f,
  0.304929229735402430f, -0.952375012719765880f, 0.304198677629829270f,
  -0.952608610358033240f,
  0.303467946572011370f, -0.952841647601198720f, 0.302737036991819140f,
  -0.953074124312172200f,
  0.302005949319228200f, -0.953306040354193750f, 0.301274683984318000f,
  -0.953537395590833280f,
  0.300543241417273400f, -0.953768189885990330f, 0.299811622048383460f,
  -0.953998423103894490f,
  0.299079826308040480f, -0.954228095109105670f, 0.298347854626741570f,
  -0.954457205766513490f,
  0.297615707435086310f, -0.954685754941338340f, 0.296883385163778270f,
  -0.954913742499130520f,
  0.296150888243623960f, -0.955141168305770670f, 0.295418217105532070f,
  -0.955368032227470240f,
  0.294685372180514330f, -0.955594334130771110f, 0.293952353899684770f,
  -0.955820073882545420f,
  0.293219162694258680f, -0.956045251349996410f, 0.292485798995553830f,
  -0.956269866400658140f,
  0.291752263234989370f, -0.956493918902394990f, 0.291018555844085090f,
  -0.956717408723403050f,
  0.290284677254462330f, -0.956940335732208940f, 0.289550627897843140f,
  -0.957162699797670100f,
  0.288816408206049480f, -0.957384500788975860f, 0.288082018611004300f,
  -0.957605738575646240f,
  0.287347459544729570f, -0.957826413027532910f, 0.286612731439347790f,
  -0.958046524014818600f,
  0.285877834727080730f, -0.958266071408017670f, 0.285142769840248720f,
  -0.958485055077976100f,
  0.284407537211271820f, -0.958703474895871600f, 0.283672137272668550f,
  -0.958921330733213060f,
  0.282936570457055390f, -0.959138622461841890f, 0.282200837197147500f,
  -0.959355349953930790f,
  0.281464937925758050f, -0.959571513081984520f, 0.280728873075797190f,
  -0.959787111718839900f,
  0.279992643080273380f, -0.960002145737665850f, 0.279256248372291240f,
  -0.960216615011963430f,
  0.278519689385053060f, -0.960430519415565790f, 0.277782966551857800f,
  -0.960643858822638470f,
  0.277046080306099950f, -0.960856633107679660f, 0.276309031081271030f,
  -0.961068842145519350f,
  0.275571819310958250f, -0.961280485811320640f, 0.274834445428843940f,
  -0.961491563980579000f,
  0.274096909868706330f, -0.961702076529122540f, 0.273359213064418790f,
  -0.961912023333112100f,
  0.272621355449948980f, -0.962121404269041580f, 0.271883337459359890f,
  -0.962330219213737400f,
  0.271145159526808070f, -0.962538468044359160f, 0.270406822086544820f,
  -0.962746150638399410f,
  0.269668325572915200f, -0.962953266873683880f, 0.268929670420357310f,
  -0.963159816628371360f,
  0.268190857063403180f, -0.963365799780954050f, 0.267451885936677740f,
  -0.963571216210257210f,
  0.266712757474898420f, -0.963776065795439840f, 0.265973472112875530f,
  -0.963980348415994110f,
  0.265234030285511900f, -0.964184063951745720f, 0.264494432427801630f,
  -0.964387212282854290f,
  0.263754678974831510f, -0.964589793289812650f, 0.263014770361779060f,
  -0.964791806853447900f,
  0.262274707023913590f, -0.964993252854920320f, 0.261534489396595630f,
  -0.965194131175724720f,
  0.260794117915275570f, -0.965394441697689400f, 0.260053593015495130f,
  -0.965594184302976830f,
  0.259312915132886350f, -0.965793358874083570f, 0.258572084703170390f,
  -0.965991965293840570f,
  0.257831102162158930f, -0.966190003445412620f, 0.257089967945753230f,
  -0.966387473212298790f,
  0.256348682489942910f, -0.966584374478333120f, 0.255607246230807550f,
  -0.966780707127683270f,
  0.254865659604514630f, -0.966976471044852070f, 0.254123923047320620f,
  -0.967171666114676640f,
  0.253382036995570270f, -0.967366292222328510f, 0.252640001885695580f,
  -0.967560349253314360f,
  0.251897818154216910f, -0.967753837093475510f, 0.251155486237742030f,
  -0.967946755628987800f,
  0.250413006572965280f, -0.968139104746362330f, 0.249670379596668520f,
  -0.968330884332445300f,
  0.248927605745720260f, -0.968522094274417270f, 0.248184685457074780f,
  -0.968712734459794780f,
  0.247441619167773440f, -0.968902804776428870f, 0.246698407314942500f,
  -0.969092305112506100f,
  0.245955050335794590f, -0.969281235356548530f, 0.245211548667627680f,
  -0.969469595397412950f,
  0.244467902747824210f, -0.969657385124292450f, 0.243724113013852130f,
  -0.969844604426714830f,
  0.242980179903263980f, -0.970031253194543970f, 0.242236103853696070f,
  -0.970217331317979160f,
  0.241491885302869300f, -0.970402838687555500f, 0.240747524688588540f,
  -0.970587775194143630f,
  0.240003022448741500f, -0.970772140728950350f, 0.239258379021300120f,
  -0.970955935183517970f,
  0.238513594844318500f, -0.971139158449725090f, 0.237768670355934210f,
  -0.971321810419786160f,
  0.237023605994367340f, -0.971503890986251780f, 0.236278402197919620f,
  -0.971685400042008540f,
  0.235533059404975460f, -0.971866337480279400f, 0.234787578054001080f,
  -0.972046703194623500f,
  0.234041958583543460f, -0.972226497078936270f, 0.233296201432231560f,
  -0.972405719027449770f,
  0.232550307038775330f, -0.972584368934732210f, 0.231804275841964780f,
  -0.972762446695688570f,
  0.231058108280671280f, -0.972939952205560070f, 0.230311804793845530f,
  -0.973116885359925130f,
  0.229565365820518870f, -0.973293246054698250f, 0.228818791799802360f,
  -0.973469034186130950f,
  0.228072083170885790f, -0.973644249650811870f, 0.227325240373038830f,
  -0.973818892345666100f,
  0.226578263845610110f, -0.973992962167955830f, 0.225831154028026200f,
  -0.974166459015280320f,
  0.225083911359792780f, -0.974339382785575860f, 0.224336536280493690f,
  -0.974511733377115720f,
  0.223589029229790020f, -0.974683510688510670f, 0.222841390647421280f,
  -0.974854714618708430f,
  0.222093620973203590f, -0.975025345066994120f, 0.221345720647030810f,
  -0.975195401932990370f,
  0.220597690108873650f, -0.975364885116656870f, 0.219849529798778750f,
  -0.975533794518291360f,
  0.219101240156869770f, -0.975702130038528570f, 0.218352821623346430f,
  -0.975869891578341030f,
  0.217604274638483670f, -0.976037079039039020f, 0.216855599642632570f,
  -0.976203692322270560f,
  0.216106797076219600f, -0.976369731330021140f, 0.215357867379745550f,
  -0.976535195964614470f,
  0.214608810993786920f, -0.976700086128711840f, 0.213859628358993830f,
  -0.976864401725312640f,
  0.213110319916091360f, -0.977028142657754390f, 0.212360886105878580f,
  -0.977191308829712280f,
  0.211611327369227610f, -0.977353900145199960f, 0.210861644147084830f,
  -0.977515916508569280f,
  0.210111836880469720f, -0.977677357824509930f, 0.209361906010474190f,
  -0.977838223998050430f,
  0.208611851978263460f, -0.977998514934557140f, 0.207861675225075150f,
  -0.978158230539735050f,
  0.207111376192218560f, -0.978317370719627650f, 0.206360955321075680f,
  -0.978475935380616830f,
  0.205610413053099320f, -0.978633924429423100f, 0.204859749829814420f,
  -0.978791337773105670f,
  0.204108966092817010f, -0.978948175319062200f, 0.203358062283773370f,
  -0.979104436975029250f,
  0.202607038844421110f, -0.979260122649082020f, 0.201855896216568160f,
  -0.979415232249634780f,
  0.201104634842091960f, -0.979569765685440520f, 0.200353255162940420f,
  -0.979723722865591170f,
  0.199601757621131050f, -0.979877103699517640f, 0.198850142658750120f,
  -0.980029908096989980f,
  0.198098410717953730f, -0.980182135968117320f, 0.197346562240966000f,
  -0.980333787223347960f,
  0.196594597670080220f, -0.980484861773469380f, 0.195842517447657990f,
  -0.980635359529608120f,
  0.195090322016128330f, -0.980785280403230430f, 0.194338011817988600f,
  -0.980934624306141640f,
  0.193585587295803750f, -0.981083391150486590f, 0.192833048892205290f,
  -0.981231580848749730f,
  0.192080397049892380f, -0.981379193313754560f, 0.191327632211630990f,
  -0.981526228458664660f,
  0.190574754820252800f, -0.981672686196983110f, 0.189821765318656580f,
  -0.981818566442552500f,
  0.189068664149806280f, -0.981963869109555240f, 0.188315451756732120f,
  -0.982108594112513610f,
  0.187562128582529740f, -0.982252741366289370f, 0.186808695070359330f,
  -0.982396310786084690f,
  0.186055151663446630f, -0.982539302287441240f, 0.185301498805082040f,
  -0.982681715786240860f,
  0.184547736938619640f, -0.982823551198705240f, 0.183793866507478390f,
  -0.982964808441396440f,
  0.183039887955141060f, -0.983105487431216290f, 0.182285801725153320f,
  -0.983245588085407070f,
  0.181531608261125130f, -0.983385110321551180f, 0.180777308006728670f,
  -0.983524054057571260f,
  0.180022901405699510f, -0.983662419211730250f, 0.179268388901835880f,
  -0.983800205702631490f,
  0.178513770938997590f, -0.983937413449218920f, 0.177759047961107140f,
  -0.984074042370776450f,
  0.177004220412148860f, -0.984210092386929030f, 0.176249288736167940f,
  -0.984345563417641900f,
  0.175494253377271400f, -0.984480455383220930f, 0.174739114779627310f,
  -0.984614768204312600f,
  0.173983873387463850f, -0.984748501801904210f, 0.173228529645070490f,
  -0.984881656097323700f,
  0.172473083996796030f, -0.985014231012239840f, 0.171717536887049970f,
  -0.985146226468662230f,
  0.170961888760301360f, -0.985277642388941220f, 0.170206140061078120f,
  -0.985408478695768420f,
  0.169450291233967930f, -0.985538735312176060f, 0.168694342723617440f,
  -0.985668412161537550f,
  0.167938294974731230f, -0.985797509167567370f, 0.167182148432072880f,
  -0.985926026254321130f,
  0.166425903540464220f, -0.986053963346195440f, 0.165669560744784140f,
  -0.986181320367928270f,
  0.164913120489970090f, -0.986308097244598670f, 0.164156583221015890f,
  -0.986434293901627070f,
  0.163399949382973230f, -0.986559910264775410f, 0.162643219420950450f,
  -0.986684946260146690f,
  0.161886393780111910f, -0.986809401814185420f, 0.161129472905678780f,
  -0.986933276853677710f,
  0.160372457242928400f, -0.987056571305750970f, 0.159615347237193090f,
  -0.987179285097874340f,
  0.158858143333861390f, -0.987301418157858430f, 0.158100845978377090f,
  -0.987422970413855410f,
  0.157343455616238280f, -0.987543941794359230f, 0.156585972692998590f,
  -0.987664332228205710f,
  0.155828397654265320f, -0.987784141644572180f, 0.155070730945700510f,
  -0.987903369972977790f,
  0.154312973013020240f, -0.988022017143283530f, 0.153555124301993500f,
  -0.988140083085692570f,
  0.152797185258443410f, -0.988257567730749460f, 0.152039156328246160f,
  -0.988374471009341280f,
  0.151281037957330250f, -0.988490792852696590f, 0.150522830591677370f,
  -0.988606533192386450f,
  0.149764534677321620f, -0.988721691960323780f, 0.149006150660348470f,
  -0.988836269088763540f,
  0.148247678986896200f, -0.988950264510302990f, 0.147489120103153680f,
  -0.989063678157881540f,
  0.146730474455361750f, -0.989176509964781010f, 0.145971742489812370f,
  -0.989288759864625170f,
  0.145212924652847520f, -0.989400427791380380f, 0.144454021390860440f,
  -0.989511513679355190f,
  0.143695033150294580f, -0.989622017463200780f, 0.142935960377642700f,
  -0.989731939077910570f,
  0.142176803519448000f, -0.989841278458820530f, 0.141417563022303130f,
  -0.989950035541608990f,
  0.140658239332849240f, -0.990058210262297120f, 0.139898832897777380f,
  -0.990165802557248400f,
  0.139139344163826280f, -0.990272812363169110f, 0.138379773577783890f,
  -0.990379239617108160f,
  0.137620121586486180f, -0.990485084256456980f, 0.136860388636816430f,
  -0.990590346218950150f,
  0.136100575175706200f, -0.990695025442664630f, 0.135340681650134330f,
  -0.990799121866020370f,
  0.134580708507126220f, -0.990902635427780010f, 0.133820656193754690f,
  -0.991005566067049370f,
  0.133060525157139180f, -0.991107913723276780f, 0.132300315844444680f,
  -0.991209678336254060f,
  0.131540028702883280f, -0.991310859846115440f, 0.130779664179711790f,
  -0.991411458193338540f,
  0.130019222722233350f, -0.991511473318743900f, 0.129258704777796270f,
  -0.991610905163495370f,
  0.128498110793793220f, -0.991709753669099530f, 0.127737441217662280f,
  -0.991808018777406430f,
  0.126976696496885980f, -0.991905700430609330f, 0.126215877078990400f,
  -0.992002798571244520f,
  0.125454983411546210f, -0.992099313142191800f, 0.124694015942167770f,
  -0.992195244086673920f,
  0.123932975118512200f, -0.992290591348257370f, 0.123171861388280650f,
  -0.992385354870851670f,
  0.122410675199216280f, -0.992479534598709970f, 0.121649416999105540f,
  -0.992573130476428810f,
  0.120888087235777220f, -0.992666142448948020f, 0.120126686357101580f,
  -0.992758570461551140f,
  0.119365214810991350f, -0.992850414459865100f, 0.118603673045400840f,
  -0.992941674389860470f,
  0.117842061508325020f, -0.993032350197851410f, 0.117080380647800550f,
  -0.993122441830495580f,
  0.116318630911904880f, -0.993211949234794500f, 0.115556812748755290f,
  -0.993300872358093280f,
  0.114794926606510250f, -0.993389211148080650f, 0.114032972933367300f,
  -0.993476965552789190f,
  0.113270952177564360f, -0.993564135520595300f, 0.112508864787378830f,
  -0.993650721000219120f,
  0.111746711211126660f, -0.993736721940724600f, 0.110984491897163380f,
  -0.993822138291519660f,
  0.110222207293883180f, -0.993906970002356060f, 0.109459857849718030f,
  -0.993991217023329380f,
  0.108697444013138670f, -0.994074879304879370f, 0.107934966232653760f,
  -0.994157956797789730f,
  0.107172424956808870f, -0.994240449453187900f, 0.106409820634187840f,
  -0.994322357222545810f,
  0.105647153713410700f, -0.994403680057679100f, 0.104884424643134970f,
  -0.994484417910747600f,
  0.104121633872054730f, -0.994564570734255420f, 0.103358781848899700f,
  -0.994644138481050710f,
  0.102595869022436280f, -0.994723121104325700f, 0.101832895841466670f,
  -0.994801518557617110f,
  0.101069862754827880f, -0.994879330794805620f, 0.100306770211392820f,
  -0.994956557770116380f,
  0.099543618660069444f, -0.995033199438118630f, 0.098780408549799664f,
  -0.995109255753726110f,
  0.098017140329560770f, -0.995184726672196820f, 0.097253814448363354f,
  -0.995259612149133390f,
  0.096490431355252607f, -0.995333912140482280f, 0.095726991499307315f,
  -0.995407626602534900f,
  0.094963495329639061f, -0.995480755491926940f, 0.094199943295393190f,
  -0.995553298765638470f,
  0.093436335845747912f, -0.995625256380994310f, 0.092672673429913366f,
  -0.995696628295663520f,
  0.091908956497132696f, -0.995767414467659820f, 0.091145185496681130f,
  -0.995837614855341610f,
  0.090381360877865011f, -0.995907229417411720f, 0.089617483090022917f,
  -0.995976258112917790f,
  0.088853552582524684f, -0.996044700901251970f, 0.088089569804770507f,
  -0.996112557742151130f,
  0.087325535206192226f, -0.996179828595696870f, 0.086561449236251239f,
  -0.996246513422315520f,
  0.085797312344439880f, -0.996312612182778000f, 0.085033124980280414f,
  -0.996378124838200210f,
  0.084268887593324127f, -0.996443051350042630f, 0.083504600633152404f,
  -0.996507391680110820f,
  0.082740264549375803f, -0.996571145790554840f, 0.081975879791633108f,
  -0.996634313643869900f,
  0.081211446809592386f, -0.996696895202896060f, 0.080446966052950097f,
  -0.996758890430818000f,
  0.079682437971430126f, -0.996820299291165670f, 0.078917863014785095f,
  -0.996881121747813850f,
  0.078153241632794315f, -0.996941357764982160f, 0.077388574275265049f,
  -0.997001007307235290f,
  0.076623861392031617f, -0.997060070339482960f, 0.075859103432954503f,
  -0.997118546826979980f,
  0.075094300847921291f, -0.997176436735326190f, 0.074329454086845867f,
  -0.997233740030466160f,
  0.073564563599667454f, -0.997290456678690210f, 0.072799629836351618f,
  -0.997346586646633230f,
  0.072034653246889416f, -0.997402129901275300f, 0.071269634281296415f,
  -0.997457086409941910f,
  0.070504573389614009f, -0.997511456140303450f, 0.069739471021907376f,
  -0.997565239060375750f,
  0.068974327628266732f, -0.997618435138519550f, 0.068209143658806454f,
  -0.997671044343441000f,
  0.067443919563664106f, -0.997723066644191640f, 0.066678655793001543f,
  -0.997774502010167820f,
  0.065913352797003930f, -0.997825350411111640f, 0.065148011025878860f,
  -0.997875611817110150f,
  0.064382630929857410f, -0.997925286198596000f, 0.063617212959193190f,
  -0.997974373526346990f,
  0.062851757564161420f, -0.998022873771486240f, 0.062086265195060247f,
  -0.998070786905482340f,
  0.061320736302208648f, -0.998118112900149180f, 0.060555171335947781f,
  -0.998164851727646240f,
  0.059789570746640007f, -0.998211003360478190f, 0.059023934984667986f,
  -0.998256567771495180f,
  0.058258264500435732f, -0.998301544933892890f, 0.057492559744367684f,
  -0.998345934821212370f,
  0.056726821166907783f, -0.998389737407340160f, 0.055961049218520520f,
  -0.998432952666508440f,
  0.055195244349690031f, -0.998475580573294770f, 0.054429407010919147f,
  -0.998517621102622210f,
  0.053663537652730679f, -0.998559074229759310f, 0.052897636725665401f,
  -0.998599939930320370f,
  0.052131704680283317f, -0.998640218180265270f, 0.051365741967162731f,
  -0.998679908955899090f,
  0.050599749036899337f, -0.998719012233872940f, 0.049833726340107257f,
  -0.998757527991183340f,
  0.049067674327418126f, -0.998795456205172410f, 0.048301593449480172f,
  -0.998832796853527990f,
  0.047535484156959261f, -0.998869549914283560f, 0.046769346900537960f,
  -0.998905715365818290f,
  0.046003182130914644f, -0.998941293186856870f, 0.045236990298804750f,
  -0.998976283356469820f,
  0.044470771854938744f, -0.999010685854073380f, 0.043704527250063421f,
  -0.999044500659429290f,
  0.042938256934940959f, -0.999077727752645360f, 0.042171961360348002f,
  -0.999110367114174890f,
  0.041405640977076712f, -0.999142418724816910f, 0.040639296235933854f,
  -0.999173882565716380f,
  0.039872927587739845f, -0.999204758618363890f, 0.039106535483329839f,
  -0.999235046864595850f,
  0.038340120373552791f, -0.999264747286594420f, 0.037573682709270514f,
  -0.999293859866887790f,
  0.036807222941358991f, -0.999322384588349540f, 0.036040741520706299f,
  -0.999350321434199440f,
  0.035274238898213947f, -0.999377670388002850f, 0.034507715524795889f,
  -0.999404431433671300f,
  0.033741171851377642f, -0.999430604555461730f, 0.032974608328897315f,
  -0.999456189737977340f,
  0.032208025408304704f, -0.999481186966166950f, 0.031441423540560343f,
  -0.999505596225325310f,
  0.030674803176636581f, -0.999529417501093140f, 0.029908164767516655f,
  -0.999552650779456990f,
  0.029141508764193740f, -0.999575296046749220f, 0.028374835617672258f,
  -0.999597353289648380f,
  0.027608145778965820f, -0.999618822495178640f, 0.026841439699098527f,
  -0.999639703650710200f,
  0.026074717829104040f, -0.999659996743959220f, 0.025307980620024630f,
  -0.999679701762987930f,
  0.024541228522912264f, -0.999698818696204250f, 0.023774461988827676f,
  -0.999717347532362190f,
  0.023007681468839410f, -0.999735288260561680f, 0.022240887414024919f,
  -0.999752640870248840f,
  0.021474080275469605f, -0.999769405351215280f, 0.020707260504265912f,
  -0.999785581693599210f,
  0.019940428551514598f, -0.999801169887884260f, 0.019173584868322699f,
  -0.999816169924900410f,
  0.018406729905804820f, -0.999830581795823400f, 0.017639864115082195f,
  -0.999844405492175240f,
  0.016872987947281773f, -0.999857641005823860f, 0.016106101853537263f,
  -0.999870288328982950f,
  0.015339206284988220f, -0.999882347454212560f, 0.014572301692779104f,
  -0.999893818374418490f,
  0.013805388528060349f, -0.999904701082852900f, 0.013038467241987433f,
  -0.999914995573113470f,
  0.012271538285719944f, -0.999924701839144500f, 0.011504602110422875f,
  -0.999933819875236000f,
  0.010737659167264572f, -0.999942349676023910f, 0.009970709907418029f,
  -0.999950291236490480f,
  0.009203754782059960f, -0.999957644551963900f, 0.008436794242369860f,
  -0.999964409618118280f,
  0.007669828739531077f, -0.999970586430974140f, 0.006902858724729877f,
  -0.999976174986897610f,
  0.006135884649154515f, -0.999981175282601110f, 0.005368906963996303f,
  -0.999985587315143200f,
  0.004601926120448672f, -0.999989411081928400f, 0.003834942569706248f,
  -0.999992646580707190f,
  0.003067956762966138f, -0.999995293809576190f, 0.002300969151425887f,
  -0.999997352766978210f,
  0.001533980186284766f, -0.999998823451701880f, 0.000766990318742846f,
  -0.999999705862882230f
};

/**   
* \par   
* cosFactor tables are generated using the formula : <pre>cos_factors[n] = 2 * cos((2n+1)*pi/(4*N))</pre>   
* \par   
* C command to generate the table   
* \par   
* <pre> for(i = 0; i< N; i++)   
* {   
*    cos_factors[i]= 2 * cos((2*i+1)*c/2);   
* } </pre>   
* \par   
* where <code>N</code> is the number of factors to generate and <code>c</code> is <code>pi/(2*N)</code>   
*/
static const float32_t cos_factors_128[128] = {
  0.999981175282601110f, 0.999830581795823400f, 0.999529417501093140f,
  0.999077727752645360f,
  0.998475580573294770f, 0.997723066644191640f, 0.996820299291165670f,
  0.995767414467659820f,
  0.994564570734255420f, 0.993211949234794500f, 0.991709753669099530f,
  0.990058210262297120f,
  0.988257567730749460f, 0.986308097244598670f, 0.984210092386929030f,
  0.981963869109555240f,
  0.979569765685440520f, 0.977028142657754390f, 0.974339382785575860f,
  0.971503890986251780f,
  0.968522094274417380f, 0.965394441697689400f, 0.962121404269041580f,
  0.958703474895871600f,
  0.955141168305770780f, 0.951435020969008340f, 0.947585591017741090f,
  0.943593458161960390f,
  0.939459223602189920f, 0.935183509938947610f, 0.930766961078983710f,
  0.926210242138311380f,
  0.921514039342042010f, 0.916679059921042700f, 0.911706032005429880f,
  0.906595704514915330f,
  0.901348847046022030f, 0.895966249756185220f, 0.890448723244757880f,
  0.884797098430937790f,
  0.879012226428633530f, 0.873094978418290090f, 0.867046245515692650f,
  0.860866938637767310f,
  0.854557988365400530f, 0.848120344803297230f, 0.841554977436898440f,
  0.834862874986380010f,
  0.828045045257755800f, 0.821102514991104650f, 0.814036329705948410f,
  0.806847553543799330f,
  0.799537269107905010f, 0.792106577300212390f, 0.784556597155575240f,
  0.776888465673232440f,
  0.769103337645579700f, 0.761202385484261780f, 0.753186799043612520f,
  0.745057785441466060f,
  0.736816568877369900f, 0.728464390448225200f, 0.720002507961381650f,
  0.711432195745216430f,
  0.702754744457225300f, 0.693971460889654000f, 0.685083667772700360f,
  0.676092703575316030f,
  0.666999922303637470f, 0.657806693297078640f, 0.648514401022112550f,
  0.639124444863775730f,
  0.629638238914927100f, 0.620057211763289210f, 0.610382806276309480f,
  0.600616479383868970f,
  0.590759701858874280f, 0.580813958095764530f, 0.570780745886967370f,
  0.560661576197336030f,
  0.550457972936604810f, 0.540171472729892970f, 0.529803624686294830f,
  0.519355990165589530f,
  0.508830142543106990f, 0.498227666972781870f, 0.487550160148436050f,
  0.476799230063322250f,
  0.465976495767966130f, 0.455083587126343840f, 0.444122144570429260f,
  0.433093818853152010f,
  0.422000270799799790f, 0.410843171057903910f, 0.399624199845646790f,
  0.388345046698826300f,
  0.377007410216418310f, 0.365612997804773960f, 0.354163525420490510f,
  0.342660717311994380f,
  0.331106305759876430f, 0.319502030816015750f, 0.307849640041534980f,
  0.296150888243623960f,
  0.284407537211271820f, 0.272621355449948980f, 0.260794117915275570f,
  0.248927605745720260f,
  0.237023605994367340f, 0.225083911359792780f, 0.213110319916091360f,
  0.201104634842091960f,
  0.189068664149806280f, 0.177004220412148860f, 0.164913120489970090f,
  0.152797185258443410f,
  0.140658239332849240f, 0.128498110793793220f, 0.116318630911904880f,
  0.104121633872054730f,
  0.091908956497132696f, 0.079682437971430126f, 0.067443919563664106f,
  0.055195244349690031f,
  0.042938256934940959f, 0.030674803176636581f, 0.018406729905804820f,
  0.006135884649154515f
};

static const float32_t cos_factors_512[512] = {
  0.999998823451701880f, 0.999989411081928400f, 0.999970586430974140f,
  0.999942349676023910f,
  0.999904701082852900f, 0.999857641005823860f, 0.999801169887884260f,
  0.999735288260561680f,
  0.999659996743959220f, 0.999575296046749220f, 0.999481186966166950f,
  0.999377670388002850f,
  0.999264747286594420f, 0.999142418724816910f, 0.999010685854073380f,
  0.998869549914283560f,
  0.998719012233872940f, 0.998559074229759310f, 0.998389737407340160f,
  0.998211003360478190f,
  0.998022873771486240f, 0.997825350411111640f, 0.997618435138519550f,
  0.997402129901275300f,
  0.997176436735326190f, 0.996941357764982160f, 0.996696895202896060f,
  0.996443051350042630f,
  0.996179828595696980f, 0.995907229417411720f, 0.995625256380994310f,
  0.995333912140482280f,
  0.995033199438118630f, 0.994723121104325700f, 0.994403680057679100f,
  0.994074879304879370f,
  0.993736721940724600f, 0.993389211148080650f, 0.993032350197851410f,
  0.992666142448948020f,
  0.992290591348257370f, 0.991905700430609330f, 0.991511473318743900f,
  0.991107913723276890f,
  0.990695025442664630f, 0.990272812363169110f, 0.989841278458820530f,
  0.989400427791380380f,
  0.988950264510302990f, 0.988490792852696590f, 0.988022017143283530f,
  0.987543941794359230f,
  0.987056571305750970f, 0.986559910264775410f, 0.986053963346195440f,
  0.985538735312176060f,
  0.985014231012239840f, 0.984480455383220930f, 0.983937413449218920f,
  0.983385110321551180f,
  0.982823551198705240f, 0.982252741366289370f, 0.981672686196983110f,
  0.981083391150486710f,
  0.980484861773469380f, 0.979877103699517640f, 0.979260122649082020f,
  0.978633924429423210f,
  0.977998514934557140f, 0.977353900145199960f, 0.976700086128711840f,
  0.976037079039039020f,
  0.975364885116656980f, 0.974683510688510670f, 0.973992962167955830f,
  0.973293246054698250f,
  0.972584368934732210f, 0.971866337480279400f, 0.971139158449725090f,
  0.970402838687555500f,
  0.969657385124292450f, 0.968902804776428870f, 0.968139104746362440f,
  0.967366292222328510f,
  0.966584374478333120f, 0.965793358874083680f, 0.964993252854920320f,
  0.964184063951745830f,
  0.963365799780954050f, 0.962538468044359160f, 0.961702076529122540f,
  0.960856633107679660f,
  0.960002145737665960f, 0.959138622461841890f, 0.958266071408017670f,
  0.957384500788975860f,
  0.956493918902395100f, 0.955594334130771110f, 0.954685754941338340f,
  0.953768189885990330f,
  0.952841647601198720f, 0.951906136807932350f, 0.950961666311575080f,
  0.950008245001843000f,
  0.949045881852700560f, 0.948074585922276230f, 0.947094366352777220f,
  0.946105232370403450f,
  0.945107193285260610f, 0.944100258491272660f, 0.943084437466093490f,
  0.942059739771017310f,
  0.941026175050889260f, 0.939983753034014050f, 0.938932483532064600f,
  0.937872376439989890f,
  0.936803441735921560f, 0.935725689481080370f, 0.934639129819680780f,
  0.933543772978836170f,
  0.932439629268462360f, 0.931326709081180430f, 0.930205022892219070f,
  0.929074581259315860f,
  0.927935394822617890f, 0.926787474304581750f, 0.925630830509872720f,
  0.924465474325262600f,
  0.923291416719527640f, 0.922108668743345180f, 0.920917241529189520f,
  0.919717146291227360f,
  0.918508394325212250f, 0.917290997008377910f, 0.916064965799331720f,
  0.914830312237946200f,
  0.913587047945250810f, 0.912335184623322750f, 0.911074734055176360f,
  0.909805708104652220f,
  0.908528118716306120f, 0.907241977915295820f, 0.905947297807268460f,
  0.904644090578246240f,
  0.903332368494511820f, 0.902012143902493180f, 0.900683429228646970f,
  0.899346236979341570f,
  0.898000579740739880f, 0.896646470178680150f, 0.895283921038557580f,
  0.893912945145203250f,
  0.892533555402764580f, 0.891145764794583180f, 0.889749586383072780f,
  0.888345033309596350f,
  0.886932118794342190f, 0.885510856136199950f, 0.884081258712634990f,
  0.882643339979562790f,
  0.881197113471222090f, 0.879742592800047410f, 0.878279791656541580f,
  0.876808723809145650f,
  0.875329403104110890f, 0.873841843465366860f, 0.872346058894391540f,
  0.870842063470078980f,
  0.869329871348606840f, 0.867809496763303320f, 0.866280954024512990f,
  0.864744257519462380f,
  0.863199421712124160f, 0.861646461143081300f, 0.860085390429390140f,
  0.858516224264442740f,
  0.856938977417828760f, 0.855353664735196030f, 0.853760301138111410f,
  0.852158901623919830f,
  0.850549481265603480f, 0.848932055211639610f, 0.847306638685858320f,
  0.845673246987299070f,
  0.844031895490066410f, 0.842382599643185850f, 0.840725374970458070f,
  0.839060237070312740f,
  0.837387201615661940f, 0.835706284353752600f, 0.834017501106018130f,
  0.832320867767929680f,
  0.830616400308846310f, 0.828904114771864870f, 0.827184027273669130f,
  0.825456154004377550f,
  0.823720511227391430f, 0.821977115279241550f, 0.820225982569434690f,
  0.818467129580298660f,
  0.816700572866827850f, 0.814926329056526620f, 0.813144414849253590f,
  0.811354847017063730f,
  0.809557642404051260f, 0.807752817926190360f, 0.805940390571176280f,
  0.804120377398265810f,
  0.802292795538115720f, 0.800457662192622820f, 0.798614994634760820f,
  0.796764810208418830f,
  0.794907126328237010f, 0.793041960479443640f, 0.791169330217690200f,
  0.789289253168885650f,
  0.787401747029031430f, 0.785506829564053930f, 0.783604518609638200f,
  0.781694832071059390f,
  0.779777787923014550f, 0.777853404209453150f, 0.775921699043407690f,
  0.773982690606822900f,
  0.772036397150384520f, 0.770082836993347900f, 0.768122028523365420f,
  0.766153990196312920f,
  0.764178740536116670f, 0.762196298134578900f, 0.760206681651202420f,
  0.758209909813015280f,
  0.756206001414394540f, 0.754194975316889170f, 0.752176850449042810f,
  0.750151645806215070f,
  0.748119380450403600f, 0.746080073510063780f, 0.744033744179929290f,
  0.741980411720831070f,
  0.739920095459516200f, 0.737852814788465980f, 0.735778589165713590f,
  0.733697438114660370f,
  0.731609381223892630f, 0.729514438146997010f, 0.727412628602375770f,
  0.725303972373060770f,
  0.723188489306527460f, 0.721066199314508110f, 0.718937122372804490f,
  0.716801278521099540f,
  0.714658687862769090f, 0.712509370564692320f, 0.710353346857062420f,
  0.708190637033195400f,
  0.706021261449339740f, 0.703845240524484940f, 0.701662594740168570f,
  0.699473344640283770f,
  0.697277510830886630f, 0.695075113980000880f, 0.692866174817424740f,
  0.690650714134534720f,
  0.688428752784090550f, 0.686200311680038700f, 0.683965411797315510f,
  0.681724074171649820f,
  0.679476319899365080f, 0.677222170137180450f, 0.674961646102012040f,
  0.672694769070772970f,
  0.670421560380173090f, 0.668142041426518560f, 0.665856233665509720f,
  0.663564158612039880f,
  0.661265837839992270f, 0.658961292982037320f, 0.656650545729429050f,
  0.654333617831800550f,
  0.652010531096959500f, 0.649681307390683190f, 0.647345968636512060f,
  0.645004536815544040f,
  0.642657033966226860f, 0.640303482184151670f, 0.637943903621844170f,
  0.635578320488556230f,
  0.633206755050057190f, 0.630829229628424470f, 0.628445766601832710f,
  0.626056388404343520f,
  0.623661117525694640f, 0.621259976511087660f, 0.618852987960976320f,
  0.616440174530853650f,
  0.614021558931038490f, 0.611597163926462020f, 0.609167012336453210f,
  0.606731127034524480f,
  0.604289530948156070f, 0.601842247058580030f, 0.599389298400564540f,
  0.596930708062196500f,
  0.594466499184664540f, 0.591996694962040990f, 0.589521318641063940f,
  0.587040393520918080f,
  0.584553942953015330f, 0.582061990340775550f, 0.579564559139405740f,
  0.577061672855679550f,
  0.574553355047715760f, 0.572039629324757050f, 0.569520519346947250f,
  0.566996048825108680f,
  0.564466241520519500f, 0.561931121244689470f, 0.559390711859136140f,
  0.556845037275160100f,
  0.554294121453620110f, 0.551737988404707450f, 0.549176662187719770f,
  0.546610166910834860f,
  0.544038526730883930f, 0.541461765853123560f, 0.538879908531008420f,
  0.536292979065963180f,
  0.533701001807152960f, 0.531104001151255000f, 0.528502001542228480f,
  0.525895027471084740f,
  0.523283103475656430f, 0.520666254140367270f, 0.518044504095999340f,
  0.515417878019463150f,
  0.512786400633563070f, 0.510150096706766700f, 0.507508991052970870f,
  0.504863108531267480f,
  0.502212474045710900f, 0.499557112545081890f, 0.496897049022654640f,
  0.494232308515959730f,
  0.491562916106550060f, 0.488888896919763230f, 0.486210276124486530f,
  0.483527078932918740f,
  0.480839330600333900f, 0.478147056424843120f, 0.475450281747155870f,
  0.472749031950342900f,
  0.470043332459595620f, 0.467333208741988530f, 0.464618686306237820f,
  0.461899790702462840f,
  0.459176547521944150f, 0.456448982396883860f, 0.453717121000163930f,
  0.450980989045103810f,
  0.448240612285220000f, 0.445496016513981740f, 0.442747227564570130f,
  0.439994271309633260f,
  0.437237173661044200f, 0.434475960569655710f, 0.431710658025057370f,
  0.428941292055329550f,
  0.426167888726799620f, 0.423390474143796100f, 0.420609074448402510f,
  0.417823715820212380f,
  0.415034424476081630f, 0.412241226669883000f, 0.409444148692257590f,
  0.406643216870369140f,
  0.403838457567654130f, 0.401029897183575790f, 0.398217562153373620f,
  0.395401478947816300f,
  0.392581674072951530f, 0.389758174069856410f, 0.386931005514388690f,
  0.384100195016935040f,
  0.381265769222162490f, 0.378427754808765620f, 0.375586178489217330f,
  0.372741067009515810f,
  0.369892447148934270f, 0.367040345719767240f, 0.364184789567079840f,
  0.361325805568454340f,
  0.358463420633736540f, 0.355597661704783960f, 0.352728555755210730f,
  0.349856129790135030f,
  0.346980410845923680f, 0.344101425989938980f, 0.341219202320282410f,
  0.338333766965541290f,
  0.335445147084531660f, 0.332553369866044220f, 0.329658462528587550f,
  0.326760452320131790f,
  0.323859366517852960f, 0.320955232427875210f, 0.318048077385015060f,
  0.315137928752522440f,
  0.312224813921825050f, 0.309308760312268780f, 0.306389795370861080f,
  0.303467946572011370f,
  0.300543241417273400f, 0.297615707435086310f, 0.294685372180514330f,
  0.291752263234989370f,
  0.288816408206049480f, 0.285877834727080730f, 0.282936570457055390f,
  0.279992643080273380f,
  0.277046080306099950f, 0.274096909868706330f, 0.271145159526808070f,
  0.268190857063403180f,
  0.265234030285511900f, 0.262274707023913590f, 0.259312915132886350f,
  0.256348682489942910f,
  0.253382036995570270f, 0.250413006572965280f, 0.247441619167773440f,
  0.244467902747824210f,
  0.241491885302869300f, 0.238513594844318500f, 0.235533059404975460f,
  0.232550307038775330f,
  0.229565365820518870f, 0.226578263845610110f, 0.223589029229790020f,
  0.220597690108873650f,
  0.217604274638483670f, 0.214608810993786920f, 0.211611327369227610f,
  0.208611851978263460f,
  0.205610413053099320f, 0.202607038844421110f, 0.199601757621131050f,
  0.196594597670080220f,
  0.193585587295803750f, 0.190574754820252800f, 0.187562128582529740f,
  0.184547736938619640f,
  0.181531608261125130f, 0.178513770938997590f, 0.175494253377271400f,
  0.172473083996796030f,
  0.169450291233967930f, 0.166425903540464220f, 0.163399949382973230f,
  0.160372457242928400f,
  0.157343455616238280f, 0.154312973013020240f, 0.151281037957330250f,
  0.148247678986896200f,
  0.145212924652847520f, 0.142176803519448000f, 0.139139344163826280f,
  0.136100575175706200f,
  0.133060525157139180f, 0.130019222722233350f, 0.126976696496885980f,
  0.123932975118512200f,
  0.120888087235777220f, 0.117842061508325020f, 0.114794926606510250f,
  0.111746711211126660f,
  0.108697444013138670f, 0.105647153713410700f, 0.102595869022436280f,
  0.099543618660069444f,
  0.096490431355252607f, 0.093436335845747912f, 0.090381360877865011f,
  0.087325535206192226f,
  0.084268887593324127f, 0.081211446809592386f, 0.078153241632794315f,
  0.075094300847921291f,
  0.072034653246889416f, 0.068974327628266732f, 0.065913352797003930f,
  0.062851757564161420f,
  0.059789570746640007f, 0.056726821166907783f, 0.053663537652730679f,
  0.050599749036899337f,
  0.047535484156959261f, 0.044470771854938744f, 0.041405640977076712f,
  0.038340120373552791f,
  0.035274238898213947f, 0.032208025408304704f, 0.029141508764193740f,
  0.026074717829104040f,
  0.023007681468839410f, 0.019940428551514598f, 0.016872987947281773f,
  0.013805388528060349f,
  0.010737659167264572f, 0.007669828739531077f, 0.004601926120448672f,
  0.001533980186284766f
};

static const float32_t cos_factors_2048[2048] = {
  0.999999926465717890f, 0.999999338191525530f, 0.999998161643486980f,
  0.999996396822294350f,
  0.999994043728985820f, 0.999991102364945590f, 0.999987572731904080f,
  0.999983454831937730f,
  0.999978748667468830f, 0.999973454241265940f, 0.999967571556443780f,
  0.999961100616462820f,
  0.999954041425129780f, 0.999946393986597460f, 0.999938158305364590f,
  0.999929334386276070f,
  0.999919922234522750f, 0.999909921855641540f, 0.999899333255515390f,
  0.999888156440373320f,
  0.999876391416790410f, 0.999864038191687680f, 0.999851096772332190f,
  0.999837567166337090f,
  0.999823449381661570f, 0.999808743426610520f, 0.999793449309835270f,
  0.999777567040332940f,
  0.999761096627446610f, 0.999744038080865430f, 0.999726391410624470f,
  0.999708156627104880f,
  0.999689333741033640f, 0.999669922763483760f, 0.999649923705874240f,
  0.999629336579970110f,
  0.999608161397882110f, 0.999586398172067070f, 0.999564046915327740f,
  0.999541107640812940f,
  0.999517580362016990f, 0.999493465092780590f, 0.999468761847290050f,
  0.999443470640077770f,
  0.999417591486021720f, 0.999391124400346050f, 0.999364069398620550f,
  0.999336426496761240f,
  0.999308195711029470f, 0.999279377058032710f, 0.999249970554724420f,
  0.999219976218403530f,
  0.999189394066714920f, 0.999158224117649430f, 0.999126466389543390f,
  0.999094120901079070f,
  0.999061187671284600f, 0.999027666719533690f, 0.998993558065545680f,
  0.998958861729386080f,
  0.998923577731465780f, 0.998887706092541290f, 0.998851246833715180f,
  0.998814199976435390f,
  0.998776565542495610f, 0.998738343554035230f, 0.998699534033539280f,
  0.998660137003838490f,
  0.998620152488108870f, 0.998579580509872500f, 0.998538421092996730f,
  0.998496674261694640f,
  0.998454340040524800f, 0.998411418454391300f, 0.998367909528543820f,
  0.998323813288577560f,
  0.998279129760433200f, 0.998233858970396850f, 0.998188000945100300f,
  0.998141555711520520f,
  0.998094523296980010f, 0.998046903729146840f, 0.997998697036034390f,
  0.997949903246001190f,
  0.997900522387751620f, 0.997850554490335110f, 0.997799999583146470f,
  0.997748857695925690f,
  0.997697128858758500f, 0.997644813102075420f, 0.997591910456652630f,
  0.997538420953611340f,
  0.997484344624417930f, 0.997429681500884180f, 0.997374431615167150f,
  0.997318594999768600f,
  0.997262171687536170f, 0.997205161711661850f, 0.997147565105683480f,
  0.997089381903483400f,
  0.997030612139289450f, 0.996971255847674320f, 0.996911313063555740f,
  0.996850783822196610f,
  0.996789668159204560f, 0.996727966110532490f, 0.996665677712478160f,
  0.996602803001684130f,
  0.996539342015137940f, 0.996475294790172160f, 0.996410661364464100f,
  0.996345441776035900f,
  0.996279636063254650f, 0.996213244264832040f, 0.996146266419824620f,
  0.996078702567633980f,
  0.996010552748005870f, 0.995941817001031350f, 0.995872495367145730f,
  0.995802587887129160f,
  0.995732094602106430f, 0.995661015553546910f, 0.995589350783264600f,
  0.995517100333418110f,
  0.995444264246510340f, 0.995370842565388990f, 0.995296835333246090f,
  0.995222242593618360f,
  0.995147064390386470f, 0.995071300767776170f, 0.994994951770357020f,
  0.994918017443043200f,
  0.994840497831093180f, 0.994762392980109930f, 0.994683702936040250f,
  0.994604427745175660f,
  0.994524567454151740f, 0.994444122109948040f, 0.994363091759888570f,
  0.994281476451641550f,
  0.994199276233218910f, 0.994116491152977070f, 0.994033121259616400f,
  0.993949166602181130f,
  0.993864627230059750f, 0.993779503192984580f, 0.993693794541031790f,
  0.993607501324621610f,
  0.993520623594518090f, 0.993433161401829360f, 0.993345114798006910f,
  0.993256483834846440f,
  0.993167268564487230f, 0.993077469039412300f, 0.992987085312448390f,
  0.992896117436765980f,
  0.992804565465879140f, 0.992712429453645460f, 0.992619709454266140f,
  0.992526405522286100f,
  0.992432517712593660f, 0.992338046080420420f, 0.992242990681341700f,
  0.992147351571276090f,
  0.992051128806485720f, 0.991954322443575950f, 0.991856932539495470f,
  0.991758959151536110f,
  0.991660402337333210f, 0.991561262154865290f, 0.991461538662453790f,
  0.991361231918763460f,
  0.991260341982802440f, 0.991158868913921350f, 0.991056812771814340f,
  0.990954173616518500f,
  0.990850951508413620f, 0.990747146508222710f, 0.990642758677011570f,
  0.990537788076188750f,
  0.990432234767505970f, 0.990326098813057330f, 0.990219380275280000f,
  0.990112079216953770f,
  0.990004195701200910f, 0.989895729791486660f, 0.989786681551618640f,
  0.989677051045747210f,
  0.989566838338365120f, 0.989456043494307710f, 0.989344666578752640f,
  0.989232707657220050f,
  0.989120166795572690f, 0.989007044060015270f, 0.988893339517095130f,
  0.988779053233701520f,
  0.988664185277066230f, 0.988548735714763200f, 0.988432704614708340f,
  0.988316092045159690f,
  0.988198898074717610f, 0.988081122772324070f, 0.987962766207263420f,
  0.987843828449161740f,
  0.987724309567986960f, 0.987604209634049160f, 0.987483528717999710f,
  0.987362266890832400f,
  0.987240424223882250f, 0.987118000788826280f, 0.986994996657682980f,
  0.986871411902812470f,
  0.986747246596916590f, 0.986622500813038480f, 0.986497174624562880f,
  0.986371268105216030f,
  0.986244781329065460f, 0.986117714370520090f, 0.985990067304330140f,
  0.985861840205586980f,
  0.985733033149723490f, 0.985603646212513400f, 0.985473679470071810f,
  0.985343132998854790f,
  0.985212006875659350f, 0.985080301177623800f, 0.984948015982227030f,
  0.984815151367289140f,
  0.984681707410970940f, 0.984547684191773960f, 0.984413081788540700f,
  0.984277900280454370f,
  0.984142139747038570f, 0.984005800268157870f, 0.983868881924017220f,
  0.983731384795162090f,
  0.983593308962478650f, 0.983454654507193270f, 0.983315421510872810f,
  0.983175610055424420f,
  0.983035220223095640f, 0.982894252096474070f, 0.982752705758487830f,
  0.982610581292404750f,
  0.982467878781833170f, 0.982324598310721280f, 0.982180739963357090f,
  0.982036303824369020f,
  0.981891289978725100f, 0.981745698511732990f, 0.981599529509040720f,
  0.981452783056635520f,
  0.981305459240844670f, 0.981157558148334830f, 0.981009079866112630f,
  0.980860024481523870f,
  0.980710392082253970f, 0.980560182756327840f, 0.980409396592109910f,
  0.980258033678303550f,
  0.980106094103951770f, 0.979953577958436740f, 0.979800485331479790f,
  0.979646816313141210f,
  0.979492570993820810f, 0.979337749464256780f, 0.979182351815526930f,
  0.979026378139047580f,
  0.978869828526574120f, 0.978712703070200420f, 0.978555001862359550f,
  0.978396724995823090f,
  0.978237872563701090f, 0.978078444659442380f, 0.977918441376834370f,
  0.977757862810002760f,
  0.977596709053411890f, 0.977434980201864260f, 0.977272676350500860f,
  0.977109797594800880f,
  0.976946344030581670f, 0.976782315753998650f, 0.976617712861545640f,
  0.976452535450054060f,
  0.976286783616693630f, 0.976120457458971910f, 0.975953557074734300f,
  0.975786082562163930f,
  0.975618034019781750f, 0.975449411546446380f, 0.975280215241354220f,
  0.975110445204038890f,
  0.974940101534371830f, 0.974769184332561770f, 0.974597693699155050f,
  0.974425629735034990f,
  0.974252992541422500f, 0.974079782219875680f, 0.973905998872289570f,
  0.973731642600896400f,
  0.973556713508265560f, 0.973381211697303290f, 0.973205137271252800f,
  0.973028490333694210f,
  0.972851270988544180f, 0.972673479340056430f, 0.972495115492821190f,
  0.972316179551765300f,
  0.972136671622152230f, 0.971956591809581720f, 0.971775940219990140f,
  0.971594716959650160f,
  0.971412922135170940f, 0.971230555853497380f, 0.971047618221911100f,
  0.970864109348029470f,
  0.970680029339806130f, 0.970495378305530560f, 0.970310156353828110f,
  0.970124363593660280f,
  0.969938000134323960f, 0.969751066085452140f, 0.969563561557013180f,
  0.969375486659311280f,
  0.969186841502985950f, 0.968997626199012420f, 0.968807840858700970f,
  0.968617485593697540f,
  0.968426560515983190f, 0.968235065737874320f, 0.968043001372022260f,
  0.967850367531413620f,
  0.967657164329369880f, 0.967463391879547550f, 0.967269050295937790f,
  0.967074139692867040f,
  0.966878660184995910f, 0.966682611887320080f, 0.966485994915169840f,
  0.966288809384209690f,
  0.966091055410438830f, 0.965892733110190860f, 0.965693842600133690f,
  0.965494383997269500f,
  0.965294357418934660f, 0.965093762982799590f, 0.964892600806868890f,
  0.964690871009481030f,
  0.964488573709308410f, 0.964285709025357480f, 0.964082277076968140f,
  0.963878277983814200f,
  0.963673711865903230f, 0.963468578843575950f, 0.963262879037507070f,
  0.963056612568704340f,
  0.962849779558509030f, 0.962642380128595710f, 0.962434414400972100f,
  0.962225882497979020f,
  0.962016784542290560f, 0.961807120656913540f, 0.961596890965187860f,
  0.961386095590786250f,
  0.961174734657714080f, 0.960962808290309780f, 0.960750316613243950f,
  0.960537259751520050f,
  0.960323637830473920f, 0.960109450975773940f, 0.959894699313420530f,
  0.959679382969746750f,
  0.959463502071417510f, 0.959247056745430090f, 0.959030047119113660f,
  0.958812473320129310f,
  0.958594335476470220f, 0.958375633716461170f, 0.958156368168758820f,
  0.957936538962351420f,
  0.957716146226558870f, 0.957495190091032570f, 0.957273670685755200f,
  0.957051588141040970f,
  0.956828942587535370f, 0.956605734156215080f, 0.956381962978387730f,
  0.956157629185692140f,
  0.955932732910098280f, 0.955707274283906560f, 0.955481253439748770f,
  0.955254670510586990f,
  0.955027525629714160f, 0.954799818930753720f, 0.954571550547659630f,
  0.954342720614716480f,
  0.954113329266538800f, 0.953883376638071770f, 0.953652862864590500f,
  0.953421788081700310f,
  0.953190152425336670f, 0.952957956031764700f, 0.952725199037579570f,
  0.952491881579706320f,
  0.952258003795399600f, 0.952023565822243570f, 0.951788567798152130f,
  0.951553009861368590f,
  0.951316892150465550f, 0.951080214804345010f, 0.950842977962238160f,
  0.950605181763705340f,
  0.950366826348635780f, 0.950127911857248100f, 0.949888438430089300f,
  0.949648406208035480f,
  0.949407815332291570f, 0.949166665944390700f, 0.948924958186195160f,
  0.948682692199895090f,
  0.948439868128009620f, 0.948196486113385580f, 0.947952546299198670f,
  0.947708048828952100f,
  0.947462993846477700f, 0.947217381495934820f, 0.946971211921810880f,
  0.946724485268921170f,
  0.946477201682408680f, 0.946229361307743820f, 0.945980964290724760f,
  0.945732010777477150f,
  0.945482500914453740f, 0.945232434848435000f, 0.944981812726528150f,
  0.944730634696167800f,
  0.944478900905115550f, 0.944226611501459810f, 0.943973766633615980f,
  0.943720366450326200f,
  0.943466411100659320f, 0.943211900734010620f, 0.942956835500102120f,
  0.942701215548981900f,
  0.942445041031024890f, 0.942188312096931770f, 0.941931028897729620f,
  0.941673191584771360f,
  0.941414800309736340f, 0.941155855224629190f, 0.940896356481780830f,
  0.940636304233847590f,
  0.940375698633811540f, 0.940114539834980280f, 0.939852827990986680f,
  0.939590563255789270f,
  0.939327745783671400f, 0.939064375729241950f, 0.938800453247434770f,
  0.938535978493508560f,
  0.938270951623047190f, 0.938005372791958840f, 0.937739242156476970f,
  0.937472559873159250f,
  0.937205326098887960f, 0.936937540990869900f, 0.936669204706636170f,
  0.936400317404042060f,
  0.936130879241267030f, 0.935860890376814640f, 0.935590350969512370f,
  0.935319261178511610f,
  0.935047621163287430f, 0.934775431083638700f, 0.934502691099687870f,
  0.934229401371880820f,
  0.933955562060986730f, 0.933681173328098410f, 0.933406235334631520f,
  0.933130748242325230f,
  0.932854712213241120f, 0.932578127409764420f, 0.932300993994602760f,
  0.932023312130786490f,
  0.931745081981668720f, 0.931466303710925090f, 0.931186977482553750f,
  0.930907103460875130f,
  0.930626681810531760f, 0.930345712696488470f, 0.930064196284032360f,
  0.929782132738772190f,
  0.929499522226638560f, 0.929216364913884040f, 0.928932660967082820f,
  0.928648410553130520f,
  0.928363613839244370f, 0.928078270992963140f, 0.927792382182146320f,
  0.927505947574975180f,
  0.927218967339951790f, 0.926931441645899130f, 0.926643370661961230f,
  0.926354754557602860f,
  0.926065593502609310f, 0.925775887667086740f, 0.925485637221461490f,
  0.925194842336480530f,
  0.924903503183210910f, 0.924611619933039970f, 0.924319192757675160f,
  0.924026221829143850f,
  0.923732707319793290f, 0.923438649402290370f, 0.923144048249621930f,
  0.922848904035094120f,
  0.922553216932332830f, 0.922256987115283030f, 0.921960214758209220f,
  0.921662900035694730f,
  0.921365043122642340f, 0.921066644194273640f, 0.920767703426128790f,
  0.920468220994067110f,
  0.920168197074266340f, 0.919867631843222950f, 0.919566525477751530f,
  0.919264878154985370f,
  0.918962690052375630f, 0.918659961347691900f, 0.918356692219021720f,
  0.918052882844770380f,
  0.917748533403661250f, 0.917443644074735220f, 0.917138215037350710f,
  0.916832246471183890f,
  0.916525738556228210f, 0.916218691472794220f, 0.915911105401509880f,
  0.915602980523320230f,
  0.915294317019487050f, 0.914985115071589310f, 0.914675374861522390f,
  0.914365096571498560f,
  0.914054280384046570f, 0.913742926482011390f, 0.913431035048554720f,
  0.913118606267154240f,
  0.912805640321603500f, 0.912492137396012650f, 0.912178097674807180f,
  0.911863521342728520f,
  0.911548408584833990f, 0.911232759586496190f, 0.910916574533403360f,
  0.910599853611558930f,
  0.910282597007281760f, 0.909964804907205660f, 0.909646477498279540f,
  0.909327614967767260f,
  0.909008217503247450f, 0.908688285292613360f, 0.908367818524072890f,
  0.908046817386148340f,
  0.907725282067676440f, 0.907403212757808110f, 0.907080609646008450f,
  0.906757472922056550f,
  0.906433802776045460f, 0.906109599398381980f, 0.905784862979786550f,
  0.905459593711293250f,
  0.905133791784249690f, 0.904807457390316540f, 0.904480590721468250f,
  0.904153191969991780f,
  0.903825261328487510f, 0.903496798989868450f, 0.903167805147360720f,
  0.902838279994502830f,
  0.902508223725145940f, 0.902177636533453620f, 0.901846518613901750f,
  0.901514870161278740f,
  0.901182691370684520f, 0.900849982437531450f, 0.900516743557543520f,
  0.900182974926756810f,
  0.899848676741518580f, 0.899513849198487980f, 0.899178492494635330f,
  0.898842606827242370f,
  0.898506192393901950f, 0.898169249392518080f, 0.897831778021305650f,
  0.897493778478790310f,
  0.897155250963808550f, 0.896816195675507300f, 0.896476612813344120f,
  0.896136502577086770f,
  0.895795865166813530f, 0.895454700782912450f, 0.895113009626081760f,
  0.894770791897329550f,
  0.894428047797973800f, 0.894084777529641990f, 0.893740981294271040f,
  0.893396659294107720f,
  0.893051811731707450f, 0.892706438809935390f, 0.892360540731965360f,
  0.892014117701280470f,
  0.891667169921672280f, 0.891319697597241390f, 0.890971700932396860f,
  0.890623180131855930f,
  0.890274135400644600f, 0.889924566944096720f, 0.889574474967854580f,
  0.889223859677868210f,
  0.888872721280395630f, 0.888521059982002260f, 0.888168875989561730f,
  0.887816169510254440f,
  0.887462940751568840f, 0.887109189921300170f, 0.886754917227550840f,
  0.886400122878730600f,
  0.886044807083555600f, 0.885688970051048960f, 0.885332611990540590f,
  0.884975733111666660f,
  0.884618333624369920f, 0.884260413738899190f, 0.883901973665809470f,
  0.883543013615961880f,
  0.883183533800523390f, 0.882823534430966620f, 0.882463015719070150f,
  0.882101977876917580f,
  0.881740421116898320f, 0.881378345651706920f, 0.881015751694342870f,
  0.880652639458111010f,
  0.880289009156621010f, 0.879924861003786860f, 0.879560195213827890f,
  0.879195012001267480f,
  0.878829311580933360f, 0.878463094167957870f, 0.878096359977777130f,
  0.877729109226131570f,
  0.877361342129065140f, 0.876993058902925890f, 0.876624259764365310f,
  0.876254944930338510f,
  0.875885114618103810f, 0.875514769045222850f, 0.875143908429560360f,
  0.874772532989284150f,
  0.874400642942864790f, 0.874028238509075740f, 0.873655319906992630f,
  0.873281887355994210f,
  0.872907941075761080f, 0.872533481286276170f, 0.872158508207824480f,
  0.871783022060993120f,
  0.871407023066670950f, 0.871030511446048260f, 0.870653487420617430f,
  0.870275951212171940f,
  0.869897903042806340f, 0.869519343134916860f, 0.869140271711200560f,
  0.868760688994655310f,
  0.868380595208579800f, 0.867999990576573510f, 0.867618875322536230f,
  0.867237249670668400f,
  0.866855113845470430f, 0.866472468071743050f, 0.866089312574586770f,
  0.865705647579402380f,
  0.865321473311889800f, 0.864936789998049020f, 0.864551597864179340f,
  0.864165897136879300f,
  0.863779688043046720f, 0.863392970809878420f, 0.863005745664870320f,
  0.862618012835816740f,
  0.862229772550811240f, 0.861841025038245330f, 0.861451770526809320f,
  0.861062009245491480f,
  0.860671741423578380f, 0.860280967290654510f, 0.859889687076602290f,
  0.859497901011601730f,
  0.859105609326130450f, 0.858712812250963520f, 0.858319510017173440f,
  0.857925702856129790f,
  0.857531390999499150f, 0.857136574679244980f, 0.856741254127627470f,
  0.856345429577203610f,
  0.855949101260826910f, 0.855552269411646860f, 0.855154934263109620f,
  0.854757096048957220f,
  0.854358755003227440f, 0.853959911360254180f, 0.853560565354666840f,
  0.853160717221390420f,
  0.852760367195645300f, 0.852359515512947090f, 0.851958162409106380f,
  0.851556308120228980f,
  0.851153952882715340f, 0.850751096933260790f, 0.850347740508854980f,
  0.849943883846782210f,
  0.849539527184620890f, 0.849134670760243630f, 0.848729314811817130f,
  0.848323459577801640f,
  0.847917105296951410f, 0.847510252208314330f, 0.847102900551231500f,
  0.846695050565337450f,
  0.846286702490559710f, 0.845877856567119000f, 0.845468513035528830f,
  0.845058672136595470f,
  0.844648334111417820f, 0.844237499201387020f, 0.843826167648186740f,
  0.843414339693792760f,
  0.843002015580472940f, 0.842589195550786710f, 0.842175879847585570f,
  0.841762068714012490f,
  0.841347762393501950f, 0.840932961129779780f, 0.840517665166862550f,
  0.840101874749058400f,
  0.839685590120966110f, 0.839268811527475230f, 0.838851539213765760f,
  0.838433773425308340f,
  0.838015514407863820f, 0.837596762407483040f, 0.837177517670507300f,
  0.836757780443567190f,
  0.836337550973583530f, 0.835916829507766360f, 0.835495616293615350f,
  0.835073911578919410f,
  0.834651715611756440f, 0.834229028640493420f, 0.833805850913786340f,
  0.833382182680579730f,
  0.832958024190106670f, 0.832533375691888680f, 0.832108237435735590f,
  0.831682609671745120f,
  0.831256492650303210f, 0.830829886622083570f, 0.830402791838047550f,
  0.829975208549443950f,
  0.829547137007808910f, 0.829118577464965980f, 0.828689530173025820f,
  0.828259995384385660f,
  0.827829973351729920f, 0.827399464328029470f, 0.826968468566541600f,
  0.826536986320809960f,
  0.826105017844664610f, 0.825672563392221390f, 0.825239623217882250f,
  0.824806197576334330f,
  0.824372286722551250f, 0.823937890911791370f, 0.823503010399598500f,
  0.823067645441801670f,
  0.822631796294514990f, 0.822195463214137170f, 0.821758646457351750f,
  0.821321346281126740f,
  0.820883562942714580f, 0.820445296699652050f, 0.820006547809759680f,
  0.819567316531142230f,
  0.819127603122188240f, 0.818687407841569680f, 0.818246730948242070f,
  0.817805572701444270f,
  0.817363933360698460f, 0.816921813185809480f, 0.816479212436865390f,
  0.816036131374236810f,
  0.815592570258576790f, 0.815148529350820830f, 0.814704008912187080f,
  0.814259009204175270f,
  0.813813530488567190f, 0.813367573027426570f, 0.812921137083098770f,
  0.812474222918210480f,
  0.812026830795669730f, 0.811578960978665890f, 0.811130613730669190f,
  0.810681789315430780f,
  0.810232487996982330f, 0.809782710039636530f, 0.809332455707985950f,
  0.808881725266903610f,
  0.808430518981542720f, 0.807978837117336310f, 0.807526679939997160f,
  0.807074047715517610f,
  0.806620940710169650f, 0.806167359190504420f, 0.805713303423352230f,
  0.805258773675822210f,
  0.804803770215302920f, 0.804348293309460780f, 0.803892343226241260f,
  0.803435920233868120f,
  0.802979024600843250f, 0.802521656595946430f, 0.802063816488235440f,
  0.801605504547046150f,
  0.801146721041991360f, 0.800687466242961610f, 0.800227740420124790f,
  0.799767543843925680f,
  0.799306876785086160f, 0.798845739514604580f, 0.798384132303756380f,
  0.797922055424093000f,
  0.797459509147442460f, 0.796996493745908750f, 0.796533009491872000f,
  0.796069056657987990f,
  0.795604635517188070f, 0.795139746342679590f, 0.794674389407944550f,
  0.794208564986740640f,
  0.793742273353100210f, 0.793275514781330630f, 0.792808289546014120f,
  0.792340597922007170f,
  0.791872440184440470f, 0.791403816608719500f, 0.790934727470523290f,
  0.790465173045804880f,
  0.789995153610791090f, 0.789524669441982190f, 0.789053720816151880f,
  0.788582308010347120f,
  0.788110431301888070f, 0.787638090968367450f, 0.787165287287651010f,
  0.786692020537876790f,
  0.786218290997455660f, 0.785744098945070360f, 0.785269444659675850f,
  0.784794328420499230f,
  0.784318750507038920f, 0.783842711199065230f, 0.783366210776619720f,
  0.782889249520015480f,
  0.782411827709836530f, 0.781933945626937630f, 0.781455603552444590f,
  0.780976801767753750f,
  0.780497540554531910f, 0.780017820194715990f, 0.779537640970513260f,
  0.779057003164400630f,
  0.778575907059125050f, 0.778094352937702790f, 0.777612341083420030f,
  0.777129871779831620f,
  0.776646945310762060f, 0.776163561960304340f, 0.775679722012820650f,
  0.775195425752941420f,
  0.774710673465565550f, 0.774225465435860680f, 0.773739801949261840f,
  0.773253683291472590f,
  0.772767109748463850f, 0.772280081606474320f, 0.771792599152010150f,
  0.771304662671844830f,
  0.770816272453018540f, 0.770327428782838890f, 0.769838131948879840f,
  0.769348382238982280f,
  0.768858179941253270f, 0.768367525344066270f, 0.767876418736060610f,
  0.767384860406141730f,
  0.766892850643480670f, 0.766400389737514230f, 0.765907477977944340f,
  0.765414115654738270f,
  0.764920303058128410f, 0.764426040478612070f, 0.763931328206951090f,
  0.763436166534172010f,
  0.762940555751565720f, 0.762444496150687210f, 0.761947988023355390f,
  0.761451031661653620f,
  0.760953627357928150f, 0.760455775404789260f, 0.759957476095110330f,
  0.759458729722028210f,
  0.758959536578942440f, 0.758459896959515430f, 0.757959811157672300f,
  0.757459279467600720f,
  0.756958302183750490f, 0.756456879600833740f, 0.755955012013824420f,
  0.755452699717958250f,
  0.754949943008732640f, 0.754446742181906440f, 0.753943097533499640f,
  0.753439009359793580f,
  0.752934477957330150f, 0.752429503622912390f, 0.751924086653603550f,
  0.751418227346727470f,
  0.750911925999867890f, 0.750405182910869330f, 0.749897998377835330f,
  0.749390372699129560f,
  0.748882306173375150f, 0.748373799099454560f, 0.747864851776509410f,
  0.747355464503940190f,
  0.746845637581406540f, 0.746335371308826320f, 0.745824665986376090f,
  0.745313521914490520f,
  0.744801939393862630f, 0.744289918725443260f, 0.743777460210440890f,
  0.743264564150321600f,
  0.742751230846809050f, 0.742237460601884000f, 0.741723253717784140f,
  0.741208610497004260f,
  0.740693531242295760f, 0.740178016256666240f, 0.739662065843380010f,
  0.739145680305957510f,
  0.738628859948174840f, 0.738111605074064260f, 0.737593915987913570f,
  0.737075792994265730f,
  0.736557236397919150f, 0.736038246503927350f, 0.735518823617598900f,
  0.734998968044496710f,
  0.734478680090438370f, 0.733957960061495940f, 0.733436808263995710f,
  0.732915225004517780f,
  0.732393210589896040f, 0.731870765327218290f, 0.731347889523825570f,
  0.730824583487312160f,
  0.730300847525525490f, 0.729776681946566090f, 0.729252087058786970f,
  0.728727063170793830f,
  0.728201610591444610f, 0.727675729629849610f, 0.727149420595371020f,
  0.726622683797622850f,
  0.726095519546471000f, 0.725567928152032300f, 0.725039909924675370f,
  0.724511465175019630f,
  0.723982594213935520f, 0.723453297352544380f, 0.722923574902217700f,
  0.722393427174577550f,
  0.721862854481496340f, 0.721331857135096290f, 0.720800435447749190f,
  0.720268589732077190f,
  0.719736320300951030f, 0.719203627467491220f, 0.718670511545067230f,
  0.718136972847297490f,
  0.717603011688049080f, 0.717068628381437480f, 0.716533823241826680f,
  0.715998596583828690f,
  0.715462948722303760f, 0.714926879972359490f, 0.714390390649351390f,
  0.713853481068882470f,
  0.713316151546802610f, 0.712778402399208980f, 0.712240233942445510f,
  0.711701646493102970f,
  0.711162640368018350f, 0.710623215884275020f, 0.710083373359202800f,
  0.709543113110376770f,
  0.709002435455618250f, 0.708461340712994160f, 0.707919829200816310f,
  0.707377901237642100f,
  0.706835557142273860f, 0.706292797233758480f, 0.705749621831387790f,
  0.705206031254697830f,
  0.704662025823468930f, 0.704117605857725430f, 0.703572771677735580f,
  0.703027523604011220f,
  0.702481861957308000f, 0.701935787058624360f, 0.701389299229202230f,
  0.700842398790526230f,
  0.700295086064323780f, 0.699747361372564990f, 0.699199225037462120f,
  0.698650677381469580f,
  0.698101718727283880f, 0.697552349397843270f, 0.697002569716327460f,
  0.696452380006157830f,
  0.695901780590996830f, 0.695350771794747800f, 0.694799353941554900f,
  0.694247527355803310f,
  0.693695292362118350f, 0.693142649285365510f, 0.692589598450650380f,
  0.692036140183318830f,
  0.691482274808955850f, 0.690928002653386280f, 0.690373324042674040f,
  0.689818239303122470f,
  0.689262748761273470f, 0.688706852743907750f, 0.688150551578044830f,
  0.687593845590942170f,
  0.687036735110095660f, 0.686479220463238950f, 0.685921301978343670f,
  0.685362979983618730f,
  0.684804254807510620f, 0.684245126778703080f, 0.683685596226116690f,
  0.683125663478908800f,
  0.682565328866473250f, 0.682004592718440830f, 0.681443455364677990f,
  0.680881917135287340f,
  0.680319978360607200f, 0.679757639371212030f, 0.679194900497911200f,
  0.678631762071749470f,
  0.678068224424006600f, 0.677504287886197430f, 0.676939952790071240f,
  0.676375219467611700f,
  0.675810088251037060f, 0.675244559472799270f, 0.674678633465584540f,
  0.674112310562312360f,
  0.673545591096136100f, 0.672978475400442090f, 0.672410963808849900f,
  0.671843056655211930f,
  0.671274754273613490f, 0.670706056998372160f, 0.670136965164037760f,
  0.669567479105392490f,
  0.668997599157450270f, 0.668427325655456820f, 0.667856658934889440f,
  0.667285599331456480f,
  0.666714147181097670f, 0.666142302819983540f, 0.665570066584515560f,
  0.664997438811325340f,
  0.664424419837275180f, 0.663851009999457340f, 0.663277209635194100f,
  0.662703019082037440f,
  0.662128438677768720f, 0.661553468760399000f, 0.660978109668168060f,
  0.660402361739545030f,
  0.659826225313227430f, 0.659249700728141490f, 0.658672788323441890f,
  0.658095488438511290f,
  0.657517801412960120f, 0.656939727586627110f, 0.656361267299578000f,
  0.655782420892106030f,
  0.655203188704731930f, 0.654623571078202680f, 0.654043568353492640f,
  0.653463180871802330f,
  0.652882408974558960f, 0.652301253003415460f, 0.651719713300251020f,
  0.651137790207170330f,
  0.650555484066503990f, 0.649972795220807530f, 0.649389724012861770f,
  0.648806270785672550f,
  0.648222435882470420f, 0.647638219646710420f, 0.647053622422071650f,
  0.646468644552457890f,
  0.645883286381996440f, 0.645297548255038380f, 0.644711430516158420f,
  0.644124933510154540f,
  0.643538057582047850f, 0.642950803077082080f, 0.642363170340724320f,
  0.641775159718663500f,
  0.641186771556811250f, 0.640598006201301030f, 0.640008863998488440f,
  0.639419345294950700f,
  0.638829450437486400f, 0.638239179773115390f, 0.637648533649078810f,
  0.637057512412838590f,
  0.636466116412077180f, 0.635874345994697720f, 0.635282201508823530f,
  0.634689683302797850f,
  0.634096791725183740f, 0.633503527124764320f, 0.632909889850541860f,
  0.632315880251737680f,
  0.631721498677792370f, 0.631126745478365340f, 0.630531621003334600f,
  0.629936125602796550f,
  0.629340259627065750f, 0.628744023426674790f, 0.628147417352374120f,
  0.627550441755131530f,
  0.626953096986132770f, 0.626355383396779990f, 0.625757301338692900f,
  0.625158851163707730f,
  0.624560033223877320f, 0.623960847871470770f, 0.623361295458973340f,
  0.622761376339086460f,
  0.622161090864726930f, 0.621560439389027270f, 0.620959422265335180f,
  0.620358039847213830f,
  0.619756292488440660f, 0.619154180543008410f, 0.618551704365123860f,
  0.617948864309208260f,
  0.617345660729896940f, 0.616742093982038830f, 0.616138164420696910f,
  0.615533872401147430f,
  0.614929218278879590f, 0.614324202409595950f, 0.613718825149211830f,
  0.613113086853854910f,
  0.612506987879865570f, 0.611900528583796070f, 0.611293709322411010f,
  0.610686530452686280f,
  0.610078992331809620f, 0.609471095317180240f, 0.608862839766408200f,
  0.608254226037314490f,
  0.607645254487930830f, 0.607035925476499760f, 0.606426239361473550f,
  0.605816196501515080f,
  0.605205797255496500f, 0.604595041982500360f, 0.603983931041818020f,
  0.603372464792950370f,
  0.602760643595607220f, 0.602148467809707320f, 0.601535937795377730f,
  0.600923053912954090f,
  0.600309816522980430f, 0.599696225986208310f, 0.599082282663597310f,
  0.598467986916314310f,
  0.597853339105733910f, 0.597238339593437530f, 0.596622988741213330f,
  0.596007286911056530f,
  0.595391234465168730f, 0.594774831765957580f, 0.594158079176036800f,
  0.593540977058226390f,
  0.592923525775551410f, 0.592305725691242400f, 0.591687577168735550f,
  0.591069080571671510f,
  0.590450236263895920f, 0.589831044609458900f, 0.589211505972615070f,
  0.588591620717822890f,
  0.587971389209745120f, 0.587350811813247660f, 0.586729888893400500f,
  0.586108620815476430f,
  0.585487007944951450f, 0.584865050647504490f, 0.584242749289016980f,
  0.583620104235572760f,
  0.582997115853457700f, 0.582373784509160220f, 0.581750110569369760f,
  0.581126094400977620f,
  0.580501736371076600f, 0.579877036846960350f, 0.579251996196123550f,
  0.578626614786261430f,
  0.578000892985269910f, 0.577374831161244880f, 0.576748429682482520f,
  0.576121688917478390f,
  0.575494609234928230f, 0.574867191003726740f, 0.574239434592967890f,
  0.573611340371944610f,
  0.572982908710148680f, 0.572354139977270030f, 0.571725034543197120f,
  0.571095592778016690f,
  0.570465815052012990f, 0.569835701735668110f, 0.569205253199661200f,
  0.568574469814869250f,
  0.567943351952365670f, 0.567311899983420800f, 0.566680114279501710f,
  0.566047995212271560f,
  0.565415543153589770f, 0.564782758475511400f, 0.564149641550287680f,
  0.563516192750364910f,
  0.562882412448384550f, 0.562248301017183150f, 0.561613858829792420f,
  0.560979086259438260f,
  0.560343983679540860f, 0.559708551463714790f, 0.559072789985768480f,
  0.558436699619704100f,
  0.557800280739717100f, 0.557163533720196340f, 0.556526458935723720f,
  0.555889056761073920f,
  0.555251327571214090f, 0.554613271741304040f, 0.553974889646695610f,
  0.553336181662932410f,
  0.552697148165749770f, 0.552057789531074980f, 0.551418106135026060f,
  0.550778098353912230f,
  0.550137766564233630f, 0.549497111142680960f, 0.548856132466135290f,
  0.548214830911667780f,
  0.547573206856539870f, 0.546931260678202190f, 0.546288992754295210f,
  0.545646403462648590f,
  0.545003493181281160f, 0.544360262288400400f, 0.543716711162402390f,
  0.543072840181871850f,
  0.542428649725581360f, 0.541784140172491660f, 0.541139311901750910f,
  0.540494165292695230f,
  0.539848700724847700f, 0.539202918577918240f, 0.538556819231804210f,
  0.537910403066588990f,
  0.537263670462542530f, 0.536616621800121150f, 0.535969257459966710f,
  0.535321577822907010f,
  0.534673583269955510f, 0.534025274182310380f, 0.533376650941355560f,
  0.532727713928658810f,
  0.532078463525973540f, 0.531428900115236910f, 0.530779024078570250f,
  0.530128835798278850f,
  0.529478335656852090f, 0.528827524036961980f, 0.528176401321464370f,
  0.527524967893398200f,
  0.526873224135984700f, 0.526221170432628170f, 0.525568807166914680f,
  0.524916134722612890f,
  0.524263153483673470f, 0.523609863834228030f, 0.522956266158590140f,
  0.522302360841254700f,
  0.521648148266897090f, 0.520993628820373810f, 0.520338802886721960f,
  0.519683670851158520f,
  0.519028233099080970f, 0.518372490016066220f, 0.517716441987871150f,
  0.517060089400432130f,
  0.516403432639863990f, 0.515746472092461380f, 0.515089208144697270f,
  0.514431641183222930f,
  0.513773771594868030f, 0.513115599766640560f, 0.512457126085725800f,
  0.511798350939487000f,
  0.511139274715464390f, 0.510479897801375700f, 0.509820220585115560f,
  0.509160243454754750f,
  0.508499966798540810f, 0.507839391004897940f, 0.507178516462425290f,
  0.506517343559898530f,
  0.505855872686268860f, 0.505194104230662240f, 0.504532038582380380f,
  0.503869676130898950f,
  0.503207017265869030f, 0.502544062377115800f, 0.501880811854638400f,
  0.501217266088609950f,
  0.500553425469377640f, 0.499889290387461380f, 0.499224861233555030f,
  0.498560138398525200f,
  0.497895122273410930f, 0.497229813249424340f, 0.496564211717949340f,
  0.495898318070542240f,
  0.495232132698931350f, 0.494565655995016010f, 0.493898888350867430f,
  0.493231830158728070f,
  0.492564481811010650f, 0.491896843700299240f, 0.491228916219348330f,
  0.490560699761082080f,
  0.489892194718595300f, 0.489223401485152030f, 0.488554320454186230f,
  0.487884952019301210f,
  0.487215296574268820f, 0.486545354513030270f, 0.485875126229695420f,
  0.485204612118541880f,
  0.484533812574016120f, 0.483862727990732320f, 0.483191358763471910f,
  0.482519705287184520f,
  0.481847767956986080f, 0.481175547168160360f, 0.480503043316157670f,
  0.479830256796594250f,
  0.479157188005253310f, 0.478483837338084080f, 0.477810205191201040f,
  0.477136291960884750f,
  0.476462098043581310f, 0.475787623835901120f, 0.475112869734620470f,
  0.474437836136679340f,
  0.473762523439182850f, 0.473086932039400220f, 0.472411062334764100f,
  0.471734914722871430f,
  0.471058489601482610f, 0.470381787368520710f, 0.469704808422072460f,
  0.469027553160387240f,
  0.468350021981876530f, 0.467672215285114710f, 0.466994133468838110f,
  0.466315776931944480f,
  0.465637146073493770f, 0.464958241292706740f, 0.464279062988965760f,
  0.463599611561814120f,
  0.462919887410955130f, 0.462239890936253280f, 0.461559622537733190f,
  0.460879082615578690f,
  0.460198271570134270f, 0.459517189801903590f, 0.458835837711549120f,
  0.458154215699893230f,
  0.457472324167916110f, 0.456790163516757220f, 0.456107734147714220f,
  0.455425036462242420f,
  0.454742070861955450f, 0.454058837748624540f, 0.453375337524177750f,
  0.452691570590700860f,
  0.452007537350436530f, 0.451323238205783520f, 0.450638673559297760f,
  0.449953843813690580f,
  0.449268749371829920f, 0.448583390636739300f, 0.447897768011597360f,
  0.447211881899738260f,
  0.446525732704651400f, 0.445839320829980350f, 0.445152646679523590f,
  0.444465710657234110f,
  0.443778513167218280f, 0.443091054613736990f, 0.442403335401204130f,
  0.441715355934187310f,
  0.441027116617407340f, 0.440338617855737300f, 0.439649860054203420f,
  0.438960843617984430f,
  0.438271568952410480f, 0.437582036462964340f, 0.436892246555280470f,
  0.436202199635143950f,
  0.435511896108492170f, 0.434821336381412350f, 0.434130520860143310f,
  0.433439449951074200f,
  0.432748124060743760f, 0.432056543595841450f, 0.431364708963206440f,
  0.430672620569826860f,
  0.429980278822840570f, 0.429287684129534720f, 0.428594836897344400f,
  0.427901737533854240f,
  0.427208386446796370f, 0.426514784044051520f, 0.425820930733648350f,
  0.425126826923762410f,
  0.424432473022717420f, 0.423737869438983950f, 0.423043016581179100f,
  0.422347914858067000f,
  0.421652564678558380f, 0.420956966451709440f, 0.420261120586723050f,
  0.419565027492946940f,
  0.418868687579875110f, 0.418172101257146430f, 0.417475268934544340f,
  0.416778191021997590f,
  0.416080867929579320f, 0.415383300067506290f, 0.414685487846140010f,
  0.413987431675985510f,
  0.413289131967690960f, 0.412590589132048380f, 0.411891803579992220f,
  0.411192775722600160f,
  0.410493505971092520f, 0.409793994736831200f, 0.409094242431320920f,
  0.408394249466208110f,
  0.407694016253280170f, 0.406993543204466460f, 0.406292830731837470f,
  0.405591879247603870f,
  0.404890689164117750f, 0.404189260893870750f, 0.403487594849495310f,
  0.402785691443763640f,
  0.402083551089587040f, 0.401381174200016790f, 0.400678561188243350f,
  0.399975712467595390f,
  0.399272628451540930f, 0.398569309553686360f, 0.397865756187775750f,
  0.397161968767691720f,
  0.396457947707453960f, 0.395753693421220080f, 0.395049206323284880f,
  0.394344486828079650f,
  0.393639535350172880f, 0.392934352304269600f, 0.392228938105210370f,
  0.391523293167972350f,
  0.390817417907668610f, 0.390111312739546910f, 0.389404978078991100f,
  0.388698414341519250f,
  0.387991621942784910f, 0.387284601298575890f, 0.386577352824813980f,
  0.385869876937555310f,
  0.385162174052989970f, 0.384454244587440870f, 0.383746088957365010f,
  0.383037707579352130f,
  0.382329100870124510f, 0.381620269246537520f, 0.380911213125578130f,
  0.380201932924366050f,
  0.379492429060152740f, 0.378782701950320600f, 0.378072752012383990f,
  0.377362579663988450f,
  0.376652185322909620f, 0.375941569407054420f, 0.375230732334460030f,
  0.374519674523293210f,
  0.373808396391851370f, 0.373096898358560690f, 0.372385180841977360f,
  0.371673244260786630f,
  0.370961089033802040f, 0.370248715579966360f, 0.369536124318350760f,
  0.368823315668153960f,
  0.368110290048703050f, 0.367397047879452820f, 0.366683589579984930f,
  0.365969915570008910f,
  0.365256026269360380f, 0.364541922098002180f, 0.363827603476023610f,
  0.363113070823639530f,
  0.362398324561191310f, 0.361683365109145950f, 0.360968192888095290f,
  0.360252808318756830f,
  0.359537211821973180f, 0.358821403818710860f, 0.358105384730061760f,
  0.357389154977241000f,
  0.356672714981588260f, 0.355956065164567010f, 0.355239205947763370f,
  0.354522137752887430f,
  0.353804861001772160f, 0.353087376116372530f, 0.352369683518766630f,
  0.351651783631154680f,
  0.350933676875858360f, 0.350215363675321740f, 0.349496844452109600f,
  0.348778119628908420f,
  0.348059189628525780f, 0.347340054873889190f, 0.346620715788047320f,
  0.345901172794169100f,
  0.345181426315542610f, 0.344461476775576480f, 0.343741324597798600f,
  0.343020970205855540f,
  0.342300414023513690f, 0.341579656474657210f, 0.340858697983289440f,
  0.340137538973531880f,
  0.339416179869623410f, 0.338694621095921190f, 0.337972863076899830f,
  0.337250906237150650f,
  0.336528751001382350f, 0.335806397794420560f, 0.335083847041206580f,
  0.334361099166798900f,
  0.333638154596370920f, 0.332915013755212650f, 0.332191677068729320f,
  0.331468144962440920f,
  0.330744417861982890f, 0.330020496193105530f, 0.329296380381672800f,
  0.328572070853663690f,
  0.327847568035170960f, 0.327122872352400510f, 0.326397984231672660f,
  0.325672904099419900f,
  0.324947632382188430f, 0.324222169506637130f, 0.323496515899536760f,
  0.322770671987770710f,
  0.322044638198334620f, 0.321318414958334910f, 0.320592002694990330f,
  0.319865401835630610f,
  0.319138612807695900f, 0.318411636038737960f, 0.317684471956418020f,
  0.316957120988508150f,
  0.316229583562890490f, 0.315501860107556040f, 0.314773951050606070f,
  0.314045856820250820f,
  0.313317577844809070f, 0.312589114552708660f, 0.311860467372486130f,
  0.311131636732785270f,
  0.310402623062358880f, 0.309673426790066490f, 0.308944048344875710f,
  0.308214488155861220f,
  0.307484746652204160f, 0.306754824263192780f, 0.306024721418221900f,
  0.305294438546791720f,
  0.304563976078509050f, 0.303833334443086470f, 0.303102514070341060f,
  0.302371515390196130f,
  0.301640338832678880f, 0.300908984827921890f, 0.300177453806162120f,
  0.299445746197739950f,
  0.298713862433100390f, 0.297981802942791920f, 0.297249568157465890f,
  0.296517158507877410f,
  0.295784574424884370f, 0.295051816339446720f, 0.294318884682627570f,
  0.293585779885591310f,
  0.292852502379604810f, 0.292119052596036540f, 0.291385430966355720f,
  0.290651637922133220f,
  0.289917673895040860f, 0.289183539316850310f, 0.288449234619434170f,
  0.287714760234765280f,
  0.286980116594915570f, 0.286245304132057120f, 0.285510323278461380f,
  0.284775174466498300f,
  0.284039858128637360f, 0.283304374697445790f, 0.282568724605589740f,
  0.281832908285833460f,
  0.281096926171038320f, 0.280360778694163810f, 0.279624466288266700f,
  0.278887989386500280f,
  0.278151348422115090f, 0.277414543828458200f, 0.276677576038972420f,
  0.275940445487197320f,
  0.275203152606767370f, 0.274465697831413220f, 0.273728081594960650f,
  0.272990304331329980f,
  0.272252366474536660f, 0.271514268458690810f, 0.270776010717996010f,
  0.270037593686750510f,
  0.269299017799346230f, 0.268560283490267890f, 0.267821391194094320f,
  0.267082341345496350f,
  0.266343134379238180f, 0.265603770730176440f, 0.264864250833259320f,
  0.264124575123527490f,
  0.263384744036113390f, 0.262644758006240100f, 0.261904617469222560f,
  0.261164322860466590f,
  0.260423874615468010f, 0.259683273169813930f, 0.258942518959180580f,
  0.258201612419334870f,
  0.257460553986133210f, 0.256719344095520720f, 0.255977983183532380f,
  0.255236471686291820f,
  0.254494810040010790f, 0.253752998680989940f, 0.253011038045617980f,
  0.252268928570370810f,
  0.251526670691812780f, 0.250784264846594550f, 0.250041711471454650f,
  0.249299011003218300f,
  0.248556163878796620f, 0.247813170535187620f, 0.247070031409475370f,
  0.246326746938829060f,
  0.245583317560504000f, 0.244839743711840750f, 0.244096025830264210f,
  0.243352164353284880f,
  0.242608159718496890f, 0.241864012363579210f, 0.241119722726294730f,
  0.240375291244489500f,
  0.239630718356093560f, 0.238886004499120170f, 0.238141150111664870f,
  0.237396155631906550f,
  0.236651021498106460f, 0.235905748148607370f, 0.235160336021834860f,
  0.234414785556295250f,
  0.233669097190576820f, 0.232923271363349120f, 0.232177308513361770f,
  0.231431209079445730f,
  0.230684973500512310f, 0.229938602215552260f, 0.229192095663636740f,
  0.228445454283916550f,
  0.227698678515621170f, 0.226951768798059980f, 0.226204725570620270f,
  0.225457549272768540f,
  0.224710240344049570f, 0.223962799224085520f, 0.223215226352576960f,
  0.222467522169301990f,
  0.221719687114115240f, 0.220971721626949060f, 0.220223626147812460f,
  0.219475401116790340f,
  0.218727046974044600f, 0.217978564159812290f, 0.217229953114406790f,
  0.216481214278216900f,
  0.215732348091705940f, 0.214983354995412820f, 0.214234235429951100f,
  0.213484989836008080f,
  0.212735618654345870f, 0.211986122325800410f, 0.211236501291280710f,
  0.210486755991769890f,
  0.209736886868323370f, 0.208986894362070070f, 0.208236778914211470f,
  0.207486540966020700f,
  0.206736180958843660f, 0.205985699334098050f, 0.205235096533272380f,
  0.204484372997927180f,
  0.203733529169694010f, 0.202982565490274460f, 0.202231482401441620f,
  0.201480280345037820f,
  0.200728959762976140f, 0.199977521097239290f, 0.199225964789878890f,
  0.198474291283016360f,
  0.197722501018842030f, 0.196970594439614370f, 0.196218571987660850f,
  0.195466434105377090f,
  0.194714181235225990f, 0.193961813819739010f, 0.193209332301514080f,
  0.192456737123216840f,
  0.191704028727579940f, 0.190951207557401860f, 0.190198274055548120f,
  0.189445228664950340f,
  0.188692071828605260f, 0.187938803989575850f, 0.187185425590990440f,
  0.186431937076041640f,
  0.185678338887987790f, 0.184924631470150870f, 0.184170815265917720f,
  0.183416890718739230f,
  0.182662858272129360f, 0.181908718369666160f, 0.181154471454990920f,
  0.180400117971807270f,
  0.179645658363882100f, 0.178891093075044830f, 0.178136422549186320f,
  0.177381647230260200f,
  0.176626767562280960f, 0.175871783989325040f, 0.175116696955530060f,
  0.174361506905093830f,
  0.173606214282275410f, 0.172850819531394200f, 0.172095323096829040f,
  0.171339725423019260f,
  0.170584026954463700f, 0.169828228135719880f, 0.169072329411405180f,
  0.168316331226194910f,
  0.167560234024823590f, 0.166804038252083870f, 0.166047744352825850f,
  0.165291352771957970f,
  0.164534863954446110f, 0.163778278345312690f, 0.163021596389637810f,
  0.162264818532558110f,
  0.161507945219266150f, 0.160750976895011390f, 0.159993914005098350f,
  0.159236756994887850f,
  0.158479506309796100f, 0.157722162395293690f, 0.156964725696906750f,
  0.156207196660216040f,
  0.155449575730855880f, 0.154691863354515400f, 0.153934059976937460f,
  0.153176166043917870f,
  0.152418182001306500f, 0.151660108295005400f, 0.150901945370970040f,
  0.150143693675208330f,
  0.149385353653779810f, 0.148626925752796540f, 0.147868410418422360f,
  0.147109808096871850f,
  0.146351119234411440f, 0.145592344277358450f, 0.144833483672080240f,
  0.144074537864995330f,
  0.143315507302571590f, 0.142556392431327340f, 0.141797193697830530f,
  0.141037911548697770f,
  0.140278546430595420f, 0.139519098790238600f, 0.138759569074390380f,
  0.137999957729862760f,
  0.137240265203515700f, 0.136480491942256310f, 0.135720638393040080f,
  0.134960705002868830f,
  0.134200692218792020f, 0.133440600487905820f, 0.132680430257352130f,
  0.131920181974319760f,
  0.131159856086043410f, 0.130399453039802740f, 0.129638973282923540f,
  0.128878417262776660f,
  0.128117785426777150f, 0.127357078222385570f, 0.126596296097105960f,
  0.125835439498487020f,
  0.125074508874121300f, 0.124313504671644300f, 0.123552427338735370f,
  0.122791277323116900f,
  0.122030055072553410f, 0.121268761034852550f, 0.120507395657864240f,
  0.119745959389479630f,
  0.118984452677632520f, 0.118222875970297250f, 0.117461229715489990f,
  0.116699514361267840f,
  0.115937730355727850f, 0.115175878147008180f, 0.114413958183287050f,
  0.113651970912781920f,
  0.112889916783750470f, 0.112127796244489750f, 0.111365609743335190f,
  0.110603357728661910f,
  0.109841040648882680f, 0.109078658952449240f, 0.108316213087851300f,
  0.107553703503615710f,
  0.106791130648307380f, 0.106028494970528530f, 0.105265796918917650f,
  0.104503036942150550f,
  0.103740215488939480f, 0.102977333008032250f, 0.102214389948213370f,
  0.101451386758302160f,
  0.100688323887153970f, 0.099925201783659226f, 0.099162020896742573f,
  0.098398781675363881f,
  0.097635484568517339f, 0.096872130025230527f, 0.096108718494565468f,
  0.095345250425617742f,
  0.094581726267515473f, 0.093818146469420494f, 0.093054511480527333f,
  0.092290821750062355f,
  0.091527077727284981f, 0.090763279861485704f, 0.089999428601987341f,
  0.089235524398144139f,
  0.088471567699340822f, 0.087707558954993645f, 0.086943498614549489f,
  0.086179387127484922f,
  0.085415224943307277f, 0.084651012511553700f, 0.083886750281790226f,
  0.083122438703613077f,
  0.082358078226646619f, 0.081593669300544638f, 0.080829212374989468f,
  0.080064707899690932f,
  0.079300156324387569f, 0.078535558098845590f, 0.077770913672857989f,
  0.077006223496245585f,
  0.076241488018856149f, 0.075476707690563416f, 0.074711882961268378f,
  0.073947014280897269f,
  0.073182102099402888f, 0.072417146866763538f, 0.071652149032982254f,
  0.070887109048087787f,
  0.070122027362133646f, 0.069356904425197236f, 0.068591740687380900f,
  0.067826536598810966f,
  0.067061292609636836f, 0.066296009170032283f, 0.065530686730193397f,
  0.064765325740339871f,
  0.063999926650714078f, 0.063234489911580136f, 0.062469015973224969f,
  0.061703505285957416f,
  0.060937958300107238f, 0.060172375466026218f, 0.059406757234087247f,
  0.058641104054683348f,
  0.057875416378229017f, 0.057109694655158132f, 0.056343939335925283f,
  0.055578150871004817f,
  0.054812329710889909f, 0.054046476306093640f, 0.053280591107148056f,
  0.052514674564603257f,
  0.051748727129028414f, 0.050982749251010900f, 0.050216741381155325f,
  0.049450703970084824f,
  0.048684637468439020f, 0.047918542326875327f, 0.047152418996068000f,
  0.046386267926707213f,
  0.045620089569500123f, 0.044853884375169933f, 0.044087652794454979f,
  0.043321395278109784f,
  0.042555112276904117f, 0.041788804241622082f, 0.041022471623063397f,
  0.040256114872041358f,
  0.039489734439384118f, 0.038723330775933762f, 0.037956904332545366f,
  0.037190455560088091f,
  0.036423984909444228f, 0.035657492831508264f, 0.034890979777187955f,
  0.034124446197403423f,
  0.033357892543086159f, 0.032591319265180385f, 0.031824726814640963f,
  0.031058115642434700f,
  0.030291486199539423f, 0.029524838936943035f, 0.028758174305644590f,
  0.027991492756653365f,
  0.027224794740987910f, 0.026458080709677145f, 0.025691351113759395f,
  0.024924606404281485f,
  0.024157847032300020f, 0.023391073448879338f, 0.022624286105092803f,
  0.021857485452021874f,
  0.021090671940755180f, 0.020323846022389572f, 0.019557008148029204f,
  0.018790158768784596f,
  0.018023298335773701f, 0.017256427300120978f, 0.016489546112956454f,
  0.015722655225417017f,
  0.014955755088644378f, 0.014188846153786343f, 0.013421928871995907f,
  0.012655003694430301f,
  0.011888071072252072f, 0.011121131456628141f, 0.010354185298728884f,
  0.009587233049729183f,
  0.008820275160807512f, 0.008053312083144991f, 0.007286344267926684f,
  0.006519372166339549f,
  0.005752396229573737f, 0.004985416908821652f, 0.004218434655277024f,
  0.003451449920135975f,
  0.002684463154596083f, 0.001917474809855460f, 0.001150485337113809f,
  0.000383495187571497f
};

/**   
 * @brief  Initialization function for the floating-point DCT4/IDCT4.  
 * @param[in,out] *S         points to an instance of floating-point DCT4/IDCT4 structure.  
 * @param[in]     *S_RFFT    points to an instance of floating-point RFFT/RIFFT structure.  
 * @param[in]     *S_CFFT    points to an instance of floating-point CFFT/CIFFT structure.  
 * @param[in]     N			 length of the DCT4.  
 * @param[in]     Nby2       half of the length of the DCT4.  
 * @param[in]     normalize  normalizing factor.  
 * @return        arm_status function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLenReal</code> is not a supported transform length.  
 * \par Normalizing factor:   
 * The normalizing factor is <code>sqrt(2/N)</code>, which depends on the size of transform <code>N</code>.   
 * Floating-point normalizing factors are mentioned in the table below for different DCT sizes:   
 * \image html dct4NormalizingF32Table.gif   
 */

arm_status arm_dct4_init_f32(
  arm_dct4_instance_f32 * S,
  arm_rfft_instance_f32 * S_RFFT,
  arm_cfft_radix4_instance_f32 * S_CFFT,
  uint16_t N,
  uint16_t Nby2,
  float32_t normalize)
{
  /*  Initialize the default arm status */
  arm_status status = ARM_MATH_SUCCESS;

  /* Initializing the pointer array with the weight table base addresses of different lengths */
  float32_t *twiddlePtr[3] =
    { (float32_t *) Weights_128, (float32_t *) Weights_512,
    (float32_t *) Weights_2048
  };

  /* Initializing the pointer array with the cos factor table base addresses of different lengths */
  float32_t *pCosFactor[3] =
    { (float32_t *) cos_factors_128, (float32_t *) cos_factors_512,
    (float32_t *) cos_factors_2048
  };

  /* Initialize the DCT4 length */
  S->N = N;

  /* Initialize the half of DCT4 length */
  S->Nby2 = Nby2;

  /* Initialize the DCT4 Normalizing factor */
  S->normalize = normalize;

  /* Initialize Real FFT Instance */
  S->pRfft = S_RFFT;

  /* Initialize Complex FFT Instance */
  S->pCfft = S_CFFT;

  switch (N)
  {
    /* Initialize the table modifier values */
  case 2048u:
    S->pTwiddle = twiddlePtr[2];
    S->pCosFactor = pCosFactor[2];
    break;
  case 512u:
    S->pTwiddle = twiddlePtr[1];
    S->pCosFactor = pCosFactor[1];
    break;
  case 128u:
    S->pTwiddle = twiddlePtr[0];
    S->pCosFactor = pCosFactor[0];
    break;
  default:
    status = ARM_MATH_ARGUMENT_ERROR;
  }

  /* Initialize the RFFT/RIFFT */
  arm_rfft_init_f32(S->pRfft, S->pCfft, S->N, 0u, 1u);

  /* return the status of DCT4 Init function */
  return (status);
}

/**   
   * @} end of DCT4_IDCT4 group   
   */
