/* ----------------------------------------------------------------------   
* Copyright (C) 2011 ARM Limited. All rights reserved.   
*   
* $Date:        17. August 2011  
* $Revision: 	V1.0.11  
*   
* Project: 	    CMSIS DSP Library   
* Title:	    arm_cfft_radix4_init_f32.c   
*   
* Description:	Radix-4 Decimation in Frequency Floating-point CFFT & CIFFT Initialization function   
*   
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
*  
** Version 1.0.11 2011/08/17 
*    Updated to support 4096 CFFT length.  
*
* Version 1.0.10 2011/7/15 
*    Big Endian support added and Merged M0 and M3/M4 Source code.  
*   
* Version 1.0.3 2010/11/29  
*    Re-organized the CMSIS folders and updated documentation.   
*    
* Version 1.0.2 2010/11/11   
*    Documentation updated.    
*   
* Version 1.0.1 2010/10/05    
*    Production release and review comments incorporated.   
*   
* Version 1.0.0 2010/09/20    
*    Production release and review comments incorporated.   
*   
* Version 0.0.5  2010/04/26    
* 	 incorporated review comments and updated with latest CMSIS layer   
*   
* Version 0.0.3  2010/03/10    
*    Initial version   
* -------------------------------------------------------------------- */


#include "arm_math.h"
#include "arm_common_tables.h"

/**   
 * @ingroup groupTransforms   
 */

/**   
 * @addtogroup CFFT_CIFFT   
 * @{   
 */

/*   
* @brief  Floating-point Twiddle factors Table Generation   
*/


/**   
* \par   
* Example code for Floating-point Twiddle factors Generation:   
* \par   
* <pre>for(i = 0; i< N; i++)   
* {   
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);   
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);   
* } </pre>   
* \par   
* where N = 1024	and PI = 3.14159265358979   
* \par   
* Cos and Sin values are in interleaved fashion   
*    
*/

static const float32_t twiddleCoef[4096*2] = {
			1.000000000000000000f,	0.000000000000000000f,	0.999998823451701880f,	0.001533980186284766f,	0.999995293809576190f,	0.003067956762965976f,
	0.999989411081928400f,	0.004601926120448571f,	0.999981175282601110f,	0.006135884649154475f,	0.999970586430974140f,	0.007669828739531097f,
	0.999957644551963900f,	0.009203754782059819f,	0.999942349676023910f,	0.010737659167264491f,	0.999924701839144500f,	0.012271538285719925f,
	0.999904701082852900f,	0.013805388528060391f,	0.999882347454212560f,	0.015339206284988100f,	0.999857641005823860f,	0.016872987947281710f,
	0.999830581795823400f,	0.018406729905804820f,	0.999801169887884260f,	0.019940428551514441f,	0.999769405351215280f,	0.021474080275469508f,
	0.999735288260561680f,	0.023007681468839369f,	0.999698818696204250f,	0.024541228522912288f,	0.999659996743959220f,	0.026074717829103901f,
	0.999618822495178640f,	0.027608145778965740f,	0.999575296046749220f,	0.029141508764193722f,	0.999529417501093140f,	0.030674803176636626f,
	0.999481186966166950f,	0.032208025408304586f,	0.999430604555461730f,	0.033741171851377580f,	0.999377670388002850f,	0.035274238898213947f,
	0.999322384588349540f,	0.036807222941358832f,	0.999264747286594420f,	0.038340120373552694f,	0.999204758618363890f,	0.039872927587739811f,
	0.999142418724816910f,	0.041405640977076739f,	0.999077727752645360f,	0.042938256934940820f,	0.999010685854073380f,	0.044470771854938668f,
	0.998941293186856870f,	0.046003182130914623f,	0.998869549914283560f,	0.047535484156959303f,	0.998795456205172410f,	0.049067674327418015f,
	0.998719012233872940f,	0.050599749036899282f,	0.998640218180265270f,	0.052131704680283324f,	0.998559074229759310f,	0.053663537652730520f,
	0.998475580573294770f,	0.055195244349689934f,	0.998389737407340160f,	0.056726821166907748f,	0.998301544933892890f,	0.058258264500435752f,
	0.998211003360478190f,	0.059789570746639868f,	0.998118112900149180f,	0.061320736302208578f,	0.998022873771486240f,	0.062851757564161406f,
	0.997925286198596000f,	0.064382630929857465f,	0.997825350411111640f,	0.065913352797003805f,	0.997723066644191640f,	0.067443919563664051f,
	0.997618435138519550f,	0.068974327628266746f,	0.997511456140303450f,	0.070504573389613856f,	0.997402129901275300f,	0.072034653246889332f,
	0.997290456678690210f,	0.073564563599667426f,	0.997176436735326190f,	0.075094300847921305f,	0.997060070339482960f,	0.076623861392031492f,
	0.996941357764982160f,	0.078153241632794232f,	0.996820299291165670f,	0.079682437971430126f,	0.996696895202896060f,	0.081211446809592441f,
	0.996571145790554840f,	0.082740264549375692f,	0.996443051350042630f,	0.084268887593324071f,	0.996312612182778000f,	0.085797312344439894f,
	0.996179828595696980f,	0.087325535206192059f,	0.996044700901251970f,	0.088853552582524600f,	0.995907229417411720f,	0.090381360877864983f,
	0.995767414467659820f,	0.091908956497132724f,	0.995625256380994310f,	0.093436335845747787f,	0.995480755491926940f,	0.094963495329638992f,
	0.995333912140482280f,	0.096490431355252593f,	0.995184726672196930f,	0.098017140329560604f,	0.995033199438118630f,	0.099543618660069319f,
	0.994879330794805620f,	0.101069862754827820f,	0.994723121104325700f,	0.102595869022436280f,	0.994564570734255420f,	0.104121633872054590f,
	0.994403680057679100f,	0.105647153713410620f,	0.994240449453187900f,	0.107172424956808840f,	0.994074879304879370f,	0.108697444013138720f,
	0.993906970002356060f,	0.110222207293883060f,	0.993736721940724600f,	0.111746711211126590f,	0.993564135520595300f,	0.113270952177564350f,
	0.993389211148080650f,	0.114794926606510080f,	0.993211949234794500f,	0.116318630911904750f,	0.993032350197851410f,	0.117842061508324980f,
	0.992850414459865100f,	0.119365214810991350f,	0.992666142448948020f,	0.120888087235777080f,	0.992479534598709970f,	0.122410675199216200f,
	0.992290591348257370f,	0.123932975118512160f,	0.992099313142191800f,	0.125454983411546230f,	0.991905700430609330f,	0.126976696496885870f,
	0.991709753669099530f,	0.128498110793793170f,	0.991511473318743900f,	0.130019222722233350f,	0.991310859846115440f,	0.131540028702883120f,
	0.991107913723276890f,	0.133060525157139060f,	0.990902635427780010f,	0.134580708507126170f,	0.990695025442664630f,	0.136100575175706200f,
	0.990485084256457090f,	0.137620121586486040f,	0.990272812363169110f,	0.139139344163826200f,	0.990058210262297120f,	0.140658239332849210f,
	0.989841278458820530f,	0.142176803519448030f,	0.989622017463200890f,	0.143695033150294470f,	0.989400427791380380f,	0.145212924652847460f,
	0.989176509964781010f,	0.146730474455361750f,	0.988950264510302990f,	0.148247678986896030f,	0.988721691960323780f,	0.149764534677321510f,
	0.988490792852696590f,	0.151281037957330220f,	0.988257567730749460f,	0.152797185258443440f,	0.988022017143283530f,	0.154312973013020100f,
	0.987784141644572180f,	0.155828397654265230f,	0.987543941794359230f,	0.157343455616238250f,	0.987301418157858430f,	0.158858143333861450f,
	0.987056571305750970f,	0.160372457242928280f,	0.986809401814185530f,	0.161886393780111830f,	0.986559910264775410f,	0.163399949382973230f,
	0.986308097244598670f,	0.164913120489969890f,	0.986053963346195440f,	0.166425903540464100f,	0.985797509167567480f,	0.167938294974731170f,
	0.985538735312176060f,	0.169450291233967960f,	0.985277642388941220f,	0.170961888760301220f,	0.985014231012239840f,	0.172473083996795950f,
	0.984748501801904210f,	0.173983873387463820f,	0.984480455383220930f,	0.175494253377271430f,	0.984210092386929030f,	0.177004220412148750f,
	0.983937413449218920f,	0.178513770938997510f,	0.983662419211730250f,	0.180022901405699510f,	0.983385110321551180f,	0.181531608261124970f,
	0.983105487431216290f,	0.183039887955140950f,	0.982823551198705240f,	0.184547736938619620f,	0.982539302287441240f,	0.186055151663446630f,
	0.982252741366289370f,	0.187562128582529600f,	0.981963869109555240f,	0.189068664149806190f,	0.981672686196983110f,	0.190574754820252740f,
	0.981379193313754560f,	0.192080397049892440f,	0.981083391150486710f,	0.193585587295803610f,	0.980785280403230430f,	0.195090322016128250f,
	0.980484861773469380f,	0.196594597670080220f,	0.980182135968117430f,	0.198098410717953560f,	0.979877103699517640f,	0.199601757621130970f,
	0.979569765685440520f,	0.201104634842091900f,	0.979260122649082020f,	0.202607038844421130f,	0.978948175319062200f,	0.204108966092816870f,
	0.978633924429423210f,	0.205610413053099240f,	0.978317370719627650f,	0.207111376192218560f,	0.977998514934557140f,	0.208611851978263490f,
	0.977677357824509930f,	0.210111836880469610f,	0.977353900145199960f,	0.211611327369227550f,	0.977028142657754390f,	0.213110319916091360f,
	0.976700086128711840f,	0.214608810993786760f,	0.976369731330021140f,	0.216106797076219520f,	0.976037079039039020f,	0.217604274638483640f,
	0.975702130038528570f,	0.219101240156869800f,	0.975364885116656980f,	0.220597690108873510f,	0.975025345066994120f,	0.222093620973203510f,
	0.974683510688510670f,	0.223589029229789990f,	0.974339382785575860f,	0.225083911359792830f,	0.973992962167955830f,	0.226578263845610000f,
	0.973644249650811980f,	0.228072083170885730f,	0.973293246054698250f,	0.229565365820518870f,	0.972939952205560180f,	0.231058108280671110f,
	0.972584368934732210f,	0.232550307038775240f,	0.972226497078936270f,	0.234041958583543430f,	0.971866337480279400f,	0.235533059404975490f,
	0.971503890986251780f,	0.237023605994367200f,	0.971139158449725090f,	0.238513594844318420f,	0.970772140728950350f,	0.240003022448741500f,
	0.970402838687555500f,	0.241491885302869330f,	0.970031253194543970f,	0.242980179903263870f,	0.969657385124292450f,	0.244467902747824150f,
	0.969281235356548530f,	0.245955050335794590f,	0.968902804776428870f,	0.247441619167773270f,	0.968522094274417380f,	0.248927605745720150f,
	0.968139104746362440f,	0.250413006572965220f,	0.967753837093475510f,	0.251897818154216970f,	0.967366292222328510f,	0.253382036995570160f,
	0.966976471044852070f,	0.254865659604514570f,	0.966584374478333120f,	0.256348682489942910f,	0.966190003445412500f,	0.257831102162158990f,
	0.965793358874083680f,	0.259312915132886230f,	0.965394441697689400f,	0.260794117915275510f,	0.964993252854920320f,	0.262274707023913590f,
	0.964589793289812760f,	0.263754678974831350f,	0.964184063951745830f,	0.265234030285511790f,	0.963776065795439840f,	0.266712757474898370f,
	0.963365799780954050f,	0.268190857063403180f,	0.962953266873683880f,	0.269668325572915090f,	0.962538468044359160f,	0.271145159526808010f,
	0.962121404269041580f,	0.272621355449948980f,	0.961702076529122540f,	0.274096909868706380f,	0.961280485811320640f,	0.275571819310958140f,
	0.960856633107679660f,	0.277046080306099900f,	0.960430519415565790f,	0.278519689385053060f,	0.960002145737665960f,	0.279992643080273220f,
	0.959571513081984520f,	0.281464937925757940f,	0.959138622461841890f,	0.282936570457055390f,	0.958703474895871600f,	0.284407537211271880f,
	0.958266071408017670f,	0.285877834727080620f,	0.957826413027532910f,	0.287347459544729510f,	0.957384500788975860f,	0.288816408206049480f,
	0.956940335732208820f,	0.290284677254462330f,	0.956493918902395100f,	0.291752263234989260f,	0.956045251349996410f,	0.293219162694258630f,
	0.955594334130771110f,	0.294685372180514330f,	0.955141168305770780f,	0.296150888243623790f,	0.954685754941338340f,	0.297615707435086200f,
	0.954228095109105670f,	0.299079826308040480f,	0.953768189885990330f,	0.300543241417273450f,	0.953306040354193860f,	0.302005949319228080f,
	0.952841647601198720f,	0.303467946572011320f,	0.952375012719765880f,	0.304929229735402370f,	0.951906136807932350f,	0.306389795370860920f,
	0.951435020969008340f,	0.307849640041534870f,	0.950961666311575080f,	0.309308760312268730f,	0.950486073949481700f,	0.310767152749611470f,
	0.950008245001843000f,	0.312224813921824880f,	0.949528180593036670f,	0.313681740398891520f,	0.949045881852700560f,	0.315137928752522440f,
	0.948561349915730270f,	0.316593375556165850f,	0.948074585922276230f,	0.318048077385014950f,	0.947585591017741090f,	0.319502030816015690f,
	0.947094366352777220f,	0.320955232427875210f,	0.946600913083283530f,	0.322407678801069850f,	0.946105232370403450f,	0.323859366517852850f,
	0.945607325380521280f,	0.325310292162262930f,	0.945107193285260610f,	0.326760452320131730f,	0.944604837261480260f,	0.328209843579092500f,
	0.944100258491272660f,	0.329658462528587490f,	0.943593458161960390f,	0.331106305759876430f,	0.943084437466093490f,	0.332553369866044220f,
	0.942573197601446870f,	0.333999651442009380f,	0.942059739771017310f,	0.335445147084531600f,	0.941544065183020810f,	0.336889853392220050f,
	0.941026175050889260f,	0.338333766965541130f,	0.940506070593268300f,	0.339776884406826850f,	0.939983753034014050f,	0.341219202320282360f,
	0.939459223602189920f,	0.342660717311994380f,	0.938932483532064600f,	0.344101425989938810f,	0.938403534063108060f,	0.345541324963989090f,
	0.937872376439989890f,	0.346980410845923680f,	0.937339011912574960f,	0.348418680249434560f,	0.936803441735921560f,	0.349856129790134920f,
	0.936265667170278260f,	0.351292756085567090f,	0.935725689481080370f,	0.352728555755210730f,	0.935183509938947610f,	0.354163525420490340f,
	0.934639129819680780f,	0.355597661704783850f,	0.934092550404258980f,	0.357030961233429980f,	0.933543772978836170f,	0.358463420633736540f,
	0.932992798834738960f,	0.359895036534988110f,	0.932439629268462360f,	0.361325805568454280f,	0.931884265581668150f,	0.362755724367397230f,
	0.931326709081180430f,	0.364184789567079890f,	0.930766961078983710f,	0.365612997804773850f,	0.930205022892219070f,	0.367040345719767180f,
	0.929640895843181330f,	0.368466829953372320f,	0.929074581259315860f,	0.369892447148934100f,	0.928506080473215590f,	0.371317193951837540f,
	0.927935394822617890f,	0.372741067009515760f,	0.927362525650401110f,	0.374164062971457930f,	0.926787474304581750f,	0.375586178489217220f,
	0.926210242138311380f,	0.377007410216418260f,	0.925630830509872720f,	0.378427754808765560f,	0.925049240782677580f,	0.379847208924051160f,
	0.924465474325262600f,	0.381265769222162380f,	0.923879532511286740f,	0.382683432365089780f,	0.923291416719527640f,	0.384100195016935040f,
	0.922701128333878630f,	0.385516053843918850f,	0.922108668743345180f,	0.386931005514388580f,	0.921514039342042010f,	0.388345046698826250f,
	0.920917241529189520f,	0.389758174069856410f,	0.920318276709110590f,	0.391170384302253870f,	0.919717146291227360f,	0.392581674072951470f,
	0.919113851690057770f,	0.393992040061048100f,	0.918508394325212250f,	0.395401478947816350f,	0.917900775621390500f,	0.396809987416710310f,
	0.917290997008377910f,	0.398217562153373560f,	0.916679059921042700f,	0.399624199845646790f,	0.916064965799331720f,	0.401029897183575620f,
	0.915448716088267830f,	0.402434650859418430f,	0.914830312237946200f,	0.403838457567654070f,	0.914209755703530690f,	0.405241314004989860f,
	0.913587047945250810f,	0.406643216870369030f,	0.912962190428398210f,	0.408044162864978690f,	0.912335184623322750f,	0.409444148692257590f,
	0.911706032005429880f,	0.410843171057903910f,	0.911074734055176360f,	0.412241226669882890f,	0.910441292258067250f,	0.413638312238434500f,
	0.909805708104652220f,	0.415034424476081630f,	0.909167983090522380f,	0.416429560097637150f,	0.908528118716306120f,	0.417823715820212270f,
	0.907886116487666260f,	0.419216888363223910f,	0.907241977915295820f,	0.420609074448402510f,	0.906595704514915330f,	0.422000270799799680f,
	0.905947297807268460f,	0.423390474143796050f,	0.905296759318118820f,	0.424779681209108810f,	0.904644090578246240f,	0.426167888726799620f,
	0.903989293123443340f,	0.427555093430282080f,	0.903332368494511820f,	0.428941292055329490f,	0.902673318237258830f,	0.430326481340082610f,
	0.902012143902493180f,	0.431710658025057260f,	0.901348847046022030f,	0.433093818853151960f,	0.900683429228646970f,	0.434475960569655650f,
	0.900015892016160280f,	0.435857079922255470f,	0.899346236979341570f,	0.437237173661044090f,	0.898674465693953820f,	0.438616238538527660f,
	0.898000579740739880f,	0.439994271309633260f,	0.897324580705418320f,	0.441371268731716670f,	0.896646470178680150f,	0.442747227564570020f,
	0.895966249756185220f,	0.444122144570429200f,	0.895283921038557580f,	0.445496016513981740f,	0.894599485631382700f,	0.446868840162374160f,
	0.893912945145203250f,	0.448240612285219890f,	0.893224301195515320f,	0.449611329654606540f,	0.892533555402764580f,	0.450980989045103860f,
	0.891840709392342720f,	0.452349587233770890f,	0.891145764794583180f,	0.453717121000163870f,	0.890448723244757880f,	0.455083587126343840f,
	0.889749586383072780f,	0.456448982396883920f,	0.889048355854664570f,	0.457813303598877170f,	0.888345033309596350f,	0.459176547521944090f,
	0.887639620402853930f,	0.460538710958240010f,	0.886932118794342190f,	0.461899790702462730f,	0.886222530148880640f,	0.463259783551860150f,
	0.885510856136199950f,	0.464618686306237820f,	0.884797098430937790f,	0.465976495767966180f,	0.884081258712634990f,	0.467333208741988420f,
	0.883363338665731580f,	0.468688822035827900f,	0.882643339979562790f,	0.470043332459595620f,	0.881921264348355050f,	0.471396736825997640f,
	0.881197113471222090f,	0.472749031950342790f,	0.880470889052160750f,	0.474100214650549970f,	0.879742592800047410f,	0.475450281747155870f,
	0.879012226428633530f,	0.476799230063322090f,	0.878279791656541580f,	0.478147056424843010f,	0.877545290207261350f,	0.479493757660153010f,
	0.876808723809145650f,	0.480839330600333960f,	0.876070094195406600f,	0.482183772079122720f,	0.875329403104110890f,	0.483527078932918740f,
	0.874586652278176110f,	0.484869248000791060f,	0.873841843465366860f,	0.486210276124486420f,	0.873094978418290090f,	0.487550160148436000f,
	0.872346058894391540f,	0.488888896919763170f,	0.871595086655950980f,	0.490226483288291160f,	0.870842063470078980f,	0.491562916106549900f,
	0.870086991108711460f,	0.492898192229784040f,	0.869329871348606840f,	0.494232308515959670f,	0.868570705971340900f,	0.495565261825772540f,
	0.867809496763303320f,	0.496897049022654470f,	0.867046245515692650f,	0.498227666972781870f,	0.866280954024512990f,	0.499557112545081840f,
	0.865513624090569090f,	0.500885382611240710f,	0.864744257519462380f,	0.502212474045710790f,	0.863972856121586810f,	0.503538383725717580f,
	0.863199421712124160f,	0.504863108531267590f,	0.862423956111040610f,	0.506186645345155230f,	0.861646461143081300f,	0.507508991052970870f,
	0.860866938637767310f,	0.508830142543106990f,	0.860085390429390140f,	0.510150096706766810f,	0.859301818357008470f,	0.511468850437970300f,
	0.858516224264442740f,	0.512786400633562960f,	0.857728610000272120f,	0.514102744193221660f,	0.856938977417828760f,	0.515417878019462930f,
	0.856147328375194470f,	0.516731799017649870f,	0.855353664735196030f,	0.518044504095999340f,	0.854557988365400530f,	0.519355990165589640f,
	0.853760301138111410f,	0.520666254140367160f,	0.852960604930363630f,	0.521975292937154390f,	0.852158901623919830f,	0.523283103475656430f,
	0.851355193105265200f,	0.524589682678468950f,	0.850549481265603480f,	0.525895027471084630f,	0.849741768000852550f,	0.527199134781901280f,
	0.848932055211639610f,	0.528502001542228480f,	0.848120344803297230f,	0.529803624686294610f,	0.847306638685858320f,	0.531104001151255000f,
	0.846490938774052130f,	0.532403127877197900f,	0.845673246987299070f,	0.533701001807152960f,	0.844853565249707120f,	0.534997619887097150f,
	0.844031895490066410f,	0.536292979065963180f,	0.843208239641845440f,	0.537587076295645390f,	0.842382599643185850f,	0.538879908531008420f,
	0.841554977436898440f,	0.540171472729892850f,	0.840725374970458070f,	0.541461765853123440f,	0.839893794195999520f,	0.542750784864515890f,
	0.839060237070312740f,	0.544038526730883820f,	0.838224705554838080f,	0.545324988422046460f,	0.837387201615661940f,	0.546610166910834860f,
	0.836547727223512010f,	0.547894059173100190f,	0.835706284353752600f,	0.549176662187719660f,	0.834862874986380010f,	0.550457972936604810f,
	0.834017501106018130f,	0.551737988404707340f,	0.833170164701913190f,	0.553016705580027470f,	0.832320867767929680f,	0.554294121453620000f,
	0.831469612302545240f,	0.555570233019602180f,	0.830616400308846310f,	0.556845037275160100f,	0.829761233794523050f,	0.558118531220556100f,
	0.828904114771864870f,	0.559390711859136140f,	0.828045045257755800f,	0.560661576197336030f,	0.827184027273669130f,	0.561931121244689470f,
	0.826321062845663530f,	0.563199344013834090f,	0.825456154004377550f,	0.564466241520519500f,	0.824589302785025290f,	0.565731810783613120f,
	0.823720511227391430f,	0.566996048825108680f,	0.822849781375826430f,	0.568258952670131490f,	0.821977115279241550f,	0.569520519346947140f,
	0.821102514991104650f,	0.570780745886967260f,	0.820225982569434690f,	0.572039629324757050f,	0.819347520076796900f,	0.573297166698042200f,
	0.818467129580298660f,	0.574553355047715760f,	0.817584813151583710f,	0.575808191417845340f,	0.816700572866827850f,	0.577061672855679440f,
	0.815814410806733780f,	0.578313796411655590f,	0.814926329056526620f,	0.579564559139405630f,	0.814036329705948410f,	0.580813958095764530f,
	0.813144414849253590f,	0.582061990340775440f,	0.812250586585203880f,	0.583308652937698290f,	0.811354847017063730f,	0.584553942953015330f,
	0.810457198252594770f,	0.585797857456438860f,	0.809557642404051260f,	0.587040393520917970f,	0.808656181588174980f,	0.588281548222645220f,
	0.807752817926190360f,	0.589521318641063940f,	0.806847553543799330f,	0.590759701858874160f,	0.805940390571176280f,	0.591996694962040990f,
	0.805031331142963660f,	0.593232295039799800f,	0.804120377398265810f,	0.594466499184664430f,	0.803207531480644940f,	0.595699304492433360f,
	0.802292795538115720f,	0.596930708062196500f,	0.801376171723140240f,	0.598160706996342270f,	0.800457662192622820f,	0.599389298400564540f,
	0.799537269107905010f,	0.600616479383868970f,	0.798614994634760820f,	0.601842247058580030f,	0.797690840943391160f,	0.603066598540348160f,
	0.796764810208418830f,	0.604289530948155960f,	0.795836904608883570f,	0.605511041404325550f,	0.794907126328237010f,	0.606731127034524480f,
	0.793975477554337170f,	0.607949784967773630f,	0.793041960479443640f,	0.609167012336453210f,	0.792106577300212390f,	0.610382806276309480f,
	0.791169330217690200f,	0.611597163926461910f,	0.790230221437310030f,	0.612810082429409710f,	0.789289253168885650f,	0.614021558931038380f,
	0.788346427626606340f,	0.615231590580626820f,	0.787401747029031430f,	0.616440174530853650f,	0.786455213599085770f,	0.617647307937803870f,
	0.785506829564053930f,	0.618852987960976320f,	0.784556597155575240f,	0.620057211763289100f,	0.783604518609638200f,	0.621259976511087550f,
	0.782650596166575730f,	0.622461279374149970f,	0.781694832071059390f,	0.623661117525694530f,	0.780737228572094490f,	0.624859488142386340f,
	0.779777787923014550f,	0.626056388404343520f,	0.778816512381475980f,	0.627251815495144080f,	0.777853404209453150f,	0.628445766601832710f,
	0.776888465673232440f,	0.629638238914926980f,	0.775921699043407690f,	0.630829229628424470f,	0.774953106594873930f,	0.632018735939809060f,
	0.773982690606822900f,	0.633206755050057190f,	0.773010453362736990f,	0.634393284163645490f,	0.772036397150384520f,	0.635578320488556110f,
	0.771060524261813820f,	0.636761861236284200f,	0.770082836993347900f,	0.637943903621844060f,	0.769103337645579700f,	0.639124444863775730f,
	0.768122028523365420f,	0.640303482184151670f,	0.767138911935820400f,	0.641481012808583160f,	0.766153990196312920f,	0.642657033966226860f,
	0.765167265622458960f,	0.643831542889791390f,	0.764178740536116670f,	0.645004536815543930f,	0.763188417263381270f,	0.646176012983316280f,
	0.762196298134578900f,	0.647345968636512060f,	0.761202385484261780f,	0.648514401022112440f,	0.760206681651202420f,	0.649681307390683190f,
	0.759209188978388070f,	0.650846684996380880f,	0.758209909813015280f,	0.652010531096959500f,	0.757208846506484570f,	0.653172842953776760f,
	0.756206001414394540f,	0.654333617831800440f,	0.755201376896536550f,	0.655492852999615350f,	0.754194975316889170f,	0.656650545729428940f,
	0.753186799043612520f,	0.657806693297078640f,	0.752176850449042810f,	0.658961292982037320f,	0.751165131909686480f,	0.660114342067420480f,
	0.750151645806215070f,	0.661265837839992270f,	0.749136394523459370f,	0.662415777590171780f,	0.748119380450403600f,	0.663564158612039770f,
	0.747100605980180130f,	0.664710978203344790f,	0.746080073510063780f,	0.665856233665509720f,	0.745057785441466060f,	0.666999922303637470f,
	0.744033744179929290f,	0.668142041426518450f,	0.743007952135121720f,	0.669282588346636010f,	0.741980411720831070f,	0.670421560380173090f,
	0.740951125354959110f,	0.671558954847018330f,	0.739920095459516200f,	0.672694769070772860f,	0.738887324460615110f,	0.673829000378756040f,
	0.737852814788465980f,	0.674961646102011930f,	0.736816568877369900f,	0.676092703575315920f,	0.735778589165713590f,	0.677222170137180330f,
	0.734738878095963500f,	0.678350043129861470f,	0.733697438114660370f,	0.679476319899364970f,	0.732654271672412820f,	0.680600997795453020f,
	0.731609381223892630f,	0.681724074171649710f,	0.730562769227827590f,	0.682845546385248080f,	0.729514438146997010f,	0.683965411797315400f,
	0.728464390448225200f,	0.685083667772700360f,	0.727412628602375770f,	0.686200311680038590f,	0.726359155084346010f,	0.687315340891759050f,
	0.725303972373060770f,	0.688428752784090440f,	0.724247082951467000f,	0.689540544737066830f,	0.723188489306527460f,	0.690650714134534600f,
	0.722128193929215350f,	0.691759258364157750f,	0.721066199314508110f,	0.692866174817424630f,	0.720002507961381650f,	0.693971460889654000f,
	0.718937122372804490f,	0.695075113980000880f,	0.717870045055731710f,	0.696177131491462990f,	0.716801278521099540f,	0.697277510830886520f,
	0.715730825283818590f,	0.698376249408972920f,	0.714658687862769090f,	0.699473344640283770f,	0.713584868780793640f,	0.700568793943248340f,
	0.712509370564692320f,	0.701662594740168450f,	0.711432195745216430f,	0.702754744457225300f,	0.710353346857062420f,	0.703845240524484940f,
	0.709272826438865690f,	0.704934080375904880f,	0.708190637033195400f,	0.706021261449339740f,	0.707106781186547570f,	0.707106781186547460f,
	0.706021261449339740f,	0.708190637033195290f,	0.704934080375904990f,	0.709272826438865580f,	0.703845240524484940f,	0.710353346857062310f,
	0.702754744457225300f,	0.711432195745216430f,	0.701662594740168570f,	0.712509370564692320f,	0.700568793943248450f,	0.713584868780793520f,
	0.699473344640283770f,	0.714658687862768980f,	0.698376249408972920f,	0.715730825283818590f,	0.697277510830886630f,	0.716801278521099540f,
	0.696177131491462990f,	0.717870045055731710f,	0.695075113980000880f,	0.718937122372804380f,	0.693971460889654000f,	0.720002507961381650f,
	0.692866174817424740f,	0.721066199314508110f,	0.691759258364157750f,	0.722128193929215350f,	0.690650714134534720f,	0.723188489306527350f,
	0.689540544737066940f,	0.724247082951466890f,	0.688428752784090550f,	0.725303972373060660f,	0.687315340891759160f,	0.726359155084346010f,
	0.686200311680038700f,	0.727412628602375770f,	0.685083667772700360f,	0.728464390448225200f,	0.683965411797315510f,	0.729514438146996900f,
	0.682845546385248080f,	0.730562769227827590f,	0.681724074171649820f,	0.731609381223892520f,	0.680600997795453130f,	0.732654271672412820f,
	0.679476319899365080f,	0.733697438114660260f,	0.678350043129861580f,	0.734738878095963390f,	0.677222170137180450f,	0.735778589165713480f,
	0.676092703575316030f,	0.736816568877369790f,	0.674961646102012040f,	0.737852814788465980f,	0.673829000378756150f,	0.738887324460615110f,
	0.672694769070772970f,	0.739920095459516090f,	0.671558954847018330f,	0.740951125354959110f,	0.670421560380173090f,	0.741980411720830960f,
	0.669282588346636010f,	0.743007952135121720f,	0.668142041426518560f,	0.744033744179929180f,	0.666999922303637470f,	0.745057785441465950f,
	0.665856233665509720f,	0.746080073510063780f,	0.664710978203344900f,	0.747100605980180130f,	0.663564158612039880f,	0.748119380450403490f,
	0.662415777590171780f,	0.749136394523459260f,	0.661265837839992270f,	0.750151645806214960f,	0.660114342067420480f,	0.751165131909686370f,
	0.658961292982037320f,	0.752176850449042700f,	0.657806693297078640f,	0.753186799043612410f,	0.656650545729429050f,	0.754194975316889170f,
	0.655492852999615460f,	0.755201376896536550f,	0.654333617831800550f,	0.756206001414394540f,	0.653172842953776760f,	0.757208846506484460f,
	0.652010531096959500f,	0.758209909813015280f,	0.650846684996380990f,	0.759209188978387960f,	0.649681307390683190f,	0.760206681651202420f,
	0.648514401022112550f,	0.761202385484261780f,	0.647345968636512060f,	0.762196298134578900f,	0.646176012983316390f,	0.763188417263381270f,
	0.645004536815544040f,	0.764178740536116670f,	0.643831542889791500f,	0.765167265622458960f,	0.642657033966226860f,	0.766153990196312810f,
	0.641481012808583160f,	0.767138911935820400f,	0.640303482184151670f,	0.768122028523365310f,	0.639124444863775730f,	0.769103337645579590f,
	0.637943903621844170f,	0.770082836993347900f,	0.636761861236284200f,	0.771060524261813710f,	0.635578320488556230f,	0.772036397150384410f,
	0.634393284163645490f,	0.773010453362736990f,	0.633206755050057190f,	0.773982690606822790f,	0.632018735939809060f,	0.774953106594873820f,
	0.630829229628424470f,	0.775921699043407580f,	0.629638238914927100f,	0.776888465673232440f,	0.628445766601832710f,	0.777853404209453040f,
	0.627251815495144190f,	0.778816512381475870f,	0.626056388404343520f,	0.779777787923014440f,	0.624859488142386450f,	0.780737228572094380f,
	0.623661117525694640f,	0.781694832071059390f,	0.622461279374150080f,	0.782650596166575730f,	0.621259976511087660f,	0.783604518609638200f,
	0.620057211763289210f,	0.784556597155575240f,	0.618852987960976320f,	0.785506829564053930f,	0.617647307937803980f,	0.786455213599085770f,
	0.616440174530853650f,	0.787401747029031320f,	0.615231590580626820f,	0.788346427626606230f,	0.614021558931038490f,	0.789289253168885650f,
	0.612810082429409710f,	0.790230221437310030f,	0.611597163926462020f,	0.791169330217690090f,	0.610382806276309480f,	0.792106577300212390f,
	0.609167012336453210f,	0.793041960479443640f,	0.607949784967773740f,	0.793975477554337170f,	0.606731127034524480f,	0.794907126328237010f,
	0.605511041404325550f,	0.795836904608883460f,	0.604289530948156070f,	0.796764810208418720f,	0.603066598540348280f,	0.797690840943391040f,
	0.601842247058580030f,	0.798614994634760820f,	0.600616479383868970f,	0.799537269107905010f,	0.599389298400564540f,	0.800457662192622710f,
	0.598160706996342380f,	0.801376171723140130f,	0.596930708062196500f,	0.802292795538115720f,	0.595699304492433470f,	0.803207531480644830f,
	0.594466499184664540f,	0.804120377398265700f,	0.593232295039799800f,	0.805031331142963660f,	0.591996694962040990f,	0.805940390571176280f,
	0.590759701858874280f,	0.806847553543799220f,	0.589521318641063940f,	0.807752817926190360f,	0.588281548222645330f,	0.808656181588174980f,
	0.587040393520918080f,	0.809557642404051260f,	0.585797857456438860f,	0.810457198252594770f,	0.584553942953015330f,	0.811354847017063730f,
	0.583308652937698290f,	0.812250586585203880f,	0.582061990340775550f,	0.813144414849253590f,	0.580813958095764530f,	0.814036329705948300f,
	0.579564559139405740f,	0.814926329056526620f,	0.578313796411655590f,	0.815814410806733780f,	0.577061672855679550f,	0.816700572866827850f,
	0.575808191417845340f,	0.817584813151583710f,	0.574553355047715760f,	0.818467129580298660f,	0.573297166698042320f,	0.819347520076796900f,
	0.572039629324757050f,	0.820225982569434690f,	0.570780745886967370f,	0.821102514991104650f,	0.569520519346947250f,	0.821977115279241550f,
	0.568258952670131490f,	0.822849781375826320f,	0.566996048825108680f,	0.823720511227391320f,	0.565731810783613230f,	0.824589302785025290f,
	0.564466241520519500f,	0.825456154004377440f,	0.563199344013834090f,	0.826321062845663420f,	0.561931121244689470f,	0.827184027273669020f,
	0.560661576197336030f,	0.828045045257755800f,	0.559390711859136140f,	0.828904114771864870f,	0.558118531220556100f,	0.829761233794523050f,
	0.556845037275160100f,	0.830616400308846200f,	0.555570233019602290f,	0.831469612302545240f,	0.554294121453620110f,	0.832320867767929680f,
	0.553016705580027580f,	0.833170164701913190f,	0.551737988404707450f,	0.834017501106018130f,	0.550457972936604810f,	0.834862874986380010f,
	0.549176662187719770f,	0.835706284353752600f,	0.547894059173100190f,	0.836547727223511890f,	0.546610166910834860f,	0.837387201615661940f,
	0.545324988422046460f,	0.838224705554837970f,	0.544038526730883930f,	0.839060237070312630f,	0.542750784864516000f,	0.839893794195999410f,
	0.541461765853123560f,	0.840725374970458070f,	0.540171472729892970f,	0.841554977436898330f,	0.538879908531008420f,	0.842382599643185960f,
	0.537587076295645510f,	0.843208239641845440f,	0.536292979065963180f,	0.844031895490066410f,	0.534997619887097260f,	0.844853565249707010f,
	0.533701001807152960f,	0.845673246987299070f,	0.532403127877198010f,	0.846490938774052020f,	0.531104001151255000f,	0.847306638685858320f,
	0.529803624686294830f,	0.848120344803297120f,	0.528502001542228480f,	0.848932055211639610f,	0.527199134781901390f,	0.849741768000852440f,
	0.525895027471084740f,	0.850549481265603370f,	0.524589682678468840f,	0.851355193105265200f,	0.523283103475656430f,	0.852158901623919830f,
	0.521975292937154390f,	0.852960604930363630f,	0.520666254140367270f,	0.853760301138111300f,	0.519355990165589530f,	0.854557988365400530f,
	0.518044504095999340f,	0.855353664735196030f,	0.516731799017649980f,	0.856147328375194470f,	0.515417878019463150f,	0.856938977417828650f,
	0.514102744193221660f,	0.857728610000272120f,	0.512786400633563070f,	0.858516224264442740f,	0.511468850437970520f,	0.859301818357008360f,
	0.510150096706766700f,	0.860085390429390140f,	0.508830142543106990f,	0.860866938637767310f,	0.507508991052970870f,	0.861646461143081300f,
	0.506186645345155450f,	0.862423956111040500f,	0.504863108531267480f,	0.863199421712124160f,	0.503538383725717580f,	0.863972856121586700f,
	0.502212474045710900f,	0.864744257519462380f,	0.500885382611240940f,	0.865513624090568980f,	0.499557112545081890f,	0.866280954024512990f,
	0.498227666972781870f,	0.867046245515692650f,	0.496897049022654640f,	0.867809496763303210f,	0.495565261825772490f,	0.868570705971340900f,
	0.494232308515959730f,	0.869329871348606730f,	0.492898192229784090f,	0.870086991108711350f,	0.491562916106550060f,	0.870842063470078860f,
	0.490226483288291100f,	0.871595086655951090f,	0.488888896919763230f,	0.872346058894391540f,	0.487550160148436050f,	0.873094978418290090f,
	0.486210276124486530f,	0.873841843465366750f,	0.484869248000791120f,	0.874586652278176110f,	0.483527078932918740f,	0.875329403104110780f,
	0.482183772079122830f,	0.876070094195406600f,	0.480839330600333900f,	0.876808723809145760f,	0.479493757660153010f,	0.877545290207261240f,
	0.478147056424843120f,	0.878279791656541460f,	0.476799230063322250f,	0.879012226428633410f,	0.475450281747155870f,	0.879742592800047410f,
	0.474100214650550020f,	0.880470889052160750f,	0.472749031950342900f,	0.881197113471221980f,	0.471396736825997810f,	0.881921264348354940f,
	0.470043332459595620f,	0.882643339979562790f,	0.468688822035827960f,	0.883363338665731580f,	0.467333208741988530f,	0.884081258712634990f,
	0.465976495767966130f,	0.884797098430937790f,	0.464618686306237820f,	0.885510856136199950f,	0.463259783551860260f,	0.886222530148880640f,
	0.461899790702462840f,	0.886932118794342080f,	0.460538710958240010f,	0.887639620402853930f,	0.459176547521944150f,	0.888345033309596240f,
	0.457813303598877290f,	0.889048355854664570f,	0.456448982396883860f,	0.889749586383072890f,	0.455083587126343840f,	0.890448723244757880f,
	0.453717121000163930f,	0.891145764794583180f,	0.452349587233771000f,	0.891840709392342720f,	0.450980989045103810f,	0.892533555402764690f,
	0.449611329654606600f,	0.893224301195515320f,	0.448240612285220000f,	0.893912945145203250f,	0.446868840162374330f,	0.894599485631382580f,
	0.445496016513981740f,	0.895283921038557580f,	0.444122144570429260f,	0.895966249756185110f,	0.442747227564570130f,	0.896646470178680150f,
	0.441371268731716620f,	0.897324580705418320f,	0.439994271309633260f,	0.898000579740739880f,	0.438616238538527710f,	0.898674465693953820f,
	0.437237173661044200f,	0.899346236979341460f,	0.435857079922255470f,	0.900015892016160280f,	0.434475960569655710f,	0.900683429228646860f,
	0.433093818853152010f,	0.901348847046022030f,	0.431710658025057370f,	0.902012143902493070f,	0.430326481340082610f,	0.902673318237258830f,
	0.428941292055329550f,	0.903332368494511820f,	0.427555093430282200f,	0.903989293123443340f,	0.426167888726799620f,	0.904644090578246240f,
	0.424779681209108810f,	0.905296759318118820f,	0.423390474143796100f,	0.905947297807268460f,	0.422000270799799790f,	0.906595704514915330f,
	0.420609074448402510f,	0.907241977915295930f,	0.419216888363223960f,	0.907886116487666150f,	0.417823715820212380f,	0.908528118716306120f,
	0.416429560097637320f,	0.909167983090522270f,	0.415034424476081630f,	0.909805708104652220f,	0.413638312238434560f,	0.910441292258067140f,
	0.412241226669883000f,	0.911074734055176250f,	0.410843171057903910f,	0.911706032005429880f,	0.409444148692257590f,	0.912335184623322750f,
	0.408044162864978740f,	0.912962190428398100f,	0.406643216870369140f,	0.913587047945250810f,	0.405241314004989860f,	0.914209755703530690f,
	0.403838457567654130f,	0.914830312237946090f,	0.402434650859418540f,	0.915448716088267830f,	0.401029897183575790f,	0.916064965799331610f,
	0.399624199845646790f,	0.916679059921042700f,	0.398217562153373620f,	0.917290997008377910f,	0.396809987416710420f,	0.917900775621390390f,
	0.395401478947816300f,	0.918508394325212250f,	0.393992040061048100f,	0.919113851690057770f,	0.392581674072951530f,	0.919717146291227360f,
	0.391170384302253980f,	0.920318276709110480f,	0.389758174069856410f,	0.920917241529189520f,	0.388345046698826300f,	0.921514039342041900f,
	0.386931005514388690f,	0.922108668743345070f,	0.385516053843919020f,	0.922701128333878520f,	0.384100195016935040f,	0.923291416719527640f,
	0.382683432365089840f,	0.923879532511286740f,	0.381265769222162490f,	0.924465474325262600f,	0.379847208924051110f,	0.925049240782677580f,
	0.378427754808765620f,	0.925630830509872720f,	0.377007410216418310f,	0.926210242138311270f,	0.375586178489217330f,	0.926787474304581750f,
	0.374164062971457990f,	0.927362525650401110f,	0.372741067009515810f,	0.927935394822617890f,	0.371317193951837600f,	0.928506080473215480f,
	0.369892447148934270f,	0.929074581259315750f,	0.368466829953372320f,	0.929640895843181330f,	0.367040345719767240f,	0.930205022892219070f,
	0.365612997804773960f,	0.930766961078983710f,	0.364184789567079840f,	0.931326709081180430f,	0.362755724367397230f,	0.931884265581668150f,
	0.361325805568454340f,	0.932439629268462360f,	0.359895036534988280f,	0.932992798834738850f,	0.358463420633736540f,	0.933543772978836170f,
	0.357030961233430030f,	0.934092550404258870f,	0.355597661704783960f,	0.934639129819680780f,	0.354163525420490510f,	0.935183509938947500f,
	0.352728555755210730f,	0.935725689481080370f,	0.351292756085567150f,	0.936265667170278260f,	0.349856129790135030f,	0.936803441735921560f,
	0.348418680249434510f,	0.937339011912574960f,	0.346980410845923680f,	0.937872376439989890f,	0.345541324963989150f,	0.938403534063108060f,
	0.344101425989938980f,	0.938932483532064490f,	0.342660717311994380f,	0.939459223602189920f,	0.341219202320282410f,	0.939983753034013940f,
	0.339776884406826960f,	0.940506070593268300f,	0.338333766965541290f,	0.941026175050889260f,	0.336889853392220050f,	0.941544065183020810f,
	0.335445147084531660f,	0.942059739771017310f,	0.333999651442009490f,	0.942573197601446870f,	0.332553369866044220f,	0.943084437466093490f,
	0.331106305759876430f,	0.943593458161960390f,	0.329658462528587550f,	0.944100258491272660f,	0.328209843579092660f,	0.944604837261480260f,
	0.326760452320131790f,	0.945107193285260610f,	0.325310292162262980f,	0.945607325380521280f,	0.323859366517852960f,	0.946105232370403340f,
	0.322407678801070020f,	0.946600913083283530f,	0.320955232427875210f,	0.947094366352777220f,	0.319502030816015750f,	0.947585591017741090f,
	0.318048077385015060f,	0.948074585922276230f,	0.316593375556165850f,	0.948561349915730270f,	0.315137928752522440f,	0.949045881852700560f,
	0.313681740398891570f,	0.949528180593036670f,	0.312224813921825050f,	0.950008245001843000f,	0.310767152749611470f,	0.950486073949481700f,
	0.309308760312268780f,	0.950961666311575080f,	0.307849640041534980f,	0.951435020969008340f,	0.306389795370861080f,	0.951906136807932230f,
	0.304929229735402430f,	0.952375012719765880f,	0.303467946572011370f,	0.952841647601198720f,	0.302005949319228200f,	0.953306040354193750f,
	0.300543241417273400f,	0.953768189885990330f,	0.299079826308040480f,	0.954228095109105670f,	0.297615707435086310f,	0.954685754941338340f,
	0.296150888243623960f,	0.955141168305770670f,	0.294685372180514330f,	0.955594334130771110f,	0.293219162694258680f,	0.956045251349996410f,
	0.291752263234989370f,	0.956493918902394990f,	0.290284677254462330f,	0.956940335732208940f,	0.288816408206049480f,	0.957384500788975860f,
	0.287347459544729570f,	0.957826413027532910f,	0.285877834727080730f,	0.958266071408017670f,	0.284407537211271820f,	0.958703474895871600f,
	0.282936570457055390f,	0.959138622461841890f,	0.281464937925758050f,	0.959571513081984520f,	0.279992643080273380f,	0.960002145737665850f,
	0.278519689385053060f,	0.960430519415565790f,	0.277046080306099950f,	0.960856633107679660f,	0.275571819310958250f,	0.961280485811320640f,
	0.274096909868706330f,	0.961702076529122540f,	0.272621355449948980f,	0.962121404269041580f,	0.271145159526808070f,	0.962538468044359160f,
	0.269668325572915200f,	0.962953266873683880f,	0.268190857063403180f,	0.963365799780954050f,	0.266712757474898420f,	0.963776065795439840f,
	0.265234030285511900f,	0.964184063951745720f,	0.263754678974831510f,	0.964589793289812650f,	0.262274707023913590f,	0.964993252854920320f,
	0.260794117915275570f,	0.965394441697689400f,	0.259312915132886350f,	0.965793358874083570f,	0.257831102162158930f,	0.966190003445412620f,
	0.256348682489942910f,	0.966584374478333120f,	0.254865659604514630f,	0.966976471044852070f,	0.253382036995570270f,	0.967366292222328510f,
	0.251897818154216910f,	0.967753837093475510f,	0.250413006572965280f,	0.968139104746362330f,	0.248927605745720260f,	0.968522094274417270f,
	0.247441619167773440f,	0.968902804776428870f,	0.245955050335794590f,	0.969281235356548530f,	0.244467902747824210f,	0.969657385124292450f,
	0.242980179903263980f,	0.970031253194543970f,	0.241491885302869300f,	0.970402838687555500f,	0.240003022448741500f,	0.970772140728950350f,
	0.238513594844318500f,	0.971139158449725090f,	0.237023605994367340f,	0.971503890986251780f,	0.235533059404975460f,	0.971866337480279400f,
	0.234041958583543460f,	0.972226497078936270f,	0.232550307038775330f,	0.972584368934732210f,	0.231058108280671280f,	0.972939952205560070f,
	0.229565365820518870f,	0.973293246054698250f,	0.228072083170885790f,	0.973644249650811870f,	0.226578263845610110f,	0.973992962167955830f,
	0.225083911359792780f,	0.974339382785575860f,	0.223589029229790020f,	0.974683510688510670f,	0.222093620973203590f,	0.975025345066994120f,
	0.220597690108873650f,	0.975364885116656870f,	0.219101240156869770f,	0.975702130038528570f,	0.217604274638483670f,	0.976037079039039020f,
	0.216106797076219600f,	0.976369731330021140f,	0.214608810993786920f,	0.976700086128711840f,	0.213110319916091360f,	0.977028142657754390f,
	0.211611327369227610f,	0.977353900145199960f,	0.210111836880469720f,	0.977677357824509930f,	0.208611851978263460f,	0.977998514934557140f,
	0.207111376192218560f,	0.978317370719627650f,	0.205610413053099320f,	0.978633924429423100f,	0.204108966092817010f,	0.978948175319062200f,
	0.202607038844421110f,	0.979260122649082020f,	0.201104634842091960f,	0.979569765685440520f,	0.199601757621131050f,	0.979877103699517640f,
	0.198098410717953730f,	0.980182135968117320f,	0.196594597670080220f,	0.980484861773469380f,	0.195090322016128330f,	0.980785280403230430f,
	0.193585587295803750f,	0.981083391150486590f,	0.192080397049892380f,	0.981379193313754560f,	0.190574754820252800f,	0.981672686196983110f,
	0.189068664149806280f,	0.981963869109555240f,	0.187562128582529740f,	0.982252741366289370f,	0.186055151663446630f,	0.982539302287441240f,
	0.184547736938619640f,	0.982823551198705240f,	0.183039887955141060f,	0.983105487431216290f,	0.181531608261125130f,	0.983385110321551180f,
	0.180022901405699510f,	0.983662419211730250f,	0.178513770938997590f,	0.983937413449218920f,	0.177004220412148860f,	0.984210092386929030f,
	0.175494253377271400f,	0.984480455383220930f,	0.173983873387463850f,	0.984748501801904210f,	0.172473083996796030f,	0.985014231012239840f,
	0.170961888760301360f,	0.985277642388941220f,	0.169450291233967930f,	0.985538735312176060f,	0.167938294974731230f,	0.985797509167567370f,
	0.166425903540464220f,	0.986053963346195440f,	0.164913120489970090f,	0.986308097244598670f,	0.163399949382973230f,	0.986559910264775410f,
	0.161886393780111910f,	0.986809401814185420f,	0.160372457242928400f,	0.987056571305750970f,	0.158858143333861390f,	0.987301418157858430f,
	0.157343455616238280f,	0.987543941794359230f,	0.155828397654265320f,	0.987784141644572180f,	0.154312973013020240f,	0.988022017143283530f,
	0.152797185258443410f,	0.988257567730749460f,	0.151281037957330250f,	0.988490792852696590f,	0.149764534677321620f,	0.988721691960323780f,
	0.148247678986896200f,	0.988950264510302990f,	0.146730474455361750f,	0.989176509964781010f,	0.145212924652847520f,	0.989400427791380380f,
	0.143695033150294580f,	0.989622017463200780f,	0.142176803519448000f,	0.989841278458820530f,	0.140658239332849240f,	0.990058210262297120f,
	0.139139344163826280f,	0.990272812363169110f,	0.137620121586486180f,	0.990485084256456980f,	0.136100575175706200f,	0.990695025442664630f,
	0.134580708507126220f,	0.990902635427780010f,	0.133060525157139180f,	0.991107913723276780f,	0.131540028702883280f,	0.991310859846115440f,
	0.130019222722233350f,	0.991511473318743900f,	0.128498110793793220f,	0.991709753669099530f,	0.126976696496885980f,	0.991905700430609330f,
	0.125454983411546210f,	0.992099313142191800f,	0.123932975118512200f,	0.992290591348257370f,	0.122410675199216280f,	0.992479534598709970f,
	0.120888087235777220f,	0.992666142448948020f,	0.119365214810991350f,	0.992850414459865100f,	0.117842061508325020f,	0.993032350197851410f,
	0.116318630911904880f,	0.993211949234794500f,	0.114794926606510250f,	0.993389211148080650f,	0.113270952177564360f,	0.993564135520595300f,
	0.111746711211126660f,	0.993736721940724600f,	0.110222207293883180f,	0.993906970002356060f,	0.108697444013138670f,	0.994074879304879370f,
	0.107172424956808870f,	0.994240449453187900f,	0.105647153713410700f,	0.994403680057679100f,	0.104121633872054730f,	0.994564570734255420f,
	0.102595869022436280f,	0.994723121104325700f,	0.101069862754827880f,	0.994879330794805620f,	0.099543618660069444f,	0.995033199438118630f,
	0.098017140329560770f,	0.995184726672196820f,	0.096490431355252607f,	0.995333912140482280f,	0.094963495329639061f,	0.995480755491926940f,
	0.093436335845747912f,	0.995625256380994310f,	0.091908956497132696f,	0.995767414467659820f,	0.090381360877865011f,	0.995907229417411720f,
	0.088853552582524684f,	0.996044700901251970f,	0.087325535206192226f,	0.996179828595696870f,	0.085797312344439880f,	0.996312612182778000f,
	0.084268887593324127f,	0.996443051350042630f,	0.082740264549375803f,	0.996571145790554840f,	0.081211446809592386f,	0.996696895202896060f,
	0.079682437971430126f,	0.996820299291165670f,	0.078153241632794315f,	0.996941357764982160f,	0.076623861392031617f,	0.997060070339482960f,
	0.075094300847921291f,	0.997176436735326190f,	0.073564563599667454f,	0.997290456678690210f,	0.072034653246889416f,	0.997402129901275300f,
	0.070504573389614009f,	0.997511456140303450f,	0.068974327628266732f,	0.997618435138519550f,	0.067443919563664106f,	0.997723066644191640f,
	0.065913352797003930f,	0.997825350411111640f,	0.064382630929857410f,	0.997925286198596000f,	0.062851757564161420f,	0.998022873771486240f,
	0.061320736302208648f,	0.998118112900149180f,	0.059789570746640007f,	0.998211003360478190f,	0.058258264500435732f,	0.998301544933892890f,
	0.056726821166907783f,	0.998389737407340160f,	0.055195244349690031f,	0.998475580573294770f,	0.053663537652730679f,	0.998559074229759310f,
	0.052131704680283317f,	0.998640218180265270f,	0.050599749036899337f,	0.998719012233872940f,	0.049067674327418126f,	0.998795456205172410f,
	0.047535484156959261f,	0.998869549914283560f,	0.046003182130914644f,	0.998941293186856870f,	0.044470771854938744f,	0.999010685854073380f,
	0.042938256934940959f,	0.999077727752645360f,	0.041405640977076712f,	0.999142418724816910f,	0.039872927587739845f,	0.999204758618363890f,
	0.038340120373552791f,	0.999264747286594420f,	0.036807222941358991f,	0.999322384588349540f,	0.035274238898213947f,	0.999377670388002850f,
	0.033741171851377642f,	0.999430604555461730f,	0.032208025408304704f,	0.999481186966166950f,	0.030674803176636581f,	0.999529417501093140f,
	0.029141508764193740f,	0.999575296046749220f,	0.027608145778965820f,	0.999618822495178640f,	0.026074717829104040f,	0.999659996743959220f,
	0.024541228522912264f,	0.999698818696204250f,	0.023007681468839410f,	0.999735288260561680f,	0.021474080275469605f,	0.999769405351215280f,
	0.019940428551514598f,	0.999801169887884260f,	0.018406729905804820f,	0.999830581795823400f,	0.016872987947281773f,	0.999857641005823860f,
	0.015339206284988220f,	0.999882347454212560f,	0.013805388528060349f,	0.999904701082852900f,	0.012271538285719944f,	0.999924701839144500f,
	0.010737659167264572f,	0.999942349676023910f,	0.009203754782059960f,	0.999957644551963900f,	0.007669828739531077f,	0.999970586430974140f,
	0.006135884649154515f,	0.999981175282601110f,	0.004601926120448672f,	0.999989411081928400f,	0.003067956762966138f,	0.999995293809576190f,
	0.001533980186284766f,	0.999998823451701880f,	0.000000000000000061f,	1.000000000000000000f,	-0.001533980186284644f,	0.999998823451701880f,
	-0.003067956762966016f,	0.999995293809576190f,	-0.004601926120448550f,	0.999989411081928400f,	-0.006135884649154393f,	0.999981175282601110f,
	-0.007669828739530955f,	0.999970586430974140f,	-0.009203754782059837f,	0.999957644551963900f,	-0.010737659167264449f,	0.999942349676023910f,
	-0.012271538285719823f,	0.999924701839144500f,	-0.013805388528060226f,	0.999904701082852900f,	-0.015339206284988098f,	0.999882347454212560f,
	-0.016872987947281651f,	0.999857641005823860f,	-0.018406729905804695f,	0.999830581795823400f,	-0.019940428551514476f,	0.999801169887884260f,
	-0.021474080275469484f,	0.999769405351215280f,	-0.023007681468839289f,	0.999735288260561680f,	-0.024541228522912142f,	0.999698818696204250f,
	-0.026074717829103915f,	0.999659996743959220f,	-0.027608145778965698f,	0.999618822495178640f,	-0.029141508764193618f,	0.999575296046749220f,
	-0.030674803176636459f,	0.999529417501093140f,	-0.032208025408304579f,	0.999481186966166950f,	-0.033741171851377517f,	0.999430604555461730f,
	-0.035274238898213822f,	0.999377670388002850f,	-0.036807222941358866f,	0.999322384588349540f,	-0.038340120373552666f,	0.999264747286594420f,
	-0.039872927587739727f,	0.999204758618363890f,	-0.041405640977076594f,	0.999142418724816910f,	-0.042938256934940834f,	0.999077727752645360f,
	-0.044470771854938619f,	0.999010685854073380f,	-0.046003182130914519f,	0.998941293186856870f,	-0.047535484156959136f,	0.998869549914283560f,
	-0.049067674327418008f,	0.998795456205172410f,	-0.050599749036899212f,	0.998719012233872940f,	-0.052131704680283192f,	0.998640218180265270f,
	-0.053663537652730554f,	0.998559074229759310f,	-0.055195244349689913f,	0.998475580573294770f,	-0.056726821166907658f,	0.998389737407340160f,
	-0.058258264500435607f,	0.998301544933892890f,	-0.059789570746639882f,	0.998211003360478190f,	-0.061320736302208530f,	0.998118112900149180f,
	-0.062851757564161309f,	0.998022873771486240f,	-0.064382630929857285f,	0.997925286198596000f,	-0.065913352797003805f,	0.997825350411111640f,
	-0.067443919563663982f,	0.997723066644191640f,	-0.068974327628266607f,	0.997618435138519550f,	-0.070504573389613898f,	0.997511456140303450f,
	-0.072034653246889291f,	0.997402129901275300f,	-0.073564563599667329f,	0.997290456678690210f,	-0.075094300847921167f,	0.997176436735326190f,
	-0.076623861392031506f,	0.997060070339482960f,	-0.078153241632794190f,	0.996941357764982160f,	-0.079682437971430015f,	0.996820299291165780f,
	-0.081211446809592261f,	0.996696895202896060f,	-0.082740264549375678f,	0.996571145790554840f,	-0.084268887593324002f,	0.996443051350042630f,
	-0.085797312344439755f,	0.996312612182778000f,	-0.087325535206192101f,	0.996179828595696870f,	-0.088853552582524559f,	0.996044700901251970f,
	-0.090381360877864886f,	0.995907229417411720f,	-0.091908956497132571f,	0.995767414467659820f,	-0.093436335845747787f,	0.995625256380994310f,
	-0.094963495329638950f,	0.995480755491926940f,	-0.096490431355252482f,	0.995333912140482280f,	-0.098017140329560645f,	0.995184726672196930f,
	-0.099543618660069319f,	0.995033199438118630f,	-0.101069862754827750f,	0.994879330794805620f,	-0.102595869022436160f,	0.994723121104325700f,
	-0.104121633872054600f,	0.994564570734255420f,	-0.105647153713410570f,	0.994403680057679100f,	-0.107172424956808760f,	0.994240449453187900f,
	-0.108697444013138560f,	0.994074879304879480f,	-0.110222207293883060f,	0.993906970002356060f,	-0.111746711211126550f,	0.993736721940724600f,
	-0.113270952177564240f,	0.993564135520595300f,	-0.114794926606510130f,	0.993389211148080650f,	-0.116318630911904750f,	0.993211949234794500f,
	-0.117842061508324890f,	0.993032350197851410f,	-0.119365214810991230f,	0.992850414459865100f,	-0.120888087235777100f,	0.992666142448948020f,
	-0.122410675199216150f,	0.992479534598709970f,	-0.123932975118512080f,	0.992290591348257370f,	-0.125454983411546070f,	0.992099313142191800f,
	-0.126976696496885870f,	0.991905700430609330f,	-0.128498110793793110f,	0.991709753669099530f,	-0.130019222722233240f,	0.991511473318744010f,
	-0.131540028702883140f,	0.991310859846115440f,	-0.133060525157139040f,	0.991107913723276890f,	-0.134580708507126110f,	0.990902635427780010f,
	-0.136100575175706060f,	0.990695025442664630f,	-0.137620121586486070f,	0.990485084256456980f,	-0.139139344163826170f,	0.990272812363169110f,
	-0.140658239332849130f,	0.990058210262297120f,	-0.142176803519447890f,	0.989841278458820530f,	-0.143695033150294440f,	0.989622017463200890f,
	-0.145212924652847410f,	0.989400427791380380f,	-0.146730474455361640f,	0.989176509964781010f,	-0.148247678986896090f,	0.988950264510302990f,
	-0.149764534677321510f,	0.988721691960323780f,	-0.151281037957330140f,	0.988490792852696700f,	-0.152797185258443300f,	0.988257567730749460f,
	-0.154312973013020130f,	0.988022017143283530f,	-0.155828397654265200f,	0.987784141644572180f,	-0.157343455616238160f,	0.987543941794359340f,
	-0.158858143333861280f,	0.987301418157858430f,	-0.160372457242928260f,	0.987056571305750970f,	-0.161886393780111770f,	0.986809401814185530f,
	-0.163399949382973110f,	0.986559910264775520f,	-0.164913120489969950f,	0.986308097244598670f,	-0.166425903540464100f,	0.986053963346195440f,
	-0.167938294974731090f,	0.985797509167567480f,	-0.169450291233967820f,	0.985538735312176060f,	-0.170961888760301240f,	0.985277642388941220f,
	-0.172473083996795920f,	0.985014231012239840f,	-0.173983873387463710f,	0.984748501801904210f,	-0.175494253377271260f,	0.984480455383220930f,
	-0.177004220412148750f,	0.984210092386929030f,	-0.178513770938997450f,	0.983937413449218920f,	-0.180022901405699400f,	0.983662419211730250f,
	-0.181531608261125020f,	0.983385110321551180f,	-0.183039887955140920f,	0.983105487431216290f,	-0.184547736938619530f,	0.982823551198705350f,
	-0.186055151663446490f,	0.982539302287441240f,	-0.187562128582529600f,	0.982252741366289370f,	-0.189068664149806160f,	0.981963869109555240f,
	-0.190574754820252660f,	0.981672686196983110f,	-0.192080397049892270f,	0.981379193313754560f,	-0.193585587295803610f,	0.981083391150486710f,
	-0.195090322016128190f,	0.980785280403230430f,	-0.196594597670080110f,	0.980484861773469380f,	-0.198098410717953620f,	0.980182135968117430f,
	-0.199601757621130940f,	0.979877103699517640f,	-0.201104634842091820f,	0.979569765685440520f,	-0.202607038844420970f,	0.979260122649082130f,
	-0.204108966092816900f,	0.978948175319062200f,	-0.205610413053099210f,	0.978633924429423210f,	-0.207111376192218450f,	0.978317370719627650f,
	-0.208611851978263320f,	0.977998514934557140f,	-0.210111836880469610f,	0.977677357824509930f,	-0.211611327369227500f,	0.977353900145200070f,
	-0.213110319916091250f,	0.977028142657754390f,	-0.214608810993786810f,	0.976700086128711840f,	-0.216106797076219490f,	0.976369731330021140f,
	-0.217604274638483560f,	0.976037079039039130f,	-0.219101240156869660f,	0.975702130038528570f,	-0.220597690108873530f,	0.975364885116656980f,
	-0.222093620973203480f,	0.975025345066994120f,	-0.223589029229789880f,	0.974683510688510670f,	-0.225083911359792670f,	0.974339382785575860f,
	-0.226578263845610000f,	0.973992962167955830f,	-0.228072083170885680f,	0.973644249650811980f,	-0.229565365820518760f,	0.973293246054698250f,
	-0.231058108280671140f,	0.972939952205560180f,	-0.232550307038775220f,	0.972584368934732210f,	-0.234041958583543320f,	0.972226497078936380f,
	-0.235533059404975350f,	0.971866337480279400f,	-0.237023605994367230f,	0.971503890986251780f,	-0.238513594844318390f,	0.971139158449725090f,
	-0.240003022448741390f,	0.970772140728950350f,	-0.241491885302869160f,	0.970402838687555500f,	-0.242980179903263870f,	0.970031253194543970f,
	-0.244467902747824100f,	0.969657385124292450f,	-0.245955050335794480f,	0.969281235356548530f,	-0.247441619167773320f,	0.968902804776428870f,
	-0.248927605745720120f,	0.968522094274417380f,	-0.250413006572965170f,	0.968139104746362440f,	-0.251897818154216800f,	0.967753837093475510f,
	-0.253382036995570160f,	0.967366292222328510f,	-0.254865659604514520f,	0.966976471044852070f,	-0.256348682489942800f,	0.966584374478333120f,
	-0.257831102162158820f,	0.966190003445412620f,	-0.259312915132886230f,	0.965793358874083680f,	-0.260794117915275460f,	0.965394441697689400f,
	-0.262274707023913480f,	0.964993252854920440f,	-0.263754678974831400f,	0.964589793289812760f,	-0.265234030285511790f,	0.964184063951745830f,
	-0.266712757474898310f,	0.963776065795439840f,	-0.268190857063403010f,	0.963365799780954050f,	-0.269668325572915090f,	0.962953266873683880f,
	-0.271145159526807960f,	0.962538468044359160f,	-0.272621355449948870f,	0.962121404269041580f,	-0.274096909868706220f,	0.961702076529122540f,
	-0.275571819310958140f,	0.961280485811320640f,	-0.277046080306099840f,	0.960856633107679660f,	-0.278519689385052950f,	0.960430519415565900f,
	-0.279992643080273270f,	0.960002145737665850f,	-0.281464937925757940f,	0.959571513081984520f,	-0.282936570457055280f,	0.959138622461842010f,
	-0.284407537211271710f,	0.958703474895871600f,	-0.285877834727080620f,	0.958266071408017670f,	-0.287347459544729460f,	0.957826413027532910f,
	-0.288816408206049370f,	0.957384500788975970f,	-0.290284677254462160f,	0.956940335732208940f,	-0.291752263234989260f,	0.956493918902395100f,
	-0.293219162694258570f,	0.956045251349996520f,	-0.294685372180514220f,	0.955594334130771110f,	-0.296150888243623840f,	0.955141168305770670f,
	-0.297615707435086200f,	0.954685754941338340f,	-0.299079826308040360f,	0.954228095109105670f,	-0.300543241417273290f,	0.953768189885990330f,
	-0.302005949319228080f,	0.953306040354193860f,	-0.303467946572011260f,	0.952841647601198720f,	-0.304929229735402260f,	0.952375012719765880f,
	-0.306389795370860970f,	0.951906136807932350f,	-0.307849640041534870f,	0.951435020969008340f,	-0.309308760312268620f,	0.950961666311575080f,
	-0.310767152749611360f,	0.950486073949481810f,	-0.312224813921824940f,	0.950008245001843000f,	-0.313681740398891410f,	0.949528180593036670f,
	-0.315137928752522330f,	0.949045881852700670f,	-0.316593375556165730f,	0.948561349915730380f,	-0.318048077385014950f,	0.948074585922276230f,
	-0.319502030816015640f,	0.947585591017741200f,	-0.320955232427875100f,	0.947094366352777220f,	-0.322407678801069850f,	0.946600913083283530f,
	-0.323859366517852850f,	0.946105232370403450f,	-0.325310292162262870f,	0.945607325380521390f,	-0.326760452320131620f,	0.945107193285260610f,
	-0.328209843579092550f,	0.944604837261480260f,	-0.329658462528587440f,	0.944100258491272660f,	-0.331106305759876320f,	0.943593458161960390f,
	-0.332553369866044060f,	0.943084437466093490f,	-0.333999651442009380f,	0.942573197601446870f,	-0.335445147084531550f,	0.942059739771017420f,
	-0.336889853392219940f,	0.941544065183020810f,	-0.338333766965541180f,	0.941026175050889260f,	-0.339776884406826850f,	0.940506070593268300f,
	-0.341219202320282300f,	0.939983753034014050f,	-0.342660717311994270f,	0.939459223602189920f,	-0.344101425989938870f,	0.938932483532064490f,
	-0.345541324963989040f,	0.938403534063108170f,	-0.346980410845923570f,	0.937872376439989890f,	-0.348418680249434400f,	0.937339011912574960f,
	-0.349856129790134920f,	0.936803441735921560f,	-0.351292756085567040f,	0.936265667170278260f,	-0.352728555755210620f,	0.935725689481080370f,
	-0.354163525420490400f,	0.935183509938947610f,	-0.355597661704783850f,	0.934639129819680780f,	-0.357030961233429920f,	0.934092550404258980f,
	-0.358463420633736430f,	0.933543772978836280f,	-0.359895036534988170f,	0.932992798834738850f,	-0.361325805568454230f,	0.932439629268462360f,
	-0.362755724367397110f,	0.931884265581668150f,	-0.364184789567079730f,	0.931326709081180540f,	-0.365612997804773850f,	0.930766961078983710f,
	-0.367040345719767120f,	0.930205022892219070f,	-0.368466829953372210f,	0.929640895843181330f,	-0.369892447148934160f,	0.929074581259315750f,
	-0.371317193951837490f,	0.928506080473215590f,	-0.372741067009515700f,	0.927935394822617890f,	-0.374164062971457880f,	0.927362525650401110f,
	-0.375586178489217220f,	0.926787474304581750f,	-0.377007410216418200f,	0.926210242138311380f,	-0.378427754808765450f,	0.925630830509872830f,
	-0.379847208924050990f,	0.925049240782677700f,	-0.381265769222162380f,	0.924465474325262600f,	-0.382683432365089730f,	0.923879532511286740f,
	-0.384100195016934930f,	0.923291416719527750f,	-0.385516053843918900f,	0.922701128333878520f,	-0.386931005514388580f,	0.922108668743345180f,
	-0.388345046698826190f,	0.921514039342042010f,	-0.389758174069856300f,	0.920917241529189520f,	-0.391170384302253870f,	0.920318276709110590f,
	-0.392581674072951410f,	0.919717146291227360f,	-0.393992040061047990f,	0.919113851690057770f,	-0.395401478947816190f,	0.918508394325212250f,
	-0.396809987416710310f,	0.917900775621390500f,	-0.398217562153373510f,	0.917290997008378020f,	-0.399624199845646680f,	0.916679059921042700f,
	-0.401029897183575680f,	0.916064965799331720f,	-0.402434650859418430f,	0.915448716088267830f,	-0.403838457567654020f,	0.914830312237946200f,
	-0.405241314004989750f,	0.914209755703530690f,	-0.406643216870369030f,	0.913587047945250810f,	-0.408044162864978630f,	0.912962190428398210f,
	-0.409444148692257480f,	0.912335184623322860f,	-0.410843171057903800f,	0.911706032005429880f,	-0.412241226669882890f,	0.911074734055176360f,
	-0.413638312238434450f,	0.910441292258067250f,	-0.415034424476081520f,	0.909805708104652330f,	-0.416429560097636990f,	0.909167983090522490f,
	-0.417823715820212270f,	0.908528118716306120f,	-0.419216888363224070f,	0.907886116487666150f,	-0.420609074448402400f,	0.907241977915295930f,
	-0.422000270799799680f,	0.906595704514915330f,	-0.423390474143795770f,	0.905947297807268570f,	-0.424779681209108690f,	0.905296759318118820f,
	-0.426167888726799670f,	0.904644090578246130f,	-0.427555093430281860f,	0.903989293123443450f,	-0.428941292055329440f,	0.903332368494511820f,
	-0.430326481340082720f,	0.902673318237258830f,	-0.431710658025057090f,	0.902012143902493290f,	-0.433093818853151900f,	0.901348847046022030f,
	-0.434475960569655820f,	0.900683429228646860f,	-0.435857079922255360f,	0.900015892016160280f,	-0.437237173661044090f,	0.899346236979341570f,
	-0.438616238538527380f,	0.898674465693953930f,	-0.439994271309633140f,	0.898000579740739880f,	-0.441371268731716730f,	0.897324580705418320f,
	-0.442747227564569800f,	0.896646470178680270f,	-0.444122144570429140f,	0.895966249756185220f,	-0.445496016513981800f,	0.895283921038557470f,
	-0.446868840162373990f,	0.894599485631382810f,	-0.448240612285219890f,	0.893912945145203250f,	-0.449611329654606710f,	0.893224301195515210f,
	-0.450980989045103700f,	0.892533555402764690f,	-0.452349587233770890f,	0.891840709392342720f,	-0.453717121000163590f,	0.891145764794583410f,
	-0.455083587126343720f,	0.890448723244757990f,	-0.456448982396883970f,	0.889749586383072780f,	-0.457813303598877010f,	0.889048355854664680f,
	-0.459176547521944030f,	0.888345033309596350f,	-0.460538710958240060f,	0.887639620402853930f,	-0.461899790702462560f,	0.886932118794342310f,
	-0.463259783551860150f,	0.886222530148880640f,	-0.464618686306237930f,	0.885510856136199840f,	-0.465976495767966010f,	0.884797098430937900f,
	-0.467333208741988420f,	0.884081258712634990f,	-0.468688822035827680f,	0.883363338665731690f,	-0.470043332459595510f,	0.882643339979562900f,
	-0.471396736825997700f,	0.881921264348355050f,	-0.472749031950342570f,	0.881197113471222200f,	-0.474100214650549910f,	0.880470889052160870f,
	-0.475450281747155980f,	0.879742592800047410f,	-0.476799230063321920f,	0.879012226428633530f,	-0.478147056424843010f,	0.878279791656541580f,
	-0.479493757660153120f,	0.877545290207261240f,	-0.480839330600333790f,	0.876808723809145760f,	-0.482183772079122720f,	0.876070094195406600f,
	-0.483527078932918460f,	0.875329403104111000f,	-0.484869248000791010f,	0.874586652278176220f,	-0.486210276124486420f,	0.873841843465366860f,
	-0.487550160148435720f,	0.873094978418290200f,	-0.488888896919763120f,	0.872346058894391540f,	-0.490226483288291210f,	0.871595086655950980f,
	-0.491562916106549790f,	0.870842063470078980f,	-0.492898192229783980f,	0.870086991108711460f,	-0.494232308515959840f,	0.869329871348606730f,
	-0.495565261825772370f,	0.868570705971341010f,	-0.496897049022654520f,	0.867809496763303210f,	-0.498227666972781590f,	0.867046245515692760f,
	-0.499557112545081780f,	0.866280954024513110f,	-0.500885382611240830f,	0.865513624090569090f,	-0.502212474045710570f,	0.864744257519462490f,
	-0.503538383725717460f,	0.863972856121586810f,	-0.504863108531267590f,	0.863199421712124160f,	-0.506186645345155120f,	0.862423956111040610f,
	-0.507508991052970760f,	0.861646461143081300f,	-0.508830142543107100f,	0.860866938637767200f,	-0.510150096706766590f,	0.860085390429390250f,
	-0.511468850437970410f,	0.859301818357008360f,	-0.512786400633562730f,	0.858516224264442960f,	-0.514102744193221660f,	0.857728610000272120f,
	-0.515417878019463040f,	0.856938977417828760f,	-0.516731799017649650f,	0.856147328375194580f,	-0.518044504095999230f,	0.855353664735196030f,
	-0.519355990165589640f,	0.854557988365400530f,	-0.520666254140366940f,	0.853760301138111520f,	-0.521975292937154280f,	0.852960604930363740f,
	-0.523283103475656540f,	0.852158901623919720f,	-0.524589682678468730f,	0.851355193105265200f,	-0.525895027471084630f,	0.850549481265603480f,
	-0.527199134781901060f,	0.849741768000852660f,	-0.528502001542228370f,	0.848932055211639720f,	-0.529803624686294720f,	0.848120344803297230f,
	-0.531104001151254780f,	0.847306638685858540f,	-0.532403127877197900f,	0.846490938774052130f,	-0.533701001807152960f,	0.845673246987299070f,
	-0.534997619887097040f,	0.844853565249707230f,	-0.536292979065963070f,	0.844031895490066410f,	-0.537587076295645620f,	0.843208239641845440f,
	-0.538879908531008310f,	0.842382599643185960f,	-0.540171472729892850f,	0.841554977436898440f,	-0.541461765853123220f,	0.840725374970458180f,
	-0.542750784864515780f,	0.839893794195999630f,	-0.544038526730883930f,	0.839060237070312630f,	-0.545324988422046240f,	0.838224705554838190f,
	-0.546610166910834860f,	0.837387201615661940f,	-0.547894059173100190f,	0.836547727223512010f,	-0.549176662187719540f,	0.835706284353752720f,
	-0.550457972936604700f,	0.834862874986380120f,	-0.551737988404707450f,	0.834017501106018020f,	-0.553016705580027360f,	0.833170164701913300f,
	-0.554294121453620110f,	0.832320867767929680f,	-0.555570233019601960f,	0.831469612302545460f,	-0.556845037275159990f,	0.830616400308846310f,
	-0.558118531220556100f,	0.829761233794523050f,	-0.559390711859135800f,	0.828904114771865100f,	-0.560661576197335920f,	0.828045045257755800f,
	-0.561931121244689470f,	0.827184027273669130f,	-0.563199344013833980f,	0.826321062845663650f,	-0.564466241520519390f,	0.825456154004377550f,
	-0.565731810783613230f,	0.824589302785025180f,	-0.566996048825108460f,	0.823720511227391540f,	-0.568258952670131490f,	0.822849781375826320f,
	-0.569520519346947250f,	0.821977115279241440f,	-0.570780745886967140f,	0.821102514991104760f,	-0.572039629324757050f,	0.820225982569434690f,
	-0.573297166698041980f,	0.819347520076797120f,	-0.574553355047715760f,	0.818467129580298770f,	-0.575808191417845340f,	0.817584813151583710f,
	-0.577061672855679330f,	0.816700572866827960f,	-0.578313796411655480f,	0.815814410806733780f,	-0.579564559139405850f,	0.814926329056526510f,
	-0.580813958095764420f,	0.814036329705948520f,	-0.582061990340775550f,	0.813144414849253590f,	-0.583308652937698400f,	0.812250586585203880f,
	-0.584553942953015220f,	0.811354847017063840f,	-0.585797857456438860f,	0.810457198252594770f,	-0.587040393520917750f,	0.809557642404051480f,
	-0.588281548222645220f,	0.808656181588175090f,	-0.589521318641063940f,	0.807752817926190360f,	-0.590759701858874050f,	0.806847553543799450f,
	-0.591996694962040880f,	0.805940390571176390f,	-0.593232295039799910f,	0.805031331142963550f,	-0.594466499184664320f,	0.804120377398265810f,
	-0.595699304492433360f,	0.803207531480644940f,	-0.596930708062196610f,	0.802292795538115610f,	-0.598160706996342160f,	0.801376171723140350f,
	-0.599389298400564540f,	0.800457662192622820f,	-0.600616479383868750f,	0.799537269107905240f,	-0.601842247058579920f,	0.798614994634760930f,
	-0.603066598540348280f,	0.797690840943391040f,	-0.604289530948155850f,	0.796764810208418940f,	-0.605511041404325430f,	0.795836904608883570f,
	-0.606731127034524590f,	0.794907126328236900f,	-0.607949784967773520f,	0.793975477554337280f,	-0.609167012336453210f,	0.793041960479443640f,
	-0.610382806276309590f,	0.792106577300212280f,	-0.611597163926461800f,	0.791169330217690310f,	-0.612810082429409710f,	0.790230221437310030f,
	-0.614021558931038160f,	0.789289253168885870f,	-0.615231590580626710f,	0.788346427626606340f,	-0.616440174530853650f,	0.787401747029031320f,
	-0.617647307937803760f,	0.786455213599085880f,	-0.618852987960976210f,	0.785506829564054040f,	-0.620057211763289210f,	0.784556597155575130f,
	-0.621259976511087440f,	0.783604518609638310f,	-0.622461279374149970f,	0.782650596166575730f,	-0.623661117525694640f,	0.781694832071059280f,
	-0.624859488142386230f,	0.780737228572094600f,	-0.626056388404343520f,	0.779777787923014440f,	-0.627251815495143860f,	0.778816512381476090f,
	-0.628445766601832600f,	0.777853404209453150f,	-0.629638238914927100f,	0.776888465673232440f,	-0.630829229628424360f,	0.775921699043407800f,
	-0.632018735939808950f,	0.774953106594873930f,	-0.633206755050057300f,	0.773982690606822790f,	-0.634393284163645380f,	0.773010453362737100f,
	-0.635578320488556110f,	0.772036397150384520f,	-0.636761861236284310f,	0.771060524261813710f,	-0.637943903621843940f,	0.770082836993348010f,
	-0.639124444863775730f,	0.769103337645579590f,	-0.640303482184151450f,	0.768122028523365530f,	-0.641481012808583050f,	0.767138911935820510f,
	-0.642657033966226860f,	0.766153990196312920f,	-0.643831542889791280f,	0.765167265622459070f,	-0.645004536815543930f,	0.764178740536116790f,
	-0.646176012983316390f,	0.763188417263381270f,	-0.647345968636511950f,	0.762196298134579010f,	-0.648514401022112440f,	0.761202385484261890f,
	-0.649681307390683300f,	0.760206681651202310f,	-0.650846684996380760f,	0.759209188978388180f,	-0.652010531096959500f,	0.758209909813015280f,
	-0.653172842953776530f,	0.757208846506484680f,	-0.654333617831800440f,	0.756206001414394540f,	-0.655492852999615460f,	0.755201376896536550f,
	-0.656650545729428830f,	0.754194975316889280f,	-0.657806693297078640f,	0.753186799043612520f,	-0.658961292982037430f,	0.752176850449042700f,
	-0.660114342067420370f,	0.751165131909686590f,	-0.661265837839992150f,	0.750151645806215070f,	-0.662415777590171890f,	0.749136394523459260f,
	-0.663564158612039660f,	0.748119380450403710f,	-0.664710978203344900f,	0.747100605980180130f,	-0.665856233665509500f,	0.746080073510064000f,
	-0.666999922303637360f,	0.745057785441466060f,	-0.668142041426518560f,	0.744033744179929290f,	-0.669282588346635900f,	0.743007952135121830f,
	-0.670421560380173090f,	0.741980411720831070f,	-0.671558954847018440f,	0.740951125354958990f,	-0.672694769070772750f,	0.739920095459516310f,
	-0.673829000378756040f,	0.738887324460615220f,	-0.674961646102012150f,	0.737852814788465870f,	-0.676092703575315810f,	0.736816568877370020f,
	-0.677222170137180450f,	0.735778589165713480f,	-0.678350043129861250f,	0.734738878095963610f,	-0.679476319899364970f,	0.733697438114660370f,
	-0.680600997795453020f,	0.732654271672412820f,	-0.681724074171649600f,	0.731609381223892740f,	-0.682845546385247970f,	0.730562769227827590f,
	-0.683965411797315510f,	0.729514438146997010f,	-0.685083667772700240f,	0.728464390448225310f,	-0.686200311680038590f,	0.727412628602375770f,
	-0.687315340891759160f,	0.726359155084345900f,	-0.688428752784090330f,	0.725303972373060880f,	-0.689540544737066940f,	0.724247082951466890f,
	-0.690650714134534380f,	0.723188489306527570f,	-0.691759258364157640f,	0.722128193929215460f,	-0.692866174817424740f,	0.721066199314508110f,
	-0.693971460889653780f,	0.720002507961381770f,	-0.695075113980000770f,	0.718937122372804490f,	-0.696177131491462990f,	0.717870045055731710f,
	-0.697277510830886400f,	0.716801278521099650f,	-0.698376249408972800f,	0.715730825283818710f,	-0.699473344640283880f,	0.714658687862768980f,
	-0.700568793943248220f,	0.713584868780793750f,	-0.701662594740168450f,	0.712509370564692320f,	-0.702754744457225080f,	0.711432195745216660f,
	-0.703845240524484830f,	0.710353346857062420f,	-0.704934080375904880f,	0.709272826438865580f,	-0.706021261449339630f,	0.708190637033195510f,
	-0.707106781186547460f,	0.707106781186547570f,	-0.708190637033195400f,	0.706021261449339740f,	-0.709272826438865470f,	0.704934080375905100f,
	-0.710353346857062310f,	0.703845240524485050f,	-0.711432195745216540f,	0.702754744457225190f,	-0.712509370564692210f,	0.701662594740168680f,
	-0.713584868780793640f,	0.700568793943248340f,	-0.714658687862768870f,	0.699473344640283990f,	-0.715730825283818590f,	0.698376249408972920f,
	-0.716801278521099540f,	0.697277510830886520f,	-0.717870045055731600f,	0.696177131491463100f,	-0.718937122372804380f,	0.695075113980000990f,
	-0.720002507961381650f,	0.693971460889654000f,	-0.721066199314507990f,	0.692866174817424850f,	-0.722128193929215230f,	0.691759258364157860f,
	-0.723188489306527460f,	0.690650714134534600f,	-0.724247082951466780f,	0.689540544737067050f,	-0.725303972373060770f,	0.688428752784090440f,
	-0.726359155084345790f,	0.687315340891759270f,	-0.727412628602375650f,	0.686200311680038700f,	-0.728464390448225200f,	0.685083667772700360f,
	-0.729514438146996790f,	0.683965411797315630f,	-0.730562769227827480f,	0.682845546385248190f,	-0.731609381223892630f,	0.681724074171649710f,
	-0.732654271672412700f,	0.680600997795453240f,	-0.733697438114660260f,	0.679476319899365080f,	-0.734738878095963500f,	0.678350043129861360f,
	-0.735778589165713370f,	0.677222170137180560f,	-0.736816568877369900f,	0.676092703575315920f,	-0.737852814788465760f,	0.674961646102012260f,
	-0.738887324460615000f,	0.673829000378756150f,	-0.739920095459516200f,	0.672694769070772860f,	-0.740951125354958880f,	0.671558954847018550f,
	-0.741980411720830960f,	0.670421560380173200f,	-0.743007952135121720f,	0.669282588346636010f,	-0.744033744179929070f,	0.668142041426518670f,
	-0.745057785441465950f,	0.666999922303637580f,	-0.746080073510063890f,	0.665856233665509610f,	-0.747100605980180020f,	0.664710978203345020f,
	-0.748119380450403600f,	0.663564158612039770f,	-0.749136394523459150f,	0.662415777590172010f,	-0.750151645806214960f,	0.661265837839992380f,
	-0.751165131909686480f,	0.660114342067420480f,	-0.752176850449042480f,	0.658961292982037540f,	-0.753186799043612410f,	0.657806693297078750f,
	-0.754194975316889170f,	0.656650545729429050f,	-0.755201376896536440f,	0.655492852999615570f,	-0.756206001414394420f,	0.654333617831800550f,
	-0.757208846506484570f,	0.653172842953776640f,	-0.758209909813015170f,	0.652010531096959720f,	-0.759209188978388070f,	0.650846684996380990f,
	-0.760206681651202200f,	0.649681307390683420f,	-0.761202385484261670f,	0.648514401022112550f,	-0.762196298134578900f,	0.647345968636512060f,
	-0.763188417263381050f,	0.646176012983316620f,	-0.764178740536116670f,	0.645004536815544040f,	-0.765167265622458960f,	0.643831542889791390f,
	-0.766153990196312700f,	0.642657033966227090f,	-0.767138911935820290f,	0.641481012808583160f,	-0.768122028523365420f,	0.640303482184151560f,
	-0.769103337645579480f,	0.639124444863775840f,	-0.770082836993347900f,	0.637943903621844060f,	-0.771060524261813600f,	0.636761861236284420f,
	-0.772036397150384410f,	0.635578320488556230f,	-0.773010453362736990f,	0.634393284163645490f,	-0.773982690606822680f,	0.633206755050057410f,
	-0.774953106594873820f,	0.632018735939809060f,	-0.775921699043407690f,	0.630829229628424470f,	-0.776888465673232330f,	0.629638238914927210f,
	-0.777853404209453040f,	0.628445766601832710f,	-0.778816512381475980f,	0.627251815495144080f,	-0.779777787923014330f,	0.626056388404343630f,
	-0.780737228572094490f,	0.624859488142386340f,	-0.781694832071059160f,	0.623661117525694860f,	-0.782650596166575620f,	0.622461279374150080f,
	-0.783604518609638200f,	0.621259976511087550f,	-0.784556597155575020f,	0.620057211763289430f,	-0.785506829564053930f,	0.618852987960976430f,
	-0.786455213599085770f,	0.617647307937803870f,	-0.787401747029031210f,	0.616440174530853760f,	-0.788346427626606230f,	0.615231590580626930f,
	-0.789289253168885760f,	0.614021558931038380f,	-0.790230221437309920f,	0.612810082429409820f,	-0.791169330217690200f,	0.611597163926461910f,
	-0.792106577300212170f,	0.610382806276309700f,	-0.793041960479443530f,	0.609167012336453320f,	-0.793975477554337170f,	0.607949784967773630f,
	-0.794907126328236790f,	0.606731127034524700f,	-0.795836904608883460f,	0.605511041404325660f,	-0.796764810208418830f,	0.604289530948155960f,
	-0.797690840943390930f,	0.603066598540348390f,	-0.798614994634760820f,	0.601842247058580140f,	-0.799537269107905120f,	0.600616479383868860f,
	-0.800457662192622710f,	0.599389298400564650f,	-0.801376171723140240f,	0.598160706996342380f,	-0.802292795538115500f,	0.596930708062196720f,
	-0.803207531480644830f,	0.595699304492433470f,	-0.804120377398265700f,	0.594466499184664430f,	-0.805031331142963440f,	0.593232295039800020f,
	-0.805940390571176280f,	0.591996694962040990f,	-0.806847553543799330f,	0.590759701858874160f,	-0.807752817926190250f,	0.589521318641064050f,
	-0.808656181588174980f,	0.588281548222645330f,	-0.809557642404051370f,	0.587040393520917970f,	-0.810457198252594660f,	0.585797857456438980f,
	-0.811354847017063730f,	0.584553942953015330f,	-0.812250586585203770f,	0.583308652937698510f,	-0.813144414849253480f,	0.582061990340775660f,
	-0.814036329705948410f,	0.580813958095764530f,	-0.814926329056526400f,	0.579564559139405970f,	-0.815814410806733670f,	0.578313796411655700f,
	-0.816700572866827850f,	0.577061672855679440f,	-0.817584813151583600f,	0.575808191417845450f,	-0.818467129580298660f,	0.574553355047715870f,
	-0.819347520076797010f,	0.573297166698042090f,	-0.820225982569434580f,	0.572039629324757270f,	-0.821102514991104650f,	0.570780745886967260f,
	-0.821977115279241330f,	0.569520519346947470f,	-0.822849781375826210f,	0.568258952670131710f,	-0.823720511227391430f,	0.566996048825108680f,
	-0.824589302785025070f,	0.565731810783613450f,	-0.825456154004377440f,	0.564466241520519500f,	-0.826321062845663530f,	0.563199344013834090f,
	-0.827184027273669020f,	0.561931121244689580f,	-0.828045045257755690f,	0.560661576197336140f,	-0.828904114771864990f,	0.559390711859136030f,
	-0.829761233794522930f,	0.558118531220556320f,	-0.830616400308846310f,	0.556845037275160100f,	-0.831469612302545350f,	0.555570233019602180f,
	-0.832320867767929570f,	0.554294121453620230f,	-0.833170164701913190f,	0.553016705580027580f,	-0.834017501106018020f,	0.551737988404707670f,
	-0.834862874986380010f,	0.550457972936604920f,	-0.835706284353752600f,	0.549176662187719660f,	-0.836547727223511890f,	0.547894059173100410f,
	-0.837387201615661820f,	0.546610166910834970f,	-0.838224705554838080f,	0.545324988422046350f,	-0.839060237070312630f,	0.544038526730884040f,
	-0.839893794195999520f,	0.542750784864515890f,	-0.840725374970458070f,	0.541461765853123330f,	-0.841554977436898330f,	0.540171472729892970f,
	-0.842382599643185850f,	0.538879908531008420f,	-0.843208239641845330f,	0.537587076295645730f,	-0.844031895490066410f,	0.536292979065963290f,
	-0.844853565249707120f,	0.534997619887097150f,	-0.845673246987298950f,	0.533701001807153190f,	-0.846490938774052020f,	0.532403127877198010f,
	-0.847306638685858430f,	0.531104001151254890f,	-0.848120344803297120f,	0.529803624686294830f,	-0.848932055211639610f,	0.528502001542228480f,
	-0.849741768000852550f,	0.527199134781901280f,	-0.850549481265603370f,	0.525895027471084850f,	-0.851355193105265200f,	0.524589682678468950f,
	-0.852158901623919610f,	0.523283103475656650f,	-0.852960604930363630f,	0.521975292937154500f,	-0.853760301138111410f,	0.520666254140367160f,
	-0.854557988365400420f,	0.519355990165589750f,	-0.855353664735195920f,	0.518044504095999450f,	-0.856147328375194470f,	0.516731799017649760f,
	-0.856938977417828650f,	0.515417878019463150f,	-0.857728610000272010f,	0.514102744193221770f,	-0.858516224264442850f,	0.512786400633562960f,
	-0.859301818357008360f,	0.511468850437970520f,	-0.860085390429390140f,	0.510150096706766810f,	-0.860866938637767090f,	0.508830142543107320f,
	-0.861646461143081300f,	0.507508991052970980f,	-0.862423956111040500f,	0.506186645345155230f,	-0.863199421712124050f,	0.504863108531267700f,
	-0.863972856121586700f,	0.503538383725717690f,	-0.864744257519462380f,	0.502212474045710680f,	-0.865513624090568980f,	0.500885382611240940f,
	-0.866280954024512990f,	0.499557112545081950f,	-0.867046245515692760f,	0.498227666972781760f,	-0.867809496763303210f,	0.496897049022654690f,
	-0.868570705971340900f,	0.495565261825772540f,	-0.869329871348606620f,	0.494232308515960010f,	-0.870086991108711350f,	0.492898192229784150f,
	-0.870842063470078980f,	0.491562916106549900f,	-0.871595086655950870f,	0.490226483288291380f,	-0.872346058894391430f,	0.488888896919763280f,
	-0.873094978418290090f,	0.487550160148435880f,	-0.873841843465366750f,	0.486210276124486580f,	-0.874586652278176110f,	0.484869248000791120f,
	-0.875329403104110890f,	0.483527078932918630f,	-0.876070094195406490f,	0.482183772079122890f,	-0.876808723809145650f,	0.480839330600333960f,
	-0.877545290207261130f,	0.479493757660153290f,	-0.878279791656541460f,	0.478147056424843180f,	-0.879012226428633530f,	0.476799230063322090f,
	-0.879742592800047300f,	0.475450281747156090f,	-0.880470889052160750f,	0.474100214650550080f,	-0.881197113471222090f,	0.472749031950342740f,
	-0.881921264348354940f,	0.471396736825997860f,	-0.882643339979562790f,	0.470043332459595680f,	-0.883363338665731690f,	0.468688822035827850f,
	-0.884081258712634880f,	0.467333208741988580f,	-0.884797098430937790f,	0.465976495767966180f,	-0.885510856136199840f,	0.464618686306238090f,
	-0.886222530148880530f,	0.463259783551860320f,	-0.886932118794342190f,	0.461899790702462730f,	-0.887639620402853820f,	0.460538710958240230f,
	-0.888345033309596240f,	0.459176547521944200f,	-0.889048355854664570f,	0.457813303598877170f,	-0.889749586383072670f,	0.456448982396884140f,
	-0.890448723244757880f,	0.455083587126343890f,	-0.891145764794583290f,	0.453717121000163760f,	-0.891840709392342610f,	0.452349587233771060f,
	-0.892533555402764580f,	0.450980989045103860f,	-0.893224301195515210f,	0.449611329654606870f,	-0.893912945145203140f,	0.448240612285220050f,
	-0.894599485631382700f,	0.446868840162374160f,	-0.895283921038557360f,	0.445496016513981960f,	-0.895966249756185110f,	0.444122144570429310f,
	-0.896646470178680270f,	0.442747227564569970f,	-0.897324580705418210f,	0.441371268731716890f,	-0.898000579740739770f,	0.439994271309633310f,
	-0.898674465693953930f,	0.438616238538527550f,	-0.899346236979341460f,	0.437237173661044250f,	-0.900015892016160170f,	0.435857079922255530f,
	-0.900683429228646750f,	0.434475960569655980f,	-0.901348847046021920f,	0.433093818853152070f,	-0.902012143902493180f,	0.431710658025057260f,
	-0.902673318237258710f,	0.430326481340082890f,	-0.903332368494511820f,	0.428941292055329600f,	-0.903989293123443340f,	0.427555093430282030f,
	-0.904644090578246130f,	0.426167888726799840f,	-0.905296759318118700f,	0.424779681209108860f,	-0.905947297807268460f,	0.423390474143795940f,
	-0.906595704514915330f,	0.422000270799799850f,	-0.907241977915295820f,	0.420609074448402560f,	-0.907886116487666040f,	0.419216888363224240f,
	-0.908528118716306120f,	0.417823715820212440f,	-0.909167983090522380f,	0.416429560097637150f,	-0.909805708104652110f,	0.415034424476081850f,
	-0.910441292258067140f,	0.413638312238434610f,	-0.911074734055176360f,	0.412241226669882830f,	-0.911706032005429770f,	0.410843171057904130f,
	-0.912335184623322750f,	0.409444148692257650f,	-0.912962190428398210f,	0.408044162864978580f,	-0.913587047945250700f,	0.406643216870369200f,
	-0.914209755703530690f,	0.405241314004989920f,	-0.914830312237945980f,	0.403838457567654410f,	-0.915448716088267720f,	0.402434650859418600f,
	-0.916064965799331720f,	0.401029897183575620f,	-0.916679059921042590f,	0.399624199845647070f,	-0.917290997008377910f,	0.398217562153373670f,
	-0.917900775621390500f,	0.396809987416710250f,	-0.918508394325212140f,	0.395401478947816520f,	-0.919113851690057770f,	0.393992040061048150f,
	-0.919717146291227360f,	0.392581674072951410f,	-0.920318276709110480f,	0.391170384302254040f,	-0.920917241529189410f,	0.389758174069856470f,
	-0.921514039342041790f,	0.388345046698826580f,	-0.922108668743345070f,	0.386931005514388750f,	-0.922701128333878630f,	0.385516053843918850f,
	-0.923291416719527520f,	0.384100195016935320f,	-0.923879532511286740f,	0.382683432365089890f,	-0.924465474325262600f,	0.381265769222162320f,
	-0.925049240782677470f,	0.379847208924051380f,	-0.925630830509872720f,	0.378427754808765670f,	-0.926210242138311380f,	0.377007410216418150f,
	-0.926787474304581750f,	0.375586178489217380f,	-0.927362525650401110f,	0.374164062971458040f,	-0.927935394822617780f,	0.372741067009516090f,
	-0.928506080473215480f,	0.371317193951837710f,	-0.929074581259315750f,	0.369892447148934100f,	-0.929640895843181210f,	0.368466829953372600f,
	-0.930205022892219070f,	0.367040345719767290f,	-0.930766961078983710f,	0.365612997804773800f,	-0.931326709081180320f,	0.364184789567080110f,
	-0.931884265581668040f,	0.362755724367397280f,	-0.932439629268462470f,	0.361325805568454170f,	-0.932992798834738850f,	0.359895036534988330f,
	-0.933543772978836170f,	0.358463420633736600f,	-0.934092550404258760f,	0.357030961233430310f,	-0.934639129819680670f,	0.355597661704784020f,
	-0.935183509938947610f,	0.354163525420490400f,	-0.935725689481080260f,	0.352728555755210950f,	-0.936265667170278260f,	0.351292756085567200f,
	-0.936803441735921670f,	0.349856129790134860f,	-0.937339011912574850f,	0.348418680249434790f,	-0.937872376439989770f,	0.346980410845923740f,
	-0.938403534063108170f,	0.345541324963988980f,	-0.938932483532064490f,	0.344101425989939040f,	-0.939459223602189920f,	0.342660717311994430f,
	-0.939983753034013820f,	0.341219202320282690f,	-0.940506070593268300f,	0.339776884406827020f,	-0.941026175050889260f,	0.338333766965541180f,
	-0.941544065183020700f,	0.336889853392220330f,	-0.942059739771017310f,	0.335445147084531710f,	-0.942573197601446870f,	0.333999651442009380f,
	-0.943084437466093380f,	0.332553369866044450f,	-0.943593458161960390f,	0.331106305759876480f,	-0.944100258491272660f,	0.329658462528587440f,
	-0.944604837261480150f,	0.328209843579092720f,	-0.945107193285260610f,	0.326760452320131840f,	-0.945607325380521170f,	0.325310292162263260f,
	-0.946105232370403340f,	0.323859366517853020f,	-0.946600913083283530f,	0.322407678801069850f,	-0.947094366352777110f,	0.320955232427875490f,
	-0.947585591017741090f,	0.319502030816015800f,	-0.948074585922276230f,	0.318048077385014890f,	-0.948561349915730270f,	0.316593375556166070f,
	-0.949045881852700560f,	0.315137928752522500f,	-0.949528180593036670f,	0.313681740398891410f,	-0.950008245001843000f,	0.312224813921825110f,
	-0.950486073949481700f,	0.310767152749611530f,	-0.950961666311574970f,	0.309308760312269000f,	-0.951435020969008340f,	0.307849640041535030f,
	-0.951906136807932350f,	0.306389795370860920f,	-0.952375012719765770f,	0.304929229735402650f,	-0.952841647601198600f,	0.303467946572011430f,
	-0.953306040354193860f,	0.302005949319228030f,	-0.953768189885990210f,	0.300543241417273680f,	-0.954228095109105560f,	0.299079826308040530f,
	-0.954685754941338340f,	0.297615707435086140f,	-0.955141168305770670f,	0.296150888243624010f,	-0.955594334130771110f,	0.294685372180514380f,
	-0.956045251349996290f,	0.293219162694258960f,	-0.956493918902394990f,	0.291752263234989430f,	-0.956940335732208820f,	0.290284677254462390f,
	-0.957384500788975860f,	0.288816408206049760f,	-0.957826413027532910f,	0.287347459544729620f,	-0.958266071408017670f,	0.285877834727080560f,
	-0.958703474895871490f,	0.284407537211272100f,	-0.959138622461841890f,	0.282936570457055450f,	-0.959571513081984520f,	0.281464937925757890f,
	-0.960002145737665850f,	0.279992643080273440f,	-0.960430519415565790f,	0.278519689385053170f,	-0.960856633107679550f,	0.277046080306100230f,
	-0.961280485811320640f,	0.275571819310958310f,	-0.961702076529122540f,	0.274096909868706380f,	-0.962121404269041470f,	0.272621355449949250f,
	-0.962538468044359160f,	0.271145159526808120f,	-0.962953266873683880f,	0.269668325572915090f,	-0.963365799780953940f,	0.268190857063403400f,
	-0.963776065795439840f,	0.266712757474898480f,	-0.964184063951745830f,	0.265234030285511730f,	-0.964589793289812650f,	0.263754678974831570f,
	-0.964993252854920320f,	0.262274707023913700f,	-0.965394441697689290f,	0.260794117915275850f,	-0.965793358874083570f,	0.259312915132886400f,
	-0.966190003445412500f,	0.257831102162158990f,	-0.966584374478333010f,	0.256348682489943190f,	-0.966976471044852070f,	0.254865659604514680f,
	-0.967366292222328510f,	0.253382036995570100f,	-0.967753837093475400f,	0.251897818154217190f,	-0.968139104746362330f,	0.250413006572965340f,
	-0.968522094274417380f,	0.248927605745720090f,	-0.968902804776428870f,	0.247441619167773490f,	-0.969281235356548420f,	0.245955050335794650f,
	-0.969657385124292340f,	0.244467902747824480f,	-0.970031253194543970f,	0.242980179903264070f,	-0.970402838687555500f,	0.241491885302869360f,
	-0.970772140728950240f,	0.240003022448741780f,	-0.971139158449725090f,	0.238513594844318550f,	-0.971503890986251780f,	0.237023605994367170f,
	-0.971866337480279290f,	0.235533059404975740f,	-0.972226497078936270f,	0.234041958583543510f,	-0.972584368934732210f,	0.232550307038775160f,
	-0.972939952205560070f,	0.231058108280671330f,	-0.973293246054698250f,	0.229565365820518920f,	-0.973644249650811870f,	0.228072083170886060f,
	-0.973992962167955830f,	0.226578263845610170f,	-0.974339382785575860f,	0.225083911359792830f,	-0.974683510688510670f,	0.223589029229790290f,
	-0.975025345066994120f,	0.222093620973203650f,	-0.975364885116656980f,	0.220597690108873510f,	-0.975702130038528460f,	0.219101240156870050f,
	-0.976037079039039020f,	0.217604274638483720f,	-0.976369731330021140f,	0.216106797076219440f,	-0.976700086128711730f,	0.214608810993786980f,
	-0.977028142657754390f,	0.213110319916091420f,	-0.977353900145199960f,	0.211611327369227890f,	-0.977677357824509930f,	0.210111836880469800f,
	-0.977998514934557140f,	0.208611851978263510f,	-0.978317370719627540f,	0.207111376192218840f,	-0.978633924429423100f,	0.205610413053099380f,
	-0.978948175319062200f,	0.204108966092816840f,	-0.979260122649082020f,	0.202607038844421380f,	-0.979569765685440520f,	0.201104634842092010f,
	-0.979877103699517640f,	0.199601757621130920f,	-0.980182135968117320f,	0.198098410717953810f,	-0.980484861773469380f,	0.196594597670080280f,
	-0.980785280403230430f,	0.195090322016128610f,	-0.981083391150486590f,	0.193585587295803800f,	-0.981379193313754560f,	0.192080397049892470f,
	-0.981672686196983110f,	0.190574754820253070f,	-0.981963869109555240f,	0.189068664149806360f,	-0.982252741366289370f,	0.187562128582529570f,
	-0.982539302287441240f,	0.186055151663446910f,	-0.982823551198705240f,	0.184547736938619700f,	-0.983105487431216290f,	0.183039887955140900f,
	-0.983385110321551180f,	0.181531608261125220f,	-0.983662419211730250f,	0.180022901405699570f,	-0.983937413449218920f,	0.178513770938997420f,
	-0.984210092386929030f,	0.177004220412148940f,	-0.984480455383220930f,	0.175494253377271450f,	-0.984748501801904210f,	0.173983873387464130f,
	-0.985014231012239840f,	0.172473083996796120f,	-0.985277642388941220f,	0.170961888760301220f,	-0.985538735312176060f,	0.169450291233968210f,
	-0.985797509167567370f,	0.167938294974731280f,	-0.986053963346195440f,	0.166425903540464050f,	-0.986308097244598560f,	0.164913120489970140f,
	-0.986559910264775410f,	0.163399949382973280f,	-0.986809401814185530f,	0.161886393780111740f,	-0.987056571305750970f,	0.160372457242928450f,
	-0.987301418157858430f,	0.158858143333861470f,	-0.987543941794359230f,	0.157343455616238550f,	-0.987784141644572180f,	0.155828397654265370f,
	-0.988022017143283530f,	0.154312973013020080f,	-0.988257567730749460f,	0.152797185258443690f,	-0.988490792852696590f,	0.151281037957330310f,
	-0.988721691960323780f,	0.149764534677321450f,	-0.988950264510302990f,	0.148247678986896250f,	-0.989176509964781010f,	0.146730474455361800f,
	-0.989400427791380380f,	0.145212924652847350f,	-0.989622017463200780f,	0.143695033150294640f,	-0.989841278458820530f,	0.142176803519448090f,
	-0.990058210262297010f,	0.140658239332849540f,	-0.990272812363169110f,	0.139139344163826340f,	-0.990485084256457090f,	0.137620121586486040f,
	-0.990695025442664630f,	0.136100575175706480f,	-0.990902635427780010f,	0.134580708507126280f,	-0.991107913723276890f,	0.133060525157139010f,
	-0.991310859846115440f,	0.131540028702883340f,	-0.991511473318743900f,	0.130019222722233430f,	-0.991709753669099530f,	0.128498110793793090f,
	-0.991905700430609330f,	0.126976696496886060f,	-0.992099313142191800f,	0.125454983411546260f,	-0.992290591348257260f,	0.123932975118512480f,
	-0.992479534598709970f,	0.122410675199216350f,	-0.992666142448948020f,	0.120888087235777060f,	-0.992850414459865100f,	0.119365214810991630f,
	-0.993032350197851410f,	0.117842061508325090f,	-0.993211949234794500f,	0.116318630911904710f,	-0.993389211148080650f,	0.114794926606510310f,
	-0.993564135520595300f,	0.113270952177564420f,	-0.993736721940724600f,	0.111746711211126500f,	-0.993906970002356060f,	0.110222207293883240f,
	-0.994074879304879370f,	0.108697444013138740f,	-0.994240449453187900f,	0.107172424956809160f,	-0.994403680057679100f,	0.105647153713410750f,
	-0.994564570734255420f,	0.104121633872054570f,	-0.994723121104325700f,	0.102595869022436560f,	-0.994879330794805620f,	0.101069862754827930f,
	-0.995033199438118630f,	0.099543618660069277f,	-0.995184726672196820f,	0.098017140329560826f,	-0.995333912140482280f,	0.096490431355252662f,
	-0.995480755491926940f,	0.094963495329638908f,	-0.995625256380994310f,	0.093436335845747967f,	-0.995767414467659820f,	0.091908956497132752f,
	-0.995907229417411720f,	0.090381360877865288f,	-0.996044700901251970f,	0.088853552582524753f,	-0.996179828595696980f,	0.087325535206192059f,
	-0.996312612182778000f,	0.085797312344440158f,	-0.996443051350042630f,	0.084268887593324182f,	-0.996571145790554840f,	0.082740264549375636f,
	-0.996696895202896060f,	0.081211446809592663f,	-0.996820299291165670f,	0.079682437971430195f,	-0.996941357764982160f,	0.078153241632794149f,
	-0.997060070339482960f,	0.076623861392031686f,	-0.997176436735326080f,	0.075094300847921347f,	-0.997290456678690210f,	0.073564563599667732f,
	-0.997402129901275300f,	0.072034653246889471f,	-0.997511456140303450f,	0.070504573389613856f,	-0.997618435138519550f,	0.068974327628267024f,
	-0.997723066644191640f,	0.067443919563664176f,	-0.997825350411111640f,	0.065913352797003763f,	-0.997925286198596000f,	0.064382630929857701f,
	-0.998022873771486240f,	0.062851757564161490f,	-0.998118112900149180f,	0.061320736302208488f,	-0.998211003360478190f,	0.059789570746640069f,
	-0.998301544933892890f,	0.058258264500435794f,	-0.998389737407340160f,	0.056726821166908067f,	-0.998475580573294770f,	0.055195244349690094f,
	-0.998559074229759310f,	0.053663537652730520f,	-0.998640218180265160f,	0.052131704680283594f,	-0.998719012233872940f,	0.050599749036899393f,
	-0.998795456205172410f,	0.049067674327417966f,	-0.998869549914283560f,	0.047535484156959538f,	-0.998941293186856870f,	0.046003182130914706f,
	-0.999010685854073380f,	0.044470771854938584f,	-0.999077727752645360f,	0.042938256934941021f,	-0.999142418724816910f,	0.041405640977076774f,
	-0.999204758618363890f,	0.039872927587740130f,	-0.999264747286594420f,	0.038340120373552854f,	-0.999322384588349540f,	0.036807222941358832f,
	-0.999377670388002850f,	0.035274238898214232f,	-0.999430604555461730f,	0.033741171851377705f,	-0.999481186966166950f,	0.032208025408304544f,
	-0.999529417501093140f,	0.030674803176636865f,	-0.999575296046749220f,	0.029141508764193802f,	-0.999618822495178640f,	0.027608145778965660f,
	-0.999659996743959220f,	0.026074717829104099f,	-0.999698818696204250f,	0.024541228522912326f,	-0.999735288260561680f,	0.023007681468839695f,
	-0.999769405351215280f,	0.021474080275469667f,	-0.999801169887884260f,	0.019940428551514438f,	-0.999830581795823400f,	0.018406729905805101f,
	-0.999857641005823860f,	0.016872987947281835f,	-0.999882347454212560f,	0.015339206284988060f,	-0.999904701082852790f,	0.013805388528060632f,
	-0.999924701839144500f,	0.012271538285720007f,	-0.999942349676023910f,	0.010737659167264411f,	-0.999957644551963900f,	0.009203754782060021f,
	-0.999970586430974140f,	0.007669828739531138f,	-0.999981175282601110f,	0.006135884649154799f,	-0.999989411081928400f,	0.004601926120448733f,
	-0.999995293809576190f,	0.003067956762965977f,	-0.999998823451701880f,	0.001533980186285049f,	-1.000000000000000000f,	0.000000000000000122f,
	-0.999998823451701880f,	-0.001533980186284804f,	-0.999995293809576190f,	-0.003067956762965732f,	-0.999989411081928400f,	-0.004601926120448488f,
	-0.999981175282601110f,	-0.006135884649154554f,	-0.999970586430974140f,	-0.007669828739530893f,	-0.999957644551963900f,	-0.009203754782059776f,
	-0.999942349676023910f,	-0.010737659167264166f,	-0.999924701839144500f,	-0.012271538285719762f,	-0.999904701082852900f,	-0.013805388528060387f,
	-0.999882347454212560f,	-0.015339206284987816f,	-0.999857641005823860f,	-0.016872987947281589f,	-0.999830581795823400f,	-0.018406729905804858f,
	-0.999801169887884260f,	-0.019940428551514192f,	-0.999769405351215280f,	-0.021474080275469421f,	-0.999735288260561680f,	-0.023007681468839448f,
	-0.999698818696204250f,	-0.024541228522912080f,	-0.999659996743959220f,	-0.026074717829103856f,	-0.999618822495178640f,	-0.027608145778965414f,
	-0.999575296046749220f,	-0.029141508764193556f,	-0.999529417501093140f,	-0.030674803176636619f,	-0.999481186966166950f,	-0.032208025408304294f,
	-0.999430604555461730f,	-0.033741171851377455f,	-0.999377670388002850f,	-0.035274238898213982f,	-0.999322384588349540f,	-0.036807222941358582f,
	-0.999264747286594420f,	-0.038340120373552611f,	-0.999204758618363890f,	-0.039872927587739887f,	-0.999142418724816910f,	-0.041405640977076531f,
	-0.999077727752645360f,	-0.042938256934940779f,	-0.999010685854073380f,	-0.044470771854938335f,	-0.998941293186856870f,	-0.046003182130914456f,
	-0.998869549914283560f,	-0.047535484156959296f,	-0.998795456205172410f,	-0.049067674327417724f,	-0.998719012233872940f,	-0.050599749036899150f,
	-0.998640218180265270f,	-0.052131704680283351f,	-0.998559074229759310f,	-0.053663537652730277f,	-0.998475580573294770f,	-0.055195244349689851f,
	-0.998389737407340160f,	-0.056726821166907818f,	-0.998301544933892890f,	-0.058258264500435551f,	-0.998211003360478190f,	-0.059789570746639827f,
	-0.998118112900149180f,	-0.061320736302208245f,	-0.998022873771486240f,	-0.062851757564161240f,	-0.997925286198596000f,	-0.064382630929857451f,
	-0.997825350411111640f,	-0.065913352797003527f,	-0.997723066644191640f,	-0.067443919563663926f,	-0.997618435138519550f,	-0.068974327628266774f,
	-0.997511456140303450f,	-0.070504573389613606f,	-0.997402129901275300f,	-0.072034653246889235f,	-0.997290456678690210f,	-0.073564563599667496f,
	-0.997176436735326190f,	-0.075094300847921097f,	-0.997060070339482960f,	-0.076623861392031437f,	-0.996941357764982160f,	-0.078153241632793899f,
	-0.996820299291165780f,	-0.079682437971429945f,	-0.996696895202896060f,	-0.081211446809592427f,	-0.996571145790554840f,	-0.082740264549375400f,
	-0.996443051350042630f,	-0.084268887593323932f,	-0.996312612182778000f,	-0.085797312344439922f,	-0.996179828595696980f,	-0.087325535206191809f,
	-0.996044700901251970f,	-0.088853552582524503f,	-0.995907229417411720f,	-0.090381360877865052f,	-0.995767414467659820f,	-0.091908956497132516f,
	-0.995625256380994310f,	-0.093436335845747731f,	-0.995480755491926940f,	-0.094963495329638659f,	-0.995333912140482280f,	-0.096490431355252412f,
	-0.995184726672196930f,	-0.098017140329560590f,	-0.995033199438118630f,	-0.099543618660069041f,	-0.994879330794805620f,	-0.101069862754827680f,
	-0.994723121104325700f,	-0.102595869022436310f,	-0.994564570734255530f,	-0.104121633872054320f,	-0.994403680057679100f,	-0.105647153713410520f,
	-0.994240449453187900f,	-0.107172424956808910f,	-0.994074879304879480f,	-0.108697444013138490f,	-0.993906970002356060f,	-0.110222207293883000f,
	-0.993736721940724710f,	-0.111746711211126250f,	-0.993564135520595300f,	-0.113270952177564170f,	-0.993389211148080650f,	-0.114794926606510070f,
	-0.993211949234794610f,	-0.116318630911904470f,	-0.993032350197851410f,	-0.117842061508324840f,	-0.992850414459865100f,	-0.119365214810991380f,
	-0.992666142448948020f,	-0.120888087235776820f,	-0.992479534598709970f,	-0.122410675199216100f,	-0.992290591348257370f,	-0.123932975118512230f,
	-0.992099313142191800f,	-0.125454983411546010f,	-0.991905700430609330f,	-0.126976696496885810f,	-0.991709753669099530f,	-0.128498110793792840f,
	-0.991511473318744010f,	-0.130019222722233180f,	-0.991310859846115440f,	-0.131540028702883090f,	-0.991107913723276890f,	-0.133060525157138760f,
	-0.990902635427780010f,	-0.134580708507126060f,	-0.990695025442664630f,	-0.136100575175706230f,	-0.990485084256457090f,	-0.137620121586485790f,
	-0.990272812363169110f,	-0.139139344163826120f,	-0.990058210262297120f,	-0.140658239332849290f,	-0.989841278458820530f,	-0.142176803519447840f,
	-0.989622017463200890f,	-0.143695033150294390f,	-0.989400427791380380f,	-0.145212924652847130f,	-0.989176509964781010f,	-0.146730474455361580f,
	-0.988950264510302990f,	-0.148247678986896030f,	-0.988721691960323780f,	-0.149764534677321200f,	-0.988490792852696700f,	-0.151281037957330080f,
	-0.988257567730749460f,	-0.152797185258443440f,	-0.988022017143283640f,	-0.154312973013019830f,	-0.987784141644572180f,	-0.155828397654265120f,
	-0.987543941794359230f,	-0.157343455616238300f,	-0.987301418157858430f,	-0.158858143333861220f,	-0.987056571305750970f,	-0.160372457242928200f,
	-0.986809401814185530f,	-0.161886393780111490f,	-0.986559910264775520f,	-0.163399949382973060f,	-0.986308097244598670f,	-0.164913120489969890f,
	-0.986053963346195440f,	-0.166425903540463830f,	-0.985797509167567480f,	-0.167938294974731030f,	-0.985538735312176060f,	-0.169450291233967990f,
	-0.985277642388941330f,	-0.170961888760300970f,	-0.985014231012239840f,	-0.172473083996795870f,	-0.984748501801904210f,	-0.173983873387463880f,
	-0.984480455383220930f,	-0.175494253377271200f,	-0.984210092386929140f,	-0.177004220412148690f,	-0.983937413449218920f,	-0.178513770938997170f,
	-0.983662419211730250f,	-0.180022901405699350f,	-0.983385110321551180f,	-0.181531608261124970f,	-0.983105487431216400f,	-0.183039887955140650f,
	-0.982823551198705350f,	-0.184547736938619480f,	-0.982539302287441240f,	-0.186055151663446660f,	-0.982252741366289480f,	-0.187562128582529320f,
	-0.981963869109555240f,	-0.189068664149806110f,	-0.981672686196983110f,	-0.190574754820252820f,	-0.981379193313754670f,	-0.192080397049892220f,
	-0.981083391150486710f,	-0.193585587295803550f,	-0.980785280403230430f,	-0.195090322016128360f,	-0.980484861773469380f,	-0.196594597670080030f,
	-0.980182135968117430f,	-0.198098410717953560f,	-0.979877103699517750f,	-0.199601757621130670f,	-0.979569765685440520f,	-0.201104634842091760f,
	-0.979260122649082020f,	-0.202607038844421130f,	-0.978948175319062200f,	-0.204108966092816620f,	-0.978633924429423210f,	-0.205610413053099160f,
	-0.978317370719627650f,	-0.207111376192218590f,	-0.977998514934557140f,	-0.208611851978263260f,	-0.977677357824510040f,	-0.210111836880469550f,
	-0.977353900145199960f,	-0.211611327369227660f,	-0.977028142657754390f,	-0.213110319916091200f,	-0.976700086128711840f,	-0.214608810993786730f,
	-0.976369731330021250f,	-0.216106797076219210f,	-0.976037079039039130f,	-0.217604274638483470f,	-0.975702130038528570f,	-0.219101240156869800f,
	-0.975364885116656980f,	-0.220597690108873260f,	-0.975025345066994120f,	-0.222093620973203430f,	-0.974683510688510670f,	-0.223589029229790040f,
	-0.974339382785575860f,	-0.225083911359792610f,	-0.973992962167955940f,	-0.226578263845609920f,	-0.973644249650811870f,	-0.228072083170885810f,
	-0.973293246054698250f,	-0.229565365820518700f,	-0.972939952205560180f,	-0.231058108280671080f,	-0.972584368934732320f,	-0.232550307038774940f,
	-0.972226497078936380f,	-0.234041958583543260f,	-0.971866337480279400f,	-0.235533059404975510f,	-0.971503890986251890f,	-0.237023605994366950f,
	-0.971139158449725200f,	-0.238513594844318330f,	-0.970772140728950240f,	-0.240003022448741530f,	-0.970402838687555500f,	-0.241491885302869110f,
	-0.970031253194543970f,	-0.242980179903263820f,	-0.969657385124292450f,	-0.244467902747824260f,	-0.969281235356548530f,	-0.245955050335794430f,
	-0.968902804776428870f,	-0.247441619167773270f,	-0.968522094274417380f,	-0.248927605745719870f,	-0.968139104746362440f,	-0.250413006572965110f,
	-0.967753837093475510f,	-0.251897818154216970f,	-0.967366292222328620f,	-0.253382036995569880f,	-0.966976471044852180f,	-0.254865659604514460f,
	-0.966584374478333120f,	-0.256348682489942910f,	-0.966190003445412620f,	-0.257831102162158770f,	-0.965793358874083680f,	-0.259312915132886180f,
	-0.965394441697689400f,	-0.260794117915275630f,	-0.964993252854920440f,	-0.262274707023913420f,	-0.964589793289812760f,	-0.263754678974831350f,
	-0.964184063951745830f,	-0.265234030285511510f,	-0.963776065795439950f,	-0.266712757474898250f,	-0.963365799780954050f,	-0.268190857063403180f,
	-0.962953266873683990f,	-0.269668325572914810f,	-0.962538468044359160f,	-0.271145159526807900f,	-0.962121404269041580f,	-0.272621355449949030f,
	-0.961702076529122540f,	-0.274096909868706160f,	-0.961280485811320640f,	-0.275571819310958090f,	-0.960856633107679550f,	-0.277046080306100010f,
	-0.960430519415565900f,	-0.278519689385052890f,	-0.960002145737665960f,	-0.279992643080273220f,	-0.959571513081984630f,	-0.281464937925757660f,
	-0.959138622461842010f,	-0.282936570457055170f,	-0.958703474895871600f,	-0.284407537211271820f,	-0.958266071408017780f,	-0.285877834727080340f,
	-0.957826413027532910f,	-0.287347459544729400f,	-0.957384500788975860f,	-0.288816408206049540f,	-0.956940335732208940f,	-0.290284677254462110f,
	-0.956493918902395100f,	-0.291752263234989210f,	-0.956045251349996410f,	-0.293219162694258740f,	-0.955594334130771110f,	-0.294685372180514160f,
	-0.955141168305770780f,	-0.296150888243623790f,	-0.954685754941338450f,	-0.297615707435085920f,	-0.954228095109105670f,	-0.299079826308040310f,
	-0.953768189885990330f,	-0.300543241417273450f,	-0.953306040354193970f,	-0.302005949319227810f,	-0.952841647601198720f,	-0.303467946572011200f,
	-0.952375012719765880f,	-0.304929229735402430f,	-0.951906136807932350f,	-0.306389795370860700f,	-0.951435020969008450f,	-0.307849640041534810f,
	-0.950961666311575080f,	-0.309308760312268780f,	-0.950486073949481810f,	-0.310767152749611310f,	-0.950008245001843000f,	-0.312224813921824880f,
	-0.949528180593036790f,	-0.313681740398891180f,	-0.949045881852700670f,	-0.315137928752522220f,	-0.948561349915730270f,	-0.316593375556165850f,
	-0.948074585922276340f,	-0.318048077385014670f,	-0.947585591017741200f,	-0.319502030816015580f,	-0.947094366352777220f,	-0.320955232427875270f,
	-0.946600913083283650f,	-0.322407678801069630f,	-0.946105232370403450f,	-0.323859366517852800f,	-0.945607325380521280f,	-0.325310292162262980f,
	-0.945107193285260610f,	-0.326760452320131570f,	-0.944604837261480260f,	-0.328209843579092500f,	-0.944100258491272770f,	-0.329658462528587210f,
	-0.943593458161960390f,	-0.331106305759876260f,	-0.943084437466093490f,	-0.332553369866044220f,	-0.942573197601446980f,	-0.333999651442009100f,
	-0.942059739771017420f,	-0.335445147084531490f,	-0.941544065183020810f,	-0.336889853392220110f,	-0.941026175050889370f,	-0.338333766965540910f,
	-0.940506070593268410f,	-0.339776884406826800f,	-0.939983753034013940f,	-0.341219202320282470f,	-0.939459223602190030f,	-0.342660717311994210f,
	-0.938932483532064600f,	-0.344101425989938810f,	-0.938403534063108280f,	-0.345541324963988760f,	-0.937872376439989890f,	-0.346980410845923510f,
	-0.937339011912574960f,	-0.348418680249434560f,	-0.936803441735921670f,	-0.349856129790134640f,	-0.936265667170278260f,	-0.351292756085566980f,
	-0.935725689481080370f,	-0.352728555755210730f,	-0.935183509938947720f,	-0.354163525420490120f,	-0.934639129819680780f,	-0.355597661704783800f,
	-0.934092550404258870f,	-0.357030961233430090f,	-0.933543772978836280f,	-0.358463420633736370f,	-0.932992798834738960f,	-0.359895036534988110f,
	-0.932439629268462470f,	-0.361325805568453950f,	-0.931884265581668150f,	-0.362755724367397060f,	-0.931326709081180430f,	-0.364184789567079890f,
	-0.930766961078983820f,	-0.365612997804773580f,	-0.930205022892219070f,	-0.367040345719767070f,	-0.929640895843181210f,	-0.368466829953372380f,
	-0.929074581259315860f,	-0.369892447148933880f,	-0.928506080473215590f,	-0.371317193951837430f,	-0.927935394822617780f,	-0.372741067009515870f,
	-0.927362525650401110f,	-0.374164062971457820f,	-0.926787474304581860f,	-0.375586178489217160f,	-0.926210242138311490f,	-0.377007410216417930f,
	-0.925630830509872830f,	-0.378427754808765390f,	-0.925049240782677580f,	-0.379847208924051160f,	-0.924465474325262710f,	-0.381265769222162100f,
	-0.923879532511286850f,	-0.382683432365089670f,	-0.923291416719527640f,	-0.384100195016935100f,	-0.922701128333878630f,	-0.385516053843918630f,
	-0.922108668743345180f,	-0.386931005514388530f,	-0.921514039342041900f,	-0.388345046698826360f,	-0.920917241529189520f,	-0.389758174069856240f,
	-0.920318276709110590f,	-0.391170384302253820f,	-0.919717146291227470f,	-0.392581674072951190f,	-0.919113851690057770f,	-0.393992040061047930f,
	-0.918508394325212250f,	-0.395401478947816300f,	-0.917900775621390610f,	-0.396809987416710030f,	-0.917290997008378020f,	-0.398217562153373450f,
	-0.916679059921042700f,	-0.399624199845646840f,	-0.916064965799331830f,	-0.401029897183575400f,	-0.915448716088267830f,	-0.402434650859418370f,
	-0.914830312237946090f,	-0.403838457567654190f,	-0.914209755703530690f,	-0.405241314004989690f,	-0.913587047945250810f,	-0.406643216870368970f,
	-0.912962190428398320f,	-0.408044162864978350f,	-0.912335184623322860f,	-0.409444148692257430f,	-0.911706032005429880f,	-0.410843171057903910f,
	-0.911074734055176470f,	-0.412241226669882610f,	-0.910441292258067250f,	-0.413638312238434390f,	-0.909805708104652220f,	-0.415034424476081630f,
	-0.909167983090522490f,	-0.416429560097636930f,	-0.908528118716306230f,	-0.417823715820212220f,	-0.907886116487666150f,	-0.419216888363224020f,
	-0.907241977915295930f,	-0.420609074448402340f,	-0.906595704514915450f,	-0.422000270799799630f,	-0.905947297807268570f,	-0.423390474143795710f,
	-0.905296759318118820f,	-0.424779681209108640f,	-0.904644090578246240f,	-0.426167888726799620f,	-0.903989293123443450f,	-0.427555093430281810f,
	-0.903332368494511930f,	-0.428941292055329380f,	-0.902673318237258830f,	-0.430326481340082670f,	-0.902012143902493290f,	-0.431710658025057040f,
	-0.901348847046022030f,	-0.433093818853151850f,	-0.900683429228646860f,	-0.434475960569655760f,	-0.900015892016160280f,	-0.435857079922255310f,
	-0.899346236979341570f,	-0.437237173661044030f,	-0.898674465693954040f,	-0.438616238538527330f,	-0.898000579740739880f,	-0.439994271309633090f,
	-0.897324580705418320f,	-0.441371268731716670f,	-0.896646470178680380f,	-0.442747227564569750f,	-0.895966249756185220f,	-0.444122144570429090f,
	-0.895283921038557470f,	-0.445496016513981740f,	-0.894599485631382810f,	-0.446868840162373940f,	-0.893912945145203250f,	-0.448240612285219830f,
	-0.893224301195515320f,	-0.449611329654606650f,	-0.892533555402764690f,	-0.450980989045103640f,	-0.891840709392342720f,	-0.452349587233770830f,
	-0.891145764794583410f,	-0.453717121000163540f,	-0.890448723244757990f,	-0.455083587126343670f,	-0.889749586383072780f,	-0.456448982396883920f,
	-0.889048355854664680f,	-0.457813303598876950f,	-0.888345033309596350f,	-0.459176547521943980f,	-0.887639620402853930f,	-0.460538710958240060f,
	-0.886932118794342310f,	-0.461899790702462510f,	-0.886222530148880640f,	-0.463259783551860090f,	-0.885510856136199950f,	-0.464618686306237870f,
	-0.884797098430937900f,	-0.465976495767965960f,	-0.884081258712634990f,	-0.467333208741988360f,	-0.883363338665731800f,	-0.468688822035827620f,
	-0.882643339979562900f,	-0.470043332459595450f,	-0.881921264348355050f,	-0.471396736825997640f,	-0.881197113471222200f,	-0.472749031950342510f,
	-0.880470889052160870f,	-0.474100214650549860f,	-0.879742592800047410f,	-0.475450281747155920f,	-0.879012226428633640f,	-0.476799230063321870f,
	-0.878279791656541580f,	-0.478147056424842950f,	-0.877545290207261240f,	-0.479493757660153060f,	-0.876808723809145760f,	-0.480839330600333740f,
	-0.876070094195406600f,	-0.482183772079122660f,	-0.875329403104111000f,	-0.483527078932918410f,	-0.874586652278176220f,	-0.484869248000790950f,
	-0.873841843465366860f,	-0.486210276124486360f,	-0.873094978418290200f,	-0.487550160148435660f,	-0.872346058894391540f,	-0.488888896919763060f,
	-0.871595086655950980f,	-0.490226483288291160f,	-0.870842063470079090f,	-0.491562916106549730f,	-0.870086991108711460f,	-0.492898192229783930f,
	-0.869329871348606730f,	-0.494232308515959780f,	-0.868570705971341010f,	-0.495565261825772320f,	-0.867809496763303320f,	-0.496897049022654470f,
	-0.867046245515692870f,	-0.498227666972781540f,	-0.866280954024513110f,	-0.499557112545081730f,	-0.865513624090569090f,	-0.500885382611240710f,
	-0.864744257519462490f,	-0.502212474045710570f,	-0.863972856121586810f,	-0.503538383725717460f,	-0.863199421712124160f,	-0.504863108531267590f,
	-0.862423956111040720f,	-0.506186645345155010f,	-0.861646461143081410f,	-0.507508991052970760f,	-0.860866938637767310f,	-0.508830142543107100f,
	-0.860085390429390250f,	-0.510150096706766590f,	-0.859301818357008470f,	-0.511468850437970300f,	-0.858516224264442960f,	-0.512786400633562730f,
	-0.857728610000272120f,	-0.514102744193221550f,	-0.856938977417828760f,	-0.515417878019462930f,	-0.856147328375194690f,	-0.516731799017649650f,
	-0.855353664735196140f,	-0.518044504095999230f,	-0.854557988365400530f,	-0.519355990165589640f,	-0.853760301138111520f,	-0.520666254140366940f,
	-0.852960604930363740f,	-0.521975292937154280f,	-0.852158901623919830f,	-0.523283103475656430f,	-0.851355193105265310f,	-0.524589682678468730f,
	-0.850549481265603480f,	-0.525895027471084630f,	-0.849741768000852660f,	-0.527199134781901060f,	-0.848932055211639720f,	-0.528502001542228260f,
	-0.848120344803297230f,	-0.529803624686294610f,	-0.847306638685858540f,	-0.531104001151254670f,	-0.846490938774052130f,	-0.532403127877197790f,
	-0.845673246987299070f,	-0.533701001807152960f,	-0.844853565249707230f,	-0.534997619887096930f,	-0.844031895490066520f,	-0.536292979065963070f,
	-0.843208239641845440f,	-0.537587076295645510f,	-0.842382599643185960f,	-0.538879908531008200f,	-0.841554977436898440f,	-0.540171472729892850f,
	-0.840725374970458180f,	-0.541461765853123220f,	-0.839893794195999630f,	-0.542750784864515780f,	-0.839060237070312740f,	-0.544038526730883820f,
	-0.838224705554838190f,	-0.545324988422046130f,	-0.837387201615662050f,	-0.546610166910834750f,	-0.836547727223512010f,	-0.547894059173100190f,
	-0.835706284353752720f,	-0.549176662187719540f,	-0.834862874986380120f,	-0.550457972936604700f,	-0.834017501106018130f,	-0.551737988404707450f,
	-0.833170164701913300f,	-0.553016705580027360f,	-0.832320867767929680f,	-0.554294121453620000f,	-0.831469612302545460f,	-0.555570233019601960f,
	-0.830616400308846430f,	-0.556845037275159880f,	-0.829761233794523050f,	-0.558118531220556100f,	-0.828904114771865100f,	-0.559390711859135800f,
	-0.828045045257755800f,	-0.560661576197335920f,	-0.827184027273669130f,	-0.561931121244689360f,	-0.826321062845663650f,	-0.563199344013833870f,
	-0.825456154004377550f,	-0.564466241520519390f,	-0.824589302785025290f,	-0.565731810783613230f,	-0.823720511227391540f,	-0.566996048825108460f,
	-0.822849781375826430f,	-0.568258952670131490f,	-0.821977115279241550f,	-0.569520519346947250f,	-0.821102514991104760f,	-0.570780745886967140f,
	-0.820225982569434690f,	-0.572039629324757050f,	-0.819347520076797120f,	-0.573297166698041980f,	-0.818467129580298770f,	-0.574553355047715650f,
	-0.817584813151583710f,	-0.575808191417845340f,	-0.816700572866827960f,	-0.577061672855679330f,	-0.815814410806733890f,	-0.578313796411655480f,
	-0.814926329056526620f,	-0.579564559139405740f,	-0.814036329705948520f,	-0.580813958095764300f,	-0.813144414849253590f,	-0.582061990340775440f,
	-0.812250586585203880f,	-0.583308652937698400f,	-0.811354847017063840f,	-0.584553942953015100f,	-0.810457198252594770f,	-0.585797857456438860f,
	-0.809557642404051480f,	-0.587040393520917750f,	-0.808656181588175090f,	-0.588281548222645110f,	-0.807752817926190360f,	-0.589521318641063940f,
	-0.806847553543799450f,	-0.590759701858873940f,	-0.805940390571176390f,	-0.591996694962040880f,	-0.805031331142963550f,	-0.593232295039799800f,
	-0.804120377398265920f,	-0.594466499184664210f,	-0.803207531480644940f,	-0.595699304492433250f,	-0.802292795538115720f,	-0.596930708062196500f,
	-0.801376171723140350f,	-0.598160706996342160f,	-0.800457662192622820f,	-0.599389298400564540f,	-0.799537269107905240f,	-0.600616479383868640f,
	-0.798614994634760930f,	-0.601842247058579920f,	-0.797690840943391160f,	-0.603066598540348160f,	-0.796764810208418940f,	-0.604289530948155850f,
	-0.795836904608883570f,	-0.605511041404325430f,	-0.794907126328236900f,	-0.606731127034524480f,	-0.793975477554337280f,	-0.607949784967773410f,
	-0.793041960479443750f,	-0.609167012336453100f,	-0.792106577300212280f,	-0.610382806276309480f,	-0.791169330217690310f,	-0.611597163926461800f,
	-0.790230221437310140f,	-0.612810082429409710f,	-0.789289253168885870f,	-0.614021558931038160f,	-0.788346427626606340f,	-0.615231590580626710f,
	-0.787401747029031430f,	-0.616440174530853650f,	-0.786455213599085990f,	-0.617647307937803650f,	-0.785506829564054040f,	-0.618852987960976210f,
	-0.784556597155575240f,	-0.620057211763289210f,	-0.783604518609638420f,	-0.621259976511087440f,	-0.782650596166575840f,	-0.622461279374149860f,
	-0.781694832071059390f,	-0.623661117525694640f,	-0.780737228572094600f,	-0.624859488142386230f,	-0.779777787923014550f,	-0.626056388404343520f,
	-0.778816512381476200f,	-0.627251815495143860f,	-0.777853404209453150f,	-0.628445766601832600f,	-0.776888465673232440f,	-0.629638238914926980f,
	-0.775921699043407800f,	-0.630829229628424250f,	-0.774953106594873930f,	-0.632018735939808950f,	-0.773982690606822790f,	-0.633206755050057300f,
	-0.773010453362737100f,	-0.634393284163645270f,	-0.772036397150384520f,	-0.635578320488556110f,	-0.771060524261813710f,	-0.636761861236284310f,
	-0.770082836993348120f,	-0.637943903621843940f,	-0.769103337645579700f,	-0.639124444863775730f,	-0.768122028523365640f,	-0.640303482184151450f,
	-0.767138911935820510f,	-0.641481012808583050f,	-0.766153990196312920f,	-0.642657033966226860f,	-0.765167265622459070f,	-0.643831542889791280f,
	-0.764178740536116790f,	-0.645004536815543820f,	-0.763188417263381270f,	-0.646176012983316390f,	-0.762196298134579120f,	-0.647345968636511840f,
	-0.761202385484261890f,	-0.648514401022112330f,	-0.760206681651202420f,	-0.649681307390683190f,	-0.759209188978388180f,	-0.650846684996380760f,
	-0.758209909813015280f,	-0.652010531096959500f,	-0.757208846506484790f,	-0.653172842953776530f,	-0.756206001414394650f,	-0.654333617831800330f,
	-0.755201376896536550f,	-0.655492852999615350f,	-0.754194975316889390f,	-0.656650545729428830f,	-0.753186799043612630f,	-0.657806693297078530f,
	-0.752176850449042700f,	-0.658961292982037320f,	-0.751165131909686590f,	-0.660114342067420260f,	-0.750151645806215070f,	-0.661265837839992150f,
	-0.749136394523459260f,	-0.662415777590171780f,	-0.748119380450403710f,	-0.663564158612039660f,	-0.747100605980180130f,	-0.664710978203344790f,
	-0.746080073510064000f,	-0.665856233665509390f,	-0.745057785441466060f,	-0.666999922303637360f,	-0.744033744179929290f,	-0.668142041426518450f,
	-0.743007952135121940f,	-0.669282588346635790f,	-0.741980411720831070f,	-0.670421560380172980f,	-0.740951125354959110f,	-0.671558954847018440f,
	-0.739920095459516310f,	-0.672694769070772750f,	-0.738887324460615220f,	-0.673829000378756040f,	-0.737852814788465980f,	-0.674961646102012040f,
	-0.736816568877370020f,	-0.676092703575315810f,	-0.735778589165713590f,	-0.677222170137180330f,	-0.734738878095963720f,	-0.678350043129861250f,
	-0.733697438114660370f,	-0.679476319899364860f,	-0.732654271672412820f,	-0.680600997795453020f,	-0.731609381223892740f,	-0.681724074171649600f,
	-0.730562769227827700f,	-0.682845546385247970f,	-0.729514438146997010f,	-0.683965411797315400f,	-0.728464390448225420f,	-0.685083667772700130f,
	-0.727412628602375880f,	-0.686200311680038480f,	-0.726359155084345900f,	-0.687315340891759160f,	-0.725303972373060880f,	-0.688428752784090330f,
	-0.724247082951467000f,	-0.689540544737066830f,	-0.723188489306527680f,	-0.690650714134534380f,	-0.722128193929215460f,	-0.691759258364157640f,
	-0.721066199314508110f,	-0.692866174817424630f,	-0.720002507961381880f,	-0.693971460889653780f,	-0.718937122372804490f,	-0.695075113980000770f,
	-0.717870045055731710f,	-0.696177131491462990f,	-0.716801278521099650f,	-0.697277510830886400f,	-0.715730825283818710f,	-0.698376249408972800f,
	-0.714658687862768980f,	-0.699473344640283880f,	-0.713584868780793750f,	-0.700568793943248220f,	-0.712509370564692320f,	-0.701662594740168450f,
	-0.711432195745216660f,	-0.702754744457225080f,	-0.710353346857062420f,	-0.703845240524484830f,	-0.709272826438865690f,	-0.704934080375904880f,
	-0.708190637033195510f,	-0.706021261449339520f,	-0.707106781186547680f,	-0.707106781186547460f,	-0.706021261449339740f,	-0.708190637033195290f,
	-0.704934080375905100f,	-0.709272826438865470f,	-0.703845240524485050f,	-0.710353346857062310f,	-0.702754744457225300f,	-0.711432195745216430f,
	-0.701662594740168680f,	-0.712509370564692210f,	-0.700568793943248450f,	-0.713584868780793520f,	-0.699473344640284100f,	-0.714658687862768760f,
	-0.698376249408973030f,	-0.715730825283818480f,	-0.697277510830886630f,	-0.716801278521099540f,	-0.696177131491463210f,	-0.717870045055731490f,
	-0.695075113980000990f,	-0.718937122372804380f,	-0.693971460889654000f,	-0.720002507961381650f,	-0.692866174817424850f,	-0.721066199314507880f,
	-0.691759258364157860f,	-0.722128193929215230f,	-0.690650714134534600f,	-0.723188489306527460f,	-0.689540544737067050f,	-0.724247082951466780f,
	-0.688428752784090550f,	-0.725303972373060660f,	-0.687315340891759390f,	-0.726359155084345680f,	-0.686200311680038700f,	-0.727412628602375650f,
	-0.685083667772700360f,	-0.728464390448225200f,	-0.683965411797315630f,	-0.729514438146996790f,	-0.682845546385248190f,	-0.730562769227827480f,
	-0.681724074171649820f,	-0.731609381223892520f,	-0.680600997795453240f,	-0.732654271672412590f,	-0.679476319899365080f,	-0.733697438114660150f,
	-0.678350043129861470f,	-0.734738878095963500f,	-0.677222170137180560f,	-0.735778589165713370f,	-0.676092703575316030f,	-0.736816568877369790f,
	-0.674961646102012260f,	-0.737852814788465760f,	-0.673829000378756260f,	-0.738887324460615000f,	-0.672694769070772970f,	-0.739920095459516090f,
	-0.671558954847018660f,	-0.740951125354958880f,	-0.670421560380173200f,	-0.741980411720830960f,	-0.669282588346636120f,	-0.743007952135121720f,
	-0.668142041426518670f,	-0.744033744179929070f,	-0.666999922303637580f,	-0.745057785441465840f,	-0.665856233665509610f,	-0.746080073510063780f,
	-0.664710978203345020f,	-0.747100605980180020f,	-0.663564158612039880f,	-0.748119380450403490f,	-0.662415777590172010f,	-0.749136394523459040f,
	-0.661265837839992380f,	-0.750151645806214960f,	-0.660114342067420480f,	-0.751165131909686370f,	-0.658961292982037540f,	-0.752176850449042480f,
	-0.657806693297078750f,	-0.753186799043612410f,	-0.656650545729429050f,	-0.754194975316889170f,	-0.655492852999615570f,	-0.755201376896536320f,
	-0.654333617831800660f,	-0.756206001414394420f,	-0.653172842953777090f,	-0.757208846506484230f,	-0.652010531096959720f,	-0.758209909813015170f,
	-0.650846684996380990f,	-0.759209188978387960f,	-0.649681307390683080f,	-0.760206681651202420f,	-0.648514401022112220f,	-0.761202385484262000f,
	-0.647345968636512500f,	-0.762196298134578560f,	-0.646176012983316620f,	-0.763188417263381050f,	-0.645004536815544040f,	-0.764178740536116560f,
	-0.643831542889791500f,	-0.765167265622458960f,	-0.642657033966226750f,	-0.766153990196313030f,	-0.641481012808583610f,	-0.767138911935820070f,
	-0.640303482184152010f,	-0.768122028523365090f,	-0.639124444863775950f,	-0.769103337645579480f,	-0.637943903621844170f,	-0.770082836993347900f,
	-0.636761861236284200f,	-0.771060524261813820f,	-0.635578320488556000f,	-0.772036397150384630f,	-0.634393284163645930f,	-0.773010453362736660f,
	-0.633206755050057520f,	-0.773982690606822570f,	-0.632018735939809170f,	-0.774953106594873820f,	-0.630829229628424580f,	-0.775921699043407580f,
	-0.629638238914926870f,	-0.776888465673232550f,	-0.628445766601833160f,	-0.777853404209452700f,	-0.627251815495144420f,	-0.778816512381475650f,
	-0.626056388404343740f,	-0.779777787923014330f,	-0.624859488142386450f,	-0.780737228572094380f,	-0.623661117525694530f,	-0.781694832071059500f,
	-0.622461279374149750f,	-0.782650596166575840f,	-0.621259976511088000f,	-0.783604518609637980f,	-0.620057211763289430f,	-0.784556597155575020f,
	-0.618852987960976430f,	-0.785506829564053820f,	-0.617647307937803980f,	-0.786455213599085770f,	-0.616440174530853540f,	-0.787401747029031430f,
	-0.615231590580627260f,	-0.788346427626605890f,	-0.614021558931038710f,	-0.789289253168885430f,	-0.612810082429409930f,	-0.790230221437309920f,
	-0.611597163926462020f,	-0.791169330217690090f,	-0.610382806276309360f,	-0.792106577300212390f,	-0.609167012336452980f,	-0.793041960479443860f,
	-0.607949784967774080f,	-0.793975477554336840f,	-0.606731127034524810f,	-0.794907126328236790f,	-0.605511041404325660f,	-0.795836904608883460f,
	-0.604289530948156070f,	-0.796764810208418720f,	-0.603066598540348050f,	-0.797690840943391160f,	-0.601842247058580470f,	-0.798614994634760490f,
	-0.600616479383869310f,	-0.799537269107904790f,	-0.599389298400564760f,	-0.800457662192622600f,	-0.598160706996342380f,	-0.801376171723140130f,
	-0.596930708062196390f,	-0.802292795538115720f,	-0.595699304492433130f,	-0.803207531480645050f,	-0.594466499184664880f,	-0.804120377398265470f,
	-0.593232295039800130f,	-0.805031331142963440f,	-0.591996694962041100f,	-0.805940390571176170f,	-0.590759701858874280f,	-0.806847553543799220f,
	-0.589521318641063830f,	-0.807752817926190470f,	-0.588281548222645780f,	-0.808656181588174650f,	-0.587040393520918300f,	-0.809557642404051040f,
	-0.585797857456439090f,	-0.810457198252594660f,	-0.584553942953015330f,	-0.811354847017063730f,	-0.583308652937698290f,	-0.812250586585203990f,
	-0.582061990340775330f,	-0.813144414849253700f,	-0.580813958095764970f,	-0.814036329705948080f,	-0.579564559139405970f,	-0.814926329056526400f,
	-0.578313796411655700f,	-0.815814410806733670f,	-0.577061672855679550f,	-0.816700572866827730f,	-0.575808191417845230f,	-0.817584813151583820f,
	-0.574553355047716320f,	-0.818467129580298320f,	-0.573297166698042540f,	-0.819347520076796680f,	-0.572039629324757270f,	-0.820225982569434460f,
	-0.570780745886967370f,	-0.821102514991104650f,	-0.569520519346947140f,	-0.821977115279241550f,	-0.568258952670131380f,	-0.822849781375826430f,
	-0.566996048825109010f,	-0.823720511227391090f,	-0.565731810783613450f,	-0.824589302785025070f,	-0.564466241520519610f,	-0.825456154004377440f,
	-0.563199344013834090f,	-0.826321062845663420f,	-0.561931121244689250f,	-0.827184027273669240f,	-0.560661576197336480f,	-0.828045045257755460f,
	-0.559390711859136470f,	-0.828904114771864650f,	-0.558118531220556320f,	-0.829761233794522930f,	-0.556845037275160100f,	-0.830616400308846200f,
	-0.555570233019602180f,	-0.831469612302545240f,	-0.554294121453619890f,	-0.832320867767929800f,	-0.553016705580027910f,	-0.833170164701912960f,
	-0.551737988404707670f,	-0.834017501106017910f,	-0.550457972936604920f,	-0.834862874986380010f,	-0.549176662187719770f,	-0.835706284353752600f,
	-0.547894059173100080f,	-0.836547727223512120f,	-0.546610166910835420f,	-0.837387201615661600f,	-0.545324988422046800f,	-0.838224705554837860f,
	-0.544038526730884150f,	-0.839060237070312520f,	-0.542750784864516000f,	-0.839893794195999410f,	-0.541461765853123440f,	-0.840725374970458070f,
	-0.540171472729892740f,	-0.841554977436898550f,	-0.538879908531008870f,	-0.842382599643185630f,	-0.537587076295645730f,	-0.843208239641845210f,
	-0.536292979065963290f,	-0.844031895490066300f,	-0.534997619887097260f,	-0.844853565249707010f,	-0.533701001807152850f,	-0.845673246987299180f,
	-0.532403127877198460f,	-0.846490938774051790f,	-0.531104001151255330f,	-0.847306638685858090f,	-0.529803624686294940f,	-0.848120344803297120f,
	-0.528502001542228590f,	-0.848932055211639610f,	-0.527199134781901280f,	-0.849741768000852550f,	-0.525895027471084520f,	-0.850549481265603590f,
	-0.524589682678469390f,	-0.851355193105264860f,	-0.523283103475656760f,	-0.852158901623919610f,	-0.521975292937154500f,	-0.852960604930363520f,
	-0.520666254140367160f,	-0.853760301138111410f,	-0.519355990165589420f,	-0.854557988365400640f,	-0.518044504095999890f,	-0.855353664735195700f,
	-0.516731799017650210f,	-0.856147328375194250f,	-0.515417878019463260f,	-0.856938977417828540f,	-0.514102744193221770f,	-0.857728610000272010f,
	-0.512786400633562960f,	-0.858516224264442850f,	-0.511468850437970190f,	-0.859301818357008470f,	-0.510150096706767250f,	-0.860085390429389920f,
	-0.508830142543107320f,	-0.860866938637767090f,	-0.507508991052970980f,	-0.861646461143081190f,	-0.506186645345155340f,	-0.862423956111040500f,
	-0.504863108531267370f,	-0.863199421712124270f,	-0.503538383725718020f,	-0.863972856121586470f,	-0.502212474045711120f,	-0.864744257519462160f,
	-0.500885382611241050f,	-0.865513624090568980f,	-0.499557112545082000f,	-0.866280954024512880f,	-0.498227666972781810f,	-0.867046245515692650f,
	-0.496897049022654360f,	-0.867809496763303320f,	-0.495565261825772980f,	-0.868570705971340670f,	-0.494232308515960060f,	-0.869329871348606620f,
	-0.492898192229784200f,	-0.870086991108711350f,	-0.491562916106549950f,	-0.870842063470078860f,	-0.490226483288291050f,	-0.871595086655951090f,
	-0.488888896919763730f,	-0.872346058894391210f,	-0.487550160148436330f,	-0.873094978418289870f,	-0.486210276124486640f,	-0.873841843465366750f,
	-0.484869248000791180f,	-0.874586652278176110f,	-0.483527078932918690f,	-0.875329403104110890f,	-0.482183772079122550f,	-0.876070094195406710f,
	-0.480839330600334400f,	-0.876808723809145430f,	-0.479493757660153340f,	-0.877545290207261130f,	-0.478147056424843230f,	-0.878279791656541460f,
	-0.476799230063322140f,	-0.879012226428633410f,	-0.475450281747155760f,	-0.879742592800047520f,	-0.474100214650550520f,	-0.880470889052160530f,
	-0.472749031950343180f,	-0.881197113471221870f,	-0.471396736825997860f,	-0.881921264348354940f,	-0.470043332459595730f,	-0.882643339979562680f,
	-0.468688822035827900f,	-0.883363338665731580f,	-0.467333208741988250f,	-0.884081258712635100f,	-0.465976495767966630f,	-0.884797098430937570f,
	-0.464618686306238150f,	-0.885510856136199730f,	-0.463259783551860370f,	-0.886222530148880530f,	-0.461899790702462790f,	-0.886932118794342190f,
	-0.460538710958239890f,	-0.887639620402854050f,	-0.459176547521944640f,	-0.888345033309596020f,	-0.457813303598877620f,	-0.889048355854664350f,
	-0.456448982396884200f,	-0.889749586383072670f,	-0.455083587126343950f,	-0.890448723244757880f,	-0.453717121000163810f,	-0.891145764794583290f,
	-0.452349587233770670f,	-0.891840709392342830f,	-0.450980989045104310f,	-0.892533555402764360f,	-0.449611329654606930f,	-0.893224301195515210f,
	-0.448240612285220110f,	-0.893912945145203140f,	-0.446868840162374210f,	-0.894599485631382700f,	-0.445496016513981630f,	-0.895283921038557580f,
	-0.444122144570429760f,	-0.895966249756184880f,	-0.442747227564570410f,	-0.896646470178680040f,	-0.441371268731716950f,	-0.897324580705418210f,
	-0.439994271309633370f,	-0.898000579740739770f,	-0.438616238538527600f,	-0.898674465693953820f,	-0.437237173661043920f,	-0.899346236979341680f,
	-0.435857079922255970f,	-0.900015892016159950f,	-0.434475960569656040f,	-0.900683429228646750f,	-0.433093818853152120f,	-0.901348847046021920f,
	-0.431710658025057310f,	-0.902012143902493180f,	-0.430326481340082500f,	-0.902673318237258830f,	-0.428941292055330050f,	-0.903332368494511600f,
	-0.427555093430282470f,	-0.903989293123443120f,	-0.426167888726799890f,	-0.904644090578246130f,	-0.424779681209108920f,	-0.905296759318118700f,
	-0.423390474143795990f,	-0.905947297807268460f,	-0.422000270799799520f,	-0.906595704514915450f,	-0.420609074448403010f,	-0.907241977915295590f,
	-0.419216888363224290f,	-0.907886116487666040f,	-0.417823715820212490f,	-0.908528118716306010f,	-0.416429560097637210f,	-0.909167983090522380f,
	-0.415034424476081520f,	-0.909805708104652330f,	-0.413638312238435110f,	-0.910441292258066910f,	-0.412241226669883280f,	-0.911074734055176140f,
	-0.410843171057904190f,	-0.911706032005429770f,	-0.409444148692257760f,	-0.912335184623322750f,	-0.408044162864978630f,	-0.912962190428398210f,
	-0.406643216870368810f,	-0.913587047945250920f,	-0.405241314004990360f,	-0.914209755703530470f,	-0.403838457567654460f,	-0.914830312237945980f,
	-0.402434650859418650f,	-0.915448716088267720f,	-0.401029897183575680f,	-0.916064965799331720f,	-0.399624199845646730f,	-0.916679059921042700f,
	-0.398217562153374170f,	-0.917290997008377680f,	-0.396809987416710750f,	-0.917900775621390270f,	-0.395401478947816580f,	-0.918508394325212140f,
	-0.393992040061048210f,	-0.919113851690057660f,	-0.392581674072951470f,	-0.919717146291227360f,	-0.391170384302253700f,	-0.920318276709110590f,
	-0.389758174069856970f,	-0.920917241529189300f,	-0.388345046698826630f,	-0.921514039342041790f,	-0.386931005514388800f,	-0.922108668743345070f,
	-0.385516053843918900f,	-0.922701128333878520f,	-0.384100195016934930f,	-0.923291416719527640f,	-0.382683432365090340f,	-0.923879532511286520f,
	-0.381265769222162760f,	-0.924465474325262490f,	-0.379847208924051440f,	-0.925049240782677470f,	-0.378427754808765730f,	-0.925630830509872720f,
	-0.377007410216418200f,	-0.926210242138311380f,	-0.375586178489217050f,	-0.926787474304581860f,	-0.374164062971458490f,	-0.927362525650400890f,
	-0.372741067009516150f,	-0.927935394822617670f,	-0.371317193951837770f,	-0.928506080473215480f,	-0.369892447148934160f,	-0.929074581259315750f,
	-0.368466829953372210f,	-0.929640895843181330f,	-0.367040345719766960f,	-0.930205022892219180f,	-0.365612997804774300f,	-0.930766961078983600f,
	-0.364184789567080170f,	-0.931326709081180320f,	-0.362755724367397340f,	-0.931884265581668040f,	-0.361325805568454230f,	-0.932439629268462360f,
	-0.359895036534987940f,	-0.932992798834738960f,	-0.358463420633737040f,	-0.933543772978835950f,	-0.357030961233430370f,	-0.934092550404258760f,
	-0.355597661704784070f,	-0.934639129819680670f,	-0.354163525420490450f,	-0.935183509938947610f,	-0.352728555755210620f,	-0.935725689481080370f,
	-0.351292756085566870f,	-0.936265667170278370f,	-0.349856129790135360f,	-0.936803441735921450f,	-0.348418680249434840f,	-0.937339011912574850f,
	-0.346980410845923790f,	-0.937872376439989770f,	-0.345541324963989040f,	-0.938403534063108170f,	-0.344101425989938650f,	-0.938932483532064600f,
	-0.342660717311994880f,	-0.939459223602189700f,	-0.341219202320282740f,	-0.939983753034013820f,	-0.339776884406827070f,	-0.940506070593268300f,
	-0.338333766965541240f,	-0.941026175050889260f,	-0.336889853392219940f,	-0.941544065183020810f,	-0.335445147084531380f,	-0.942059739771017420f,
	-0.333999651442009830f,	-0.942573197601446760f,	-0.332553369866044500f,	-0.943084437466093380f,	-0.331106305759876540f,	-0.943593458161960270f,
	-0.329658462528587490f,	-0.944100258491272660f,	-0.328209843579092330f,	-0.944604837261480370f,	-0.326760452320132290f,	-0.945107193285260380f,
	-0.325310292162263310f,	-0.945607325380521170f,	-0.323859366517853080f,	-0.946105232370403340f,	-0.322407678801069910f,	-0.946600913083283530f,
	-0.320955232427875160f,	-0.947094366352777220f,	-0.319502030816015410f,	-0.947585591017741200f,	-0.318048077385015390f,	-0.948074585922276110f,
	-0.316593375556166180f,	-0.948561349915730160f,	-0.315137928752522560f,	-0.949045881852700560f,	-0.313681740398891460f,	-0.949528180593036670f,
	-0.312224813921824770f,	-0.950008245001843110f,	-0.310767152749612030f,	-0.950486073949481590f,	-0.309308760312269060f,	-0.950961666311574970f,
	-0.307849640041535090f,	-0.951435020969008340f,	-0.306389795370861030f,	-0.951906136807932350f,	-0.304929229735402320f,	-0.952375012719765880f,
	-0.303467946572011040f,	-0.952841647601198720f,	-0.302005949319228530f,	-0.953306040354193750f,	-0.300543241417273730f,	-0.953768189885990210f,
	-0.299079826308040590f,	-0.954228095109105560f,	-0.297615707435086200f,	-0.954685754941338340f,	-0.296150888243623680f,	-0.955141168305770780f,
	-0.294685372180514880f,	-0.955594334130770880f,	-0.293219162694259020f,	-0.956045251349996290f,	-0.291752263234989480f,	-0.956493918902394990f,
	-0.290284677254462440f,	-0.956940335732208820f,	-0.288816408206049370f,	-0.957384500788975970f,	-0.287347459544729290f,	-0.957826413027533020f,
	-0.285877834727081060f,	-0.958266071408017560f,	-0.284407537211272150f,	-0.958703474895871490f,	-0.282936570457055500f,	-0.959138622461841890f,
	-0.281464937925757940f,	-0.959571513081984520f,	-0.279992643080273050f,	-0.960002145737665960f,	-0.278519689385053610f,	-0.960430519415565680f,
	-0.277046080306100280f,	-0.960856633107679550f,	-0.275571819310958370f,	-0.961280485811320530f,	-0.274096909868706440f,	-0.961702076529122540f,
	-0.272621355449948870f,	-0.962121404269041580f,	-0.271145159526807790f,	-0.962538468044359270f,	-0.269668325572915530f,	-0.962953266873683770f,
	-0.268190857063403510f,	-0.963365799780953940f,	-0.266712757474898530f,	-0.963776065795439840f,	-0.265234030285511790f,	-0.964184063951745830f,
	-0.263754678974831240f,	-0.964589793289812760f,	-0.262274707023914140f,	-0.964993252854920210f,	-0.260794117915275900f,	-0.965394441697689290f,
	-0.259312915132886460f,	-0.965793358874083570f,	-0.257831102162159040f,	-0.966190003445412500f,	-0.256348682489942800f,	-0.966584374478333120f,
	-0.254865659604514350f,	-0.966976471044852180f,	-0.253382036995570600f,	-0.967366292222328390f,	-0.251897818154217250f,	-0.967753837093475400f,
	-0.250413006572965390f,	-0.968139104746362330f,	-0.248927605745720150f,	-0.968522094274417270f,	-0.247441619167773130f,	-0.968902804776428870f,
	-0.245955050335795150f,	-0.969281235356548310f,	-0.244467902747824540f,	-0.969657385124292340f,	-0.242980179903264120f,	-0.970031253194543970f,
	-0.241491885302869410f,	-0.970402838687555500f,	-0.240003022448741390f,	-0.970772140728950350f,	-0.238513594844318190f,	-0.971139158449725200f,
	-0.237023605994367670f,	-0.971503890986251670f,	-0.235533059404975790f,	-0.971866337480279290f,	-0.234041958583543570f,	-0.972226497078936270f,
	-0.232550307038775220f,	-0.972584368934732210f,	-0.231058108280670940f,	-0.972939952205560180f,	-0.229565365820519420f,	-0.973293246054698140f,
	-0.228072083170886120f,	-0.973644249650811870f,	-0.226578263845610220f,	-0.973992962167955830f,	-0.225083911359792920f,	-0.974339382785575860f,
	-0.223589029229789900f,	-0.974683510688510670f,	-0.222093620973203290f,	-0.975025345066994230f,	-0.220597690108873980f,	-0.975364885116656870f,
	-0.219101240156870100f,	-0.975702130038528460f,	-0.217604274638483780f,	-0.976037079039039020f,	-0.216106797076219490f,	-0.976369731330021140f,
	-0.214608810993786620f,	-0.976700086128711840f,	-0.213110319916091920f,	-0.977028142657754280f,	-0.211611327369227970f,	-0.977353900145199960f,
	-0.210111836880469860f,	-0.977677357824509930f,	-0.208611851978263570f,	-0.977998514934557030f,	-0.207111376192218480f,	-0.978317370719627650f,
	-0.205610413053099020f,	-0.978633924429423210f,	-0.204108966092817340f,	-0.978948175319062090f,	-0.202607038844421440f,	-0.979260122649082020f,
	-0.201104634842092070f,	-0.979569765685440520f,	-0.199601757621130970f,	-0.979877103699517640f,	-0.198098410717953420f,	-0.980182135968117430f,
	-0.196594597670080780f,	-0.980484861773469270f,	-0.195090322016128660f,	-0.980785280403230320f,	-0.193585587295803860f,	-0.981083391150486590f,
	-0.192080397049892520f,	-0.981379193313754560f,	-0.190574754820252680f,	-0.981672686196983110f,	-0.189068664149805970f,	-0.981963869109555350f,
	-0.187562128582530070f,	-0.982252741366289370f,	-0.186055151663446970f,	-0.982539302287441240f,	-0.184547736938619780f,	-0.982823551198705240f,
	-0.183039887955140950f,	-0.983105487431216290f,	-0.181531608261124830f,	-0.983385110321551290f,	-0.180022901405700070f,	-0.983662419211730140f,
	-0.178513770938997920f,	-0.983937413449218810f,	-0.177004220412149000f,	-0.984210092386929030f,	-0.175494253377271510f,	-0.984480455383220930f,
	-0.173983873387463740f,	-0.984748501801904210f,	-0.172473083996795730f,	-0.985014231012239840f,	-0.170961888760301690f,	-0.985277642388941110f,
	-0.169450291233968290f,	-0.985538735312176060f,	-0.167938294974731340f,	-0.985797509167567370f,	-0.166425903540464100f,	-0.986053963346195440f,
	-0.164913120489969760f,	-0.986308097244598670f,	-0.163399949382973780f,	-0.986559910264775410f,	-0.161886393780112240f,	-0.986809401814185420f,
	-0.160372457242928510f,	-0.987056571305750970f,	-0.158858143333861530f,	-0.987301418157858320f,	-0.157343455616238190f,	-0.987543941794359340f,
	-0.155828397654264980f,	-0.987784141644572180f,	-0.154312973013020580f,	-0.988022017143283530f,	-0.152797185258443740f,	-0.988257567730749460f,
	-0.151281037957330360f,	-0.988490792852696590f,	-0.149764534677321510f,	-0.988721691960323780f,	-0.148247678986895890f,	-0.988950264510302990f,
	-0.146730474455362300f,	-0.989176509964780900f,	-0.145212924652847850f,	-0.989400427791380270f,	-0.143695033150294690f,	-0.989622017463200780f,
	-0.142176803519448140f,	-0.989841278458820530f,	-0.140658239332849160f,	-0.990058210262297120f,	-0.139139344163825980f,	-0.990272812363169110f,
	-0.137620121586486540f,	-0.990485084256456980f,	-0.136100575175706530f,	-0.990695025442664630f,	-0.134580708507126360f,	-0.990902635427780010f,
	-0.133060525157139060f,	-0.991107913723276890f,	-0.131540028702882950f,	-0.991310859846115440f,	-0.130019222722233930f,	-0.991511473318743900f,
	-0.128498110793793590f,	-0.991709753669099530f,	-0.126976696496886120f,	-0.991905700430609330f,	-0.125454983411546320f,	-0.992099313142191800f,
	-0.123932975118512090f,	-0.992290591348257370f,	-0.122410675199215960f,	-0.992479534598710080f,	-0.120888087235777570f,	-0.992666142448947910f,
	-0.119365214810991690f,	-0.992850414459865100f,	-0.117842061508325140f,	-0.993032350197851410f,	-0.116318630911904770f,	-0.993211949234794500f,
	-0.114794926606509930f,	-0.993389211148080650f,	-0.113270952177564920f,	-0.993564135520595300f,	-0.111746711211127000f,	-0.993736721940724600f,
	-0.110222207293883310f,	-0.993906970002356060f,	-0.108697444013138800f,	-0.994074879304879370f,	-0.107172424956808770f,	-0.994240449453187900f,
	-0.105647153713410380f,	-0.994403680057679100f,	-0.104121633872055070f,	-0.994564570734255420f,	-0.102595869022436610f,	-0.994723121104325700f,
	-0.101069862754827990f,	-0.994879330794805620f,	-0.099543618660069347f,	-0.995033199438118630f,	-0.098017140329560451f,	-0.995184726672196930f,
	-0.096490431355253162f,	-0.995333912140482170f,	-0.094963495329639408f,	-0.995480755491926940f,	-0.093436335845748036f,	-0.995625256380994310f,
	-0.091908956497132821f,	-0.995767414467659820f,	-0.090381360877864914f,	-0.995907229417411720f,	-0.088853552582524364f,	-0.996044700901251970f,
	-0.087325535206192559f,	-0.996179828595696870f,	-0.085797312344440227f,	-0.996312612182778000f,	-0.084268887593324238f,	-0.996443051350042630f,
	-0.082740264549375706f,	-0.996571145790554840f,	-0.081211446809592289f,	-0.996696895202896060f,	-0.079682437971430695f,	-0.996820299291165670f,
	-0.078153241632794648f,	-0.996941357764982050f,	-0.076623861392031742f,	-0.997060070339482960f,	-0.075094300847921402f,	-0.997176436735326080f,
	-0.073564563599667357f,	-0.997290456678690210f,	-0.072034653246889097f,	-0.997402129901275300f,	-0.070504573389614356f,	-0.997511456140303450f,
	-0.068974327628267079f,	-0.997618435138519550f,	-0.067443919563664231f,	-0.997723066644191640f,	-0.065913352797003832f,	-0.997825350411111640f,
	-0.064382630929857312f,	-0.997925286198596000f,	-0.062851757564161989f,	-0.998022873771486130f,	-0.061320736302208995f,	-0.998118112900149180f,
	-0.059789570746640132f,	-0.998211003360478190f,	-0.058258264500435857f,	-0.998301544933892780f,	-0.056726821166907686f,	-0.998389737407340160f,
	-0.055195244349689712f,	-0.998475580573294770f,	-0.053663537652731026f,	-0.998559074229759310f,	-0.052131704680283657f,	-0.998640218180265160f,
	-0.050599749036899455f,	-0.998719012233872940f,	-0.049067674327418029f,	-0.998795456205172410f,	-0.047535484156959157f,	-0.998869549914283560f,
	-0.046003182130915206f,	-0.998941293186856870f,	-0.044470771854939084f,	-0.999010685854073270f,	-0.042938256934941084f,	-0.999077727752645360f,
	-0.041405640977076837f,	-0.999142418724816910f,	-0.039872927587739748f,	-0.999204758618363890f,	-0.038340120373552472f,	-0.999264747286594420f,
	-0.036807222941359331f,	-0.999322384588349430f,	-0.035274238898214294f,	-0.999377670388002850f,	-0.033741171851377760f,	-0.999430604555461730f,
	-0.032208025408304600f,	-0.999481186966166950f,	-0.030674803176636484f,	-0.999529417501093140f,	-0.029141508764194309f,	-0.999575296046749220f,
	-0.027608145778966163f,	-0.999618822495178640f,	-0.026074717829104161f,	-0.999659996743959220f,	-0.024541228522912389f,	-0.999698818696204250f,
	-0.023007681468839310f,	-0.999735288260561680f,	-0.021474080275469286f,	-0.999769405351215280f,	-0.019940428551514944f,	-0.999801169887884260f,
	-0.018406729905805164f,	-0.999830581795823400f,	-0.016872987947281894f,	-0.999857641005823860f,	-0.015339206284988121f,	-0.999882347454212560f,
	-0.013805388528060250f,	-0.999904701082852900f,	-0.012271538285720512f,	-0.999924701839144500f,	-0.010737659167264916f,	-0.999942349676023910f,
	-0.009203754782060083f,	-0.999957644551963900f,	-0.007669828739531199f,	-0.999970586430974140f,	-0.006135884649154416f,	-0.999981175282601110f,
	-0.004601926120448350f,	-0.999989411081928400f,	-0.003067956762966483f,	-0.999995293809576190f,	-0.001533980186285111f,	-0.999998823451701880f,
	-0.000000000000000184f,	-1.000000000000000000f,	0.001533980186284743f,	-0.999998823451701880f,	0.003067956762966115f,	-0.999995293809576190f,
	0.004601926120447982f,	-0.999989411081928400f,	0.006135884649154049f,	-0.999981175282601110f,	0.007669828739530832f,	-0.999970586430974140f,
	0.009203754782059715f,	-0.999957644551963900f,	0.010737659167264548f,	-0.999942349676023910f,	0.012271538285720144f,	-0.999924701839144500f,
	0.013805388528059882f,	-0.999904701082852900f,	0.015339206284987753f,	-0.999882347454212560f,	0.016872987947281527f,	-0.999857641005823860f,
	0.018406729905804796f,	-0.999830581795823400f,	0.019940428551514577f,	-0.999801169887884260f,	0.021474080275468918f,	-0.999769405351215280f,
	0.023007681468838945f,	-0.999735288260561680f,	0.024541228522912021f,	-0.999698818696204250f,	0.026074717829103793f,	-0.999659996743959220f,
	0.027608145778965795f,	-0.999618822495178640f,	0.029141508764193941f,	-0.999575296046749220f,	0.030674803176636116f,	-0.999529417501093140f,
	0.032208025408304232f,	-0.999481186966166950f,	0.033741171851377399f,	-0.999430604555461730f,	0.035274238898213926f,	-0.999377670388002850f,
	0.036807222941358964f,	-0.999322384588349540f,	0.038340120373552104f,	-0.999264747286594530f,	0.039872927587739380f,	-0.999204758618363890f,
	0.041405640977076469f,	-0.999142418724816910f,	0.042938256934940716f,	-0.999077727752645360f,	0.044470771854938716f,	-0.999010685854073380f,
	0.046003182130914845f,	-0.998941293186856870f,	0.047535484156958789f,	-0.998869549914283560f,	0.049067674327417661f,	-0.998795456205172410f,
	0.050599749036899087f,	-0.998719012233872940f,	0.052131704680283289f,	-0.998640218180265270f,	0.053663537652730658f,	-0.998559074229759310f,
	0.055195244349689344f,	-0.998475580573294770f,	0.056726821166907318f,	-0.998389737407340160f,	0.058258264500435489f,	-0.998301544933892890f,
	0.059789570746639764f,	-0.998211003360478190f,	0.061320736302208627f,	-0.998118112900149180f,	0.062851757564161628f,	-0.998022873771486240f,
	0.064382630929856952f,	-0.997925286198596000f,	0.065913352797003458f,	-0.997825350411111640f,	0.067443919563663871f,	-0.997723066644191640f,
	0.068974327628266718f,	-0.997618435138519550f,	0.070504573389613995f,	-0.997511456140303450f,	0.072034653246888722f,	-0.997402129901275300f,
	0.073564563599666982f,	-0.997290456678690210f,	0.075094300847921042f,	-0.997176436735326190f,	0.076623861392031381f,	-0.997060070339482960f,
	0.078153241632794287f,	-0.996941357764982160f,	0.079682437971430334f,	-0.996820299291165670f,	0.081211446809591914f,	-0.996696895202896060f,
	0.082740264549375331f,	-0.996571145790554840f,	0.084268887593323877f,	-0.996443051350042630f,	0.085797312344439852f,	-0.996312612182778000f,
	0.087325535206192198f,	-0.996179828595696870f,	0.088853552582524004f,	-0.996044700901252080f,	0.090381360877864539f,	-0.995907229417411720f,
	0.091908956497132446f,	-0.995767414467659820f,	0.093436335845747662f,	-0.995625256380994310f,	0.094963495329639047f,	-0.995480755491926940f,
	0.096490431355252801f,	-0.995333912140482280f,	0.098017140329560090f,	-0.995184726672196930f,	0.099543618660068972f,	-0.995033199438118630f,
	0.101069862754827630f,	-0.994879330794805620f,	0.102595869022436250f,	-0.994723121104325700f,	0.104121633872054700f,	-0.994564570734255420f,
	0.105647153713410020f,	-0.994403680057679100f,	0.107172424956808410f,	-0.994240449453188010f,	0.108697444013138440f,	-0.994074879304879480f,
	0.110222207293882930f,	-0.993906970002356060f,	0.111746711211126640f,	-0.993736721940724600f,	0.113270952177564550f,	-0.993564135520595300f,
	0.114794926606509570f,	-0.993389211148080760f,	0.116318630911904410f,	-0.993211949234794610f,	0.117842061508324780f,	-0.993032350197851410f,
	0.119365214810991330f,	-0.992850414459865100f,	0.120888087235777190f,	-0.992666142448948020f,	0.122410675199215600f,	-0.992479534598710080f,
	0.123932975118511730f,	-0.992290591348257370f,	0.125454983411545960f,	-0.992099313142191800f,	0.126976696496885750f,	-0.991905700430609330f,
	0.128498110793793220f,	-0.991709753669099530f,	0.130019222722233570f,	-0.991511473318743900f,	0.131540028702882590f,	-0.991310859846115440f,
	0.133060525157138700f,	-0.991107913723276890f,	0.134580708507125970f,	-0.990902635427780010f,	0.136100575175706170f,	-0.990695025442664630f,
	0.137620121586486180f,	-0.990485084256456980f,	0.139139344163825620f,	-0.990272812363169220f,	0.140658239332848790f,	-0.990058210262297120f,
	0.142176803519447780f,	-0.989841278458820530f,	0.143695033150294330f,	-0.989622017463200890f,	0.145212924652847490f,	-0.989400427791380380f,
	0.146730474455361940f,	-0.989176509964780900f,	0.148247678986895530f,	-0.988950264510303100f,	0.149764534677321150f,	-0.988721691960323780f,
	0.151281037957330000f,	-0.988490792852696700f,	0.152797185258443380f,	-0.988257567730749460f,	0.154312973013020220f,	-0.988022017143283530f,
	0.155828397654264620f,	-0.987784141644572290f,	0.157343455616237800f,	-0.987543941794359340f,	0.158858143333861170f,	-0.987301418157858430f,
	0.160372457242928150f,	-0.987056571305751080f,	0.161886393780111880f,	-0.986809401814185420f,	0.163399949382973420f,	-0.986559910264775410f,
	0.164913120489969390f,	-0.986308097244598780f,	0.166425903540463740f,	-0.986053963346195550f,	0.167938294974730980f,	-0.985797509167567480f,
	0.169450291233967930f,	-0.985538735312176060f,	0.170961888760301330f,	-0.985277642388941220f,	0.172473083996795370f,	-0.985014231012239950f,
	0.173983873387463380f,	-0.984748501801904320f,	0.175494253377271150f,	-0.984480455383220930f,	0.177004220412148640f,	-0.984210092386929140f,
	0.178513770938997560f,	-0.983937413449218920f,	0.180022901405699710f,	-0.983662419211730250f,	0.181531608261124470f,	-0.983385110321551290f,
	0.183039887955140590f,	-0.983105487431216400f,	0.184547736938619420f,	-0.982823551198705350f,	0.186055151663446610f,	-0.982539302287441240f,
	0.187562128582529710f,	-0.982252741366289370f,	0.189068664149805610f,	-0.981963869109555350f,	0.190574754820252320f,	-0.981672686196983220f,
	0.192080397049892160f,	-0.981379193313754670f,	0.193585587295803500f,	-0.981083391150486710f,	0.195090322016128300f,	-0.980785280403230430f,
	0.196594597670080420f,	-0.980484861773469270f,	0.198098410717953060f,	-0.980182135968117540f,	0.199601757621130610f,	-0.979877103699517750f,
	0.201104634842091710f,	-0.979569765685440630f,	0.202607038844421080f,	-0.979260122649082020f,	0.204108966092816980f,	-0.978948175319062200f,
	0.205610413053098660f,	-0.978633924429423320f,	0.207111376192218120f,	-0.978317370719627770f,	0.208611851978263210f,	-0.977998514934557140f,
	0.210111836880469500f,	-0.977677357824510040f,	0.211611327369227610f,	-0.977353900145199960f,	0.213110319916091560f,	-0.977028142657754280f,
	0.214608810993786260f,	-0.976700086128711950f,	0.216106797076219130f,	-0.976369731330021250f,	0.217604274638483420f,	-0.976037079039039130f,
	0.219101240156869740f,	-0.975702130038528570f,	0.220597690108873620f,	-0.975364885116656980f,	0.222093620973202930f,	-0.975025345066994230f,
	0.223589029229789540f,	-0.974683510688510780f,	0.225083911359792550f,	-0.974339382785575970f,	0.226578263845609860f,	-0.973992962167955940f,
	0.228072083170885760f,	-0.973644249650811870f,	0.229565365820519060f,	-0.973293246054698140f,	0.231058108280670580f,	-0.972939952205560290f,
	0.232550307038774880f,	-0.972584368934732320f,	0.234041958583543210f,	-0.972226497078936380f,	0.235533059404975460f,	-0.971866337480279400f,
	0.237023605994367310f,	-0.971503890986251780f,	0.238513594844317830f,	-0.971139158449725310f,	0.240003022448741030f,	-0.970772140728950460f,
	0.241491885302869050f,	-0.970402838687555500f,	0.242980179903263760f,	-0.970031253194543970f,	0.244467902747824180f,	-0.969657385124292450f,
	0.245955050335794790f,	-0.969281235356548420f,	0.247441619167772770f,	-0.968902804776428980f,	0.248927605745719790f,	-0.968522094274417380f,
	0.250413006572965060f,	-0.968139104746362440f,	0.251897818154216910f,	-0.967753837093475510f,	0.253382036995570270f,	-0.967366292222328510f,
	0.254865659604513960f,	-0.966976471044852290f,	0.256348682489942470f,	-0.966584374478333230f,	0.257831102162158710f,	-0.966190003445412620f,
	0.259312915132886120f,	-0.965793358874083680f,	0.260794117915275510f,	-0.965394441697689400f,	0.262274707023913810f,	-0.964993252854920320f,
	0.263754678974830850f,	-0.964589793289812870f,	0.265234030285511460f,	-0.964184063951745940f,	0.266712757474898200f,	-0.963776065795439950f,
	0.268190857063403120f,	-0.963365799780954050f,	0.269668325572915200f,	-0.962953266873683880f,	0.271145159526807400f,	-0.962538468044359390f,
	0.272621355449948530f,	-0.962121404269041690f,	0.274096909868706110f,	-0.961702076529122650f,	0.275571819310958030f,	-0.961280485811320640f,
	0.277046080306099950f,	-0.960856633107679660f,	0.278519689385053280f,	-0.960430519415565790f,	0.279992643080272720f,	-0.960002145737666070f,
	0.281464937925757610f,	-0.959571513081984630f,	0.282936570457055110f,	-0.959138622461842010f,	0.284407537211271770f,	-0.958703474895871600f,
	0.285877834727080670f,	-0.958266071408017670f,	0.287347459544728900f,	-0.957826413027533020f,	0.288816408206049040f,	-0.957384500788976080f,
	0.290284677254462050f,	-0.956940335732208940f,	0.291752263234989150f,	-0.956493918902395100f,	0.293219162694258680f,	-0.956045251349996410f,
	0.294685372180514490f,	-0.955594334130770990f,	0.296150888243623290f,	-0.955141168305770890f,	0.297615707435085860f,	-0.954685754941338450f,
	0.299079826308040250f,	-0.954228095109105670f,	0.300543241417273400f,	-0.953768189885990330f,	0.302005949319228140f,	-0.953306040354193860f,
	0.303467946572010700f,	-0.952841647601198830f,	0.304929229735401990f,	-0.952375012719765990f,	0.306389795370860640f,	-0.951906136807932460f,
	0.307849640041534760f,	-0.951435020969008450f,	0.309308760312268730f,	-0.950961666311575080f,	0.310767152749611640f,	-0.950486073949481700f,
	0.312224813921824380f,	-0.950008245001843220f,	0.313681740398891130f,	-0.949528180593036790f,	0.315137928752522170f,	-0.949045881852700670f,
	0.316593375556165790f,	-0.948561349915730270f,	0.318048077385015000f,	-0.948074585922276230f,	0.319502030816015080f,	-0.947585591017741310f,
	0.320955232427874770f,	-0.947094366352777330f,	0.322407678801069570f,	-0.946600913083283650f,	0.323859366517852740f,	-0.946105232370403450f,
	0.325310292162262930f,	-0.945607325380521280f,	0.326760452320131960f,	-0.945107193285260490f,	0.328209843579092000f,	-0.944604837261480480f,
	0.329658462528587160f,	-0.944100258491272770f,	0.331106305759876210f,	-0.943593458161960390f,	0.332553369866044170f,	-0.943084437466093490f,
	0.333999651442009490f,	-0.942573197601446870f,	0.335445147084531050f,	-0.942059739771017530f,	0.336889853392219610f,	-0.941544065183020920f,
	0.338333766965540850f,	-0.941026175050889370f,	0.339776884406826740f,	-0.940506070593268410f,	0.341219202320282410f,	-0.939983753034013940f,
	0.342660717311994540f,	-0.939459223602189810f,	0.344101425989938310f,	-0.938932483532064710f,	0.345541324963988710f,	-0.938403534063108280f,
	0.346980410845923460f,	-0.937872376439989890f,	0.348418680249434510f,	-0.937339011912574960f,	0.349856129790135030f,	-0.936803441735921560f,
	0.351292756085566540f,	-0.936265667170278480f,	0.352728555755210280f,	-0.935725689481080480f,	0.354163525420490070f,	-0.935183509938947720f,
	0.355597661704783740f,	-0.934639129819680780f,	0.357030961233430030f,	-0.934092550404258870f,	0.358463420633736710f,	-0.933543772978836170f,
	0.359895036534987610f,	-0.932992798834739070f,	0.361325805568453890f,	-0.932439629268462580f,	0.362755724367397000f,	-0.931884265581668150f,
	0.364184789567079840f,	-0.931326709081180540f,	0.365612997804773960f,	-0.930766961078983710f,	0.367040345719766570f,	-0.930205022892219290f,
	0.368466829953371880f,	-0.929640895843181440f,	0.369892447148933820f,	-0.929074581259315970f,	0.371317193951837380f,	-0.928506080473215590f,
	0.372741067009515810f,	-0.927935394822617890f,	0.374164062971458160f,	-0.927362525650401000f,	0.375586178489216720f,	-0.926787474304581970f,
	0.377007410216417870f,	-0.926210242138311490f,	0.378427754808765340f,	-0.925630830509872830f,	0.379847208924051110f,	-0.925049240782677580f,
	0.381265769222162430f,	-0.924465474325262600f,	0.382683432365090000f,	-0.923879532511286630f,	0.384100195016934600f,	-0.923291416719527860f,
	0.385516053843918570f,	-0.922701128333878740f,	0.386931005514388470f,	-0.922108668743345180f,	0.388345046698826300f,	-0.921514039342041900f,
	0.389758174069856630f,	-0.920917241529189410f,	0.391170384302253370f,	-0.920318276709110820f,	0.392581674072951140f,	-0.919717146291227470f,
	0.393992040061047880f,	-0.919113851690057880f,	0.395401478947816240f,	-0.918508394325212250f,	0.396809987416710420f,	-0.917900775621390390f,
	0.398217562153373840f,	-0.917290997008377800f,	0.399624199845646400f,	-0.916679059921042820f,	0.401029897183575350f,	-0.916064965799331830f,
	0.402434650859418320f,	-0.915448716088267830f,	0.403838457567654130f,	-0.914830312237946200f,	0.405241314004990030f,	-0.914209755703530580f,
	0.406643216870368470f,	-0.913587047945251030f,	0.408044162864978300f,	-0.912962190428398320f,	0.409444148692257370f,	-0.912335184623322860f,
	0.410843171057903860f,	-0.911706032005429880f,	0.412241226669882940f,	-0.911074734055176250f,	0.413638312238434780f,	-0.910441292258067140f,
	0.415034424476081190f,	-0.909805708104652440f,	0.416429560097636870f,	-0.909167983090522490f,	0.417823715820212160f,	-0.908528118716306230f,
	0.419216888363223960f,	-0.907886116487666260f,	0.420609074448402680f,	-0.907241977915295820f,	0.422000270799799180f,	-0.906595704514915560f,
	0.423390474143795660f,	-0.905947297807268680f,	0.424779681209108580f,	-0.905296759318118930f,	0.426167888726799560f,	-0.904644090578246240f,
	0.427555093430282140f,	-0.903989293123443340f,	0.428941292055329710f,	-0.903332368494511710f,	0.430326481340082170f,	-0.902673318237259050f,
	0.431710658025056980f,	-0.902012143902493290f,	0.433093818853151790f,	-0.901348847046022140f,	0.434475960569655710f,	-0.900683429228646860f,
	0.435857079922255640f,	-0.900015892016160170f,	0.437237173661043590f,	-0.899346236979341800f,	0.438616238538527270f,	-0.898674465693954040f,
	0.439994271309633030f,	-0.898000579740739990f,	0.441371268731716620f,	-0.897324580705418320f,	0.442747227564570080f,	-0.896646470178680150f,
	0.444122144570429420f,	-0.895966249756185000f,	0.445496016513981300f,	-0.895283921038557700f,	0.446868840162373880f,	-0.894599485631382810f,
	0.448240612285219770f,	-0.893912945145203360f,	0.449611329654606600f,	-0.893224301195515320f,	0.450980989045104030f,	-0.892533555402764580f,
	0.452349587233770390f,	-0.891840709392342950f,	0.453717121000163480f,	-0.891145764794583410f,	0.455083587126343610f,	-0.890448723244757990f,
	0.456448982396883860f,	-0.889749586383072890f,	0.457813303598877290f,	-0.889048355854664570f,	0.459176547521944370f,	-0.888345033309596240f,
	0.460538710958239560f,	-0.887639620402854160f,	0.461899790702462450f,	-0.886932118794342310f,	0.463259783551860040f,	-0.886222530148880750f,
	0.464618686306237820f,	-0.885510856136199950f,	0.465976495767966290f,	-0.884797098430937680f,	0.467333208741987920f,	-0.884081258712635320f,
	0.468688822035827570f,	-0.883363338665731800f,	0.470043332459595400f,	-0.882643339979562900f,	0.471396736825997590f,	-0.881921264348355050f,
	0.472749031950342850f,	-0.881197113471221980f,	0.474100214650550240f,	-0.880470889052160640f,	0.475450281747155480f,	-0.879742592800047630f,
	0.476799230063321870f,	-0.879012226428633640f,	0.478147056424842900f,	-0.878279791656541580f,	0.479493757660153010f,	-0.877545290207261350f,
	0.480839330600334070f,	-0.876808723809145650f,	0.482183772079122220f,	-0.876070094195406930f,	0.483527078932918350f,	-0.875329403104111000f,
	0.484869248000790900f,	-0.874586652278176220f,	0.486210276124486310f,	-0.873841843465366860f,	0.487550160148436000f,	-0.873094978418290090f,
	0.488888896919763390f,	-0.872346058894391430f,	0.490226483288290720f,	-0.871595086655951310f,	0.491562916106549670f,	-0.870842063470079090f,
	0.492898192229783870f,	-0.870086991108711460f,	0.494232308515959730f,	-0.869329871348606730f,	0.495565261825772650f,	-0.868570705971340780f,
	0.496897049022654020f,	-0.867809496763303550f,	0.498227666972781480f,	-0.867046245515692870f,	0.499557112545081670f,	-0.866280954024513110f,
	0.500885382611240710f,	-0.865513624090569090f,	0.502212474045710900f,	-0.864744257519462380f,	0.503538383725717800f,	-0.863972856121586590f,
	0.504863108531267150f,	-0.863199421712124380f,	0.506186645345155010f,	-0.862423956111040720f,	0.507508991052970650f,	-0.861646461143081410f,
	0.508830142543106990f,	-0.860866938637767310f,	0.510150096706766920f,	-0.860085390429390140f,	0.511468850437969860f,	-0.859301818357008700f,
	0.512786400633562620f,	-0.858516224264442960f,	0.514102744193221550f,	-0.857728610000272230f,	0.515417878019462930f,	-0.856938977417828760f,
	0.516731799017649980f,	-0.856147328375194470f,	0.518044504095999560f,	-0.855353664735195920f,	0.519355990165589200f,	-0.854557988365400760f,
	0.520666254140366820f,	-0.853760301138111520f,	0.521975292937154170f,	-0.852960604930363740f,	0.523283103475656430f,	-0.852158901623919830f,
	0.524589682678469060f,	-0.851355193105265080f,	0.525895027471084190f,	-0.850549481265603700f,	0.527199134781900950f,	-0.849741768000852660f,
	0.528502001542228260f,	-0.848932055211639720f,	0.529803624686294610f,	-0.848120344803297340f,	0.531104001151255000f,	-0.847306638685858320f,
	0.532403127877198120f,	-0.846490938774052020f,	0.533701001807152520f,	-0.845673246987299290f,	0.534997619887096930f,	-0.844853565249707230f,
	0.536292979065962960f,	-0.844031895490066520f,	0.537587076295645510f,	-0.843208239641845440f,	0.538879908531008530f,	-0.842382599643185850f,
	0.540171472729892410f,	-0.841554977436898780f,	0.541461765853123110f,	-0.840725374970458290f,	0.542750784864515670f,	-0.839893794195999630f,
	0.544038526730883820f,	-0.839060237070312740f,	0.545324988422046460f,	-0.838224705554837970f,	0.546610166910835080f,	-0.837387201615661820f,
	0.547894059173099750f,	-0.836547727223512230f,	0.549176662187719430f,	-0.835706284353752720f,	0.550457972936604700f,	-0.834862874986380120f,
	0.551737988404707450f,	-0.834017501106018130f,	0.553016705580027690f,	-0.833170164701913070f,	0.554294121453619670f,	-0.832320867767930020f,
	0.555570233019601840f,	-0.831469612302545460f,	0.556845037275159880f,	-0.830616400308846430f,	0.558118531220555990f,	-0.829761233794523160f,
	0.559390711859136140f,	-0.828904114771864870f,	0.560661576197336250f,	-0.828045045257755690f,	0.561931121244689020f,	-0.827184027273669350f,
	0.563199344013833870f,	-0.826321062845663650f,	0.564466241520519270f,	-0.825456154004377660f,	0.565731810783613120f,	-0.824589302785025290f,
	0.566996048825108790f,	-0.823720511227391320f,	0.568258952670131050f,	-0.822849781375826650f,	0.569520519346946810f,	-0.821977115279241780f,
	0.570780745886967030f,	-0.821102514991104870f,	0.572039629324756940f,	-0.820225982569434690f,	0.573297166698042200f,	-0.819347520076796900f,
	0.574553355047715990f,	-0.818467129580298550f,	0.575808191417844890f,	-0.817584813151584040f,	0.577061672855679220f,	-0.816700572866827960f,
	0.578313796411655370f,	-0.815814410806733890f,	0.579564559139405740f,	-0.814926329056526620f,	0.580813958095764640f,	-0.814036329705948300f,
	0.582061990340775100f,	-0.813144414849253930f,	0.583308652937697960f,	-0.812250586585204220f,	0.584553942953015100f,	-0.811354847017063950f,
	0.585797857456438750f,	-0.810457198252594880f,	0.587040393520918080f,	-0.809557642404051260f,	0.588281548222645450f,	-0.808656181588174870f,
	0.589521318641063500f,	-0.807752817926190690f,	0.590759701858873940f,	-0.806847553543799450f,	0.591996694962040770f,	-0.805940390571176390f,
	0.593232295039799800f,	-0.805031331142963660f,	0.594466499184664540f,	-0.804120377398265700f,	0.595699304492432910f,	-0.803207531480645280f,
	0.596930708062196170f,	-0.802292795538115940f,	0.598160706996342050f,	-0.801376171723140350f,	0.599389298400564430f,	-0.800457662192622820f,
	0.600616479383868970f,	-0.799537269107905010f,	0.601842247058580250f,	-0.798614994634760710f,	0.603066598540347830f,	-0.797690840943391380f,
	0.604289530948155740f,	-0.796764810208418940f,	0.605511041404325320f,	-0.795836904608883680f,	0.606731127034524480f,	-0.794907126328237010f,
	0.607949784967773740f,	-0.793975477554337060f,	0.609167012336452650f,	-0.793041960479444090f,	0.610382806276309140f,	-0.792106577300212610f,
	0.611597163926461680f,	-0.791169330217690310f,	0.612810082429409600f,	-0.790230221437310140f,	0.614021558931038490f,	-0.789289253168885650f,
	0.615231590580627040f,	-0.788346427626606120f,	0.616440174530853200f,	-0.787401747029031760f,	0.617647307937803650f,	-0.786455213599085990f,
	0.618852987960976210f,	-0.785506829564054040f,	0.620057211763289100f,	-0.784556597155575240f,	0.621259976511087660f,	-0.783604518609638200f,
	0.622461279374149520f,	-0.782650596166576060f,	0.623661117525694200f,	-0.781694832071059720f,	0.624859488142386120f,	-0.780737228572094600f,
	0.626056388404343410f,	-0.779777787923014550f,	0.627251815495144190f,	-0.778816512381475980f,	0.628445766601832820f,	-0.777853404209452930f,
	0.629638238914926650f,	-0.776888465673232780f,	0.630829229628424250f,	-0.775921699043407800f,	0.632018735939808840f,	-0.774953106594874040f,
	0.633206755050057190f,	-0.773982690606822790f,	0.634393284163645600f,	-0.773010453362736880f,	0.635578320488555670f,	-0.772036397150384860f,
	0.636761861236283870f,	-0.771060524261814040f,	0.637943903621843830f,	-0.770082836993348120f,	0.639124444863775620f,	-0.769103337645579700f,
	0.640303482184151670f,	-0.768122028523365420f,	0.641481012808583270f,	-0.767138911935820290f,	0.642657033966226420f,	-0.766153990196313250f,
	0.643831542889791160f,	-0.765167265622459180f,	0.645004536815543820f,	-0.764178740536116900f,	0.646176012983316280f,	-0.763188417263381270f,
	0.647345968636512170f,	-0.762196298134578900f,	0.648514401022112000f,	-0.761202385484262220f,	0.649681307390682860f,	-0.760206681651202750f,
	0.650846684996380650f,	-0.759209188978388180f,	0.652010531096959500f,	-0.758209909813015390f,	0.653172842953776760f,	-0.757208846506484570f,
	0.654333617831800660f,	-0.756206001414394420f,	0.655492852999615020f,	-0.755201376896536880f,	0.656650545729428710f,	-0.754194975316889390f,
	0.657806693297078530f,	-0.753186799043612630f,	0.658961292982037320f,	-0.752176850449042700f,	0.660114342067420590f,	-0.751165131909686370f,
	0.661265837839991710f,	-0.750151645806215410f,	0.662415777590171450f,	-0.749136394523459590f,	0.663564158612039550f,	-0.748119380450403710f,
	0.664710978203344790f,	-0.747100605980180240f,	0.665856233665509720f,	-0.746080073510063780f,	0.666999922303637690f,	-0.745057785441465840f,
	0.668142041426518120f,	-0.744033744179929620f,	0.669282588346635790f,	-0.743007952135121940f,	0.670421560380172980f,	-0.741980411720831180f,
	0.671558954847018330f,	-0.740951125354959110f,	0.672694769070772970f,	-0.739920095459516090f,	0.673829000378755590f,	-0.738887324460615560f,
	0.674961646102011700f,	-0.737852814788466320f,	0.676092703575315700f,	-0.736816568877370020f,	0.677222170137180330f,	-0.735778589165713590f,
	0.678350043129861470f,	-0.734738878095963390f,	0.679476319899365190f,	-0.733697438114660150f,	0.680600997795452690f,	-0.732654271672413150f,
	0.681724074171649490f,	-0.731609381223892850f,	0.682845546385247970f,	-0.730562769227827700f,	0.683965411797315400f,	-0.729514438146997010f,
	0.685083667772700470f,	-0.728464390448225090f,	0.686200311680038140f,	-0.727412628602376210f,	0.687315340891758830f,	-0.726359155084346340f,
	0.688428752784090210f,	-0.725303972373060990f,	0.689540544737066830f,	-0.724247082951467000f,	0.690650714134534720f,	-0.723188489306527350f,
	0.691759258364157860f,	-0.722128193929215230f,	0.692866174817424300f,	-0.721066199314508440f,	0.693971460889653780f,	-0.720002507961381880f,
	0.695075113980000770f,	-0.718937122372804600f,	0.696177131491462870f,	-0.717870045055731820f,	0.697277510830886630f,	-0.716801278521099430f,
	0.698376249408972360f,	-0.715730825283819040f,	0.699473344640283430f,	-0.714658687862769320f,	0.700568793943248110f,	-0.713584868780793750f,
	0.701662594740168450f,	-0.712509370564692430f,	0.702754744457225300f,	-0.711432195745216430f,	0.703845240524485050f,	-0.710353346857062200f,
	0.704934080375904550f,	-0.709272826438866020f,	0.706021261449339520f,	-0.708190637033195620f,	0.707106781186547350f,	-0.707106781186547680f,
	0.708190637033195290f,	-0.706021261449339850f,	0.709272826438865690f,	-0.704934080375904880f,	0.710353346857061860f,	-0.703845240524485380f,
	0.711432195745216100f,	-0.702754744457225630f,	0.712509370564692100f,	-0.701662594740168680f,	0.713584868780793520f,	-0.700568793943248450f,
	0.714658687862769090f,	-0.699473344640283770f,	0.715730825283818820f,	-0.698376249408972690f,	0.716801278521099090f,	-0.697277510830886960f,
	0.717870045055731490f,	-0.696177131491463210f,	0.718937122372804270f,	-0.695075113980000990f,	0.720002507961381540f,	-0.693971460889654000f,
	0.721066199314508220f,	-0.692866174817424630f,	0.722128193929214900f,	-0.691759258364158190f,	0.723188489306527130f,	-0.690650714134534940f,
	0.724247082951466670f,	-0.689540544737067160f,	0.725303972373060660f,	-0.688428752784090550f,	0.726359155084346010f,	-0.687315340891759050f,
	0.727412628602375880f,	-0.686200311680038480f,	0.728464390448224860f,	-0.685083667772700800f,	0.729514438146996790f,	-0.683965411797315740f,
	0.730562769227827370f,	-0.682845546385248190f,	0.731609381223892520f,	-0.681724074171649820f,	0.732654271672412930f,	-0.680600997795453020f,
	0.733697438114659820f,	-0.679476319899365520f,	0.734738878095963170f,	-0.678350043129861800f,	0.735778589165713370f,	-0.677222170137180670f,
	0.736816568877369790f,	-0.676092703575316030f,	0.737852814788465980f,	-0.674961646102012040f,	0.738887324460615220f,	-0.673829000378755930f,
	0.739920095459515760f,	-0.672694769070773300f,	0.740951125354958880f,	-0.671558954847018660f,	0.741980411720830850f,	-0.670421560380173310f,
	0.743007952135121610f,	-0.669282588346636120f,	0.744033744179929290f,	-0.668142041426518450f,	0.745057785441465500f,	-0.666999922303638030f,
	0.746080073510063450f,	-0.665856233665510060f,	0.747100605980179910f,	-0.664710978203345130f,	0.748119380450403490f,	-0.663564158612039880f,
	0.749136394523459370f,	-0.662415777590171780f,	0.750151645806215180f,	-0.661265837839992040f,	0.751165131909686030f,	-0.660114342067420920f,
	0.752176850449042480f,	-0.658961292982037650f,	0.753186799043612300f,	-0.657806693297078860f,	0.754194975316889060f,	-0.656650545729429050f,
	0.755201376896536550f,	-0.655492852999615350f,	0.756206001414394090f,	-0.654333617831801000f,	0.757208846506484230f,	-0.653172842953777090f,
	0.758209909813015060f,	-0.652010531096959720f,	0.759209188978387960f,	-0.650846684996380990f,	0.760206681651202420f,	-0.649681307390683190f,
	0.761202385484261890f,	-0.648514401022112330f,	0.762196298134578560f,	-0.647345968636512500f,	0.763188417263381050f,	-0.646176012983316620f,
	0.764178740536116560f,	-0.645004536815544150f,	0.765167265622458850f,	-0.643831542889791500f,	0.766153990196312920f,	-0.642657033966226750f,
	0.767138911935819960f,	-0.641481012808583610f,	0.768122028523365090f,	-0.640303482184152010f,	0.769103337645579480f,	-0.639124444863775950f,
	0.770082836993347900f,	-0.637943903621844170f,	0.771060524261813820f,	-0.636761861236284200f,	0.772036397150384630f,	-0.635578320488556000f,
	0.773010453362736660f,	-0.634393284163645930f,	0.773982690606822570f,	-0.633206755050057520f,	0.774953106594873710f,	-0.632018735939809170f,
	0.775921699043407580f,	-0.630829229628424580f,	0.776888465673232550f,	-0.629638238914926980f,	0.777853404209452700f,	-0.628445766601833160f,
	0.778816512381475650f,	-0.627251815495144530f,	0.779777787923014330f,	-0.626056388404343740f,	0.780737228572094380f,	-0.624859488142386450f,
	0.781694832071059390f,	-0.623661117525694530f,	0.782650596166575840f,	-0.622461279374149860f,	0.783604518609637870f,	-0.621259976511088000f,
	0.784556597155575020f,	-0.620057211763289540f,	0.785506829564053820f,	-0.618852987960976540f,	0.786455213599085770f,	-0.617647307937803980f,
	0.787401747029031430f,	-0.616440174530853540f,	0.788346427626605890f,	-0.615231590580627370f,	0.789289253168885430f,	-0.614021558931038820f,
	0.790230221437309810f,	-0.612810082429409930f,	0.791169330217690090f,	-0.611597163926462020f,	0.792106577300212390f,	-0.610382806276309480f,
	0.793041960479443750f,	-0.609167012336453100f,	0.793975477554336840f,	-0.607949784967774080f,	0.794907126328236680f,	-0.606731127034524810f,
	0.795836904608883340f,	-0.605511041404325660f,	0.796764810208418720f,	-0.604289530948156070f,	0.797690840943391160f,	-0.603066598540348160f,
	0.798614994634760490f,	-0.601842247058580580f,	0.799537269107904790f,	-0.600616479383869310f,	0.800457662192622600f,	-0.599389298400564760f,
	0.801376171723140130f,	-0.598160706996342380f,	0.802292795538115720f,	-0.596930708062196500f,	0.803207531480645050f,	-0.595699304492433250f,
	0.804120377398265360f,	-0.594466499184664880f,	0.805031331142963320f,	-0.593232295039800130f,	0.805940390571176170f,	-0.591996694962041100f,
	0.806847553543799220f,	-0.590759701858874280f,	0.807752817926190360f,	-0.589521318641063830f,	0.808656181588174650f,	-0.588281548222645780f,
	0.809557642404051040f,	-0.587040393520918410f,	0.810457198252594660f,	-0.585797857456439090f,	0.811354847017063620f,	-0.584553942953015440f,
	0.812250586585203880f,	-0.583308652937698290f,	0.813144414849253590f,	-0.582061990340775440f,	0.814036329705948080f,	-0.580813958095764970f,
	0.814926329056526400f,	-0.579564559139406080f,	0.815814410806733670f,	-0.578313796411655810f,	0.816700572866827730f,	-0.577061672855679550f,
	0.817584813151583710f,	-0.575808191417845230f,	0.818467129580298320f,	-0.574553355047716320f,	0.819347520076796680f,	-0.573297166698042650f,
	0.820225982569434460f,	-0.572039629324757380f,	0.821102514991104540f,	-0.570780745886967370f,	0.821977115279241550f,	-0.569520519346947140f,
	0.822849781375826430f,	-0.568258952670131380f,	0.823720511227391090f,	-0.566996048825109120f,	0.824589302785025070f,	-0.565731810783613560f,
	0.825456154004377330f,	-0.564466241520519610f,	0.826321062845663420f,	-0.563199344013834200f,	0.827184027273669130f,	-0.561931121244689360f,
	0.828045045257755350f,	-0.560661576197336590f,	0.828904114771864650f,	-0.559390711859136470f,	0.829761233794522820f,	-0.558118531220556440f,
	0.830616400308846200f,	-0.556845037275160220f,	0.831469612302545240f,	-0.555570233019602180f,	0.832320867767929680f,	-0.554294121453620000f,
	0.833170164701912850f,	-0.553016705580028020f,	0.834017501106017910f,	-0.551737988404707780f,	0.834862874986379900f,	-0.550457972936605030f,
	0.835706284353752490f,	-0.549176662187719770f,	0.836547727223512010f,	-0.547894059173100080f,	0.837387201615661600f,	-0.546610166910835420f,
	0.838224705554837750f,	-0.545324988422046800f,	0.839060237070312520f,	-0.544038526730884150f,	0.839893794195999410f,	-0.542750784864516000f,
	0.840725374970458070f,	-0.541461765853123440f,	0.841554977436898440f,	-0.540171472729892740f,	0.842382599643185510f,	-0.538879908531008870f,
	0.843208239641845210f,	-0.537587076295645840f,	0.844031895490066300f,	-0.536292979065963400f,	0.844853565249707010f,	-0.534997619887097260f,
	0.845673246987299070f,	-0.533701001807152850f,	0.846490938774051680f,	-0.532403127877198570f,	0.847306638685858090f,	-0.531104001151255440f,
	0.848120344803297120f,	-0.529803624686294940f,	0.848932055211639500f,	-0.528502001542228590f,	0.849741768000852440f,	-0.527199134781901390f,
	0.850549481265603480f,	-0.525895027471084520f,	0.851355193105264860f,	-0.524589682678469390f,	0.852158901623919610f,	-0.523283103475656760f,
	0.852960604930363520f,	-0.521975292937154500f,	0.853760301138111300f,	-0.520666254140367270f,	0.854557988365400530f,	-0.519355990165589530f,
	0.855353664735195700f,	-0.518044504095999890f,	0.856147328375194250f,	-0.516731799017650320f,	0.856938977417828540f,	-0.515417878019463260f,
	0.857728610000272010f,	-0.514102744193221880f,	0.858516224264442740f,	-0.512786400633563070f,	0.859301818357008470f,	-0.511468850437970300f,
	0.860085390429389920f,	-0.510150096706767250f,	0.860866938637767090f,	-0.508830142543107430f,	0.861646461143081190f,	-0.507508991052971090f,
	0.862423956111040500f,	-0.506186645345155340f,	0.863199421712124160f,	-0.504863108531267480f,	0.863972856121586360f,	-0.503538383725718130f,
	0.864744257519462160f,	-0.502212474045711230f,	0.865513624090568870f,	-0.500885382611241050f,	0.866280954024512880f,	-0.499557112545082000f,
	0.867046245515692650f,	-0.498227666972781870f,	0.867809496763303320f,	-0.496897049022654410f,	0.868570705971340560f,	-0.495565261825773040f,
	0.869329871348606620f,	-0.494232308515960120f,	0.870086991108711350f,	-0.492898192229784260f,	0.870842063470078860f,	-0.491562916106550010f,
	0.871595086655951090f,	-0.490226483288291100f,	0.872346058894391210f,	-0.488888896919763730f,	0.873094978418289870f,	-0.487550160148436380f,
	0.873841843465366640f,	-0.486210276124486700f,	0.874586652278176000f,	-0.484869248000791230f,	0.875329403104110890f,	-0.483527078932918740f,
	0.876070094195406710f,	-0.482183772079122610f,	0.876808723809145430f,	-0.480839330600334460f,	0.877545290207261130f,	-0.479493757660153400f,
	0.878279791656541350f,	-0.478147056424843290f,	0.879012226428633410f,	-0.476799230063322200f,	0.879742592800047520f,	-0.475450281747155810f,
	0.880470889052160420f,	-0.474100214650550580f,	0.881197113471221760f,	-0.472749031950343230f,	0.881921264348354830f,	-0.471396736825997920f,
	0.882643339979562680f,	-0.470043332459595790f,	0.883363338665731580f,	-0.468688822035827960f,	0.884081258712635100f,	-0.467333208741988300f,
	0.884797098430937460f,	-0.465976495767966680f,	0.885510856136199730f,	-0.464618686306238200f,	0.886222530148880530f,	-0.463259783551860430f,
	0.886932118794342080f,	-0.461899790702462840f,	0.887639620402853930f,	-0.460538710958239950f,	0.888345033309596020f,	-0.459176547521944700f,
	0.889048355854664350f,	-0.457813303598877670f,	0.889749586383072670f,	-0.456448982396884250f,	0.890448723244757770f,	-0.455083587126344000f,
	0.891145764794583180f,	-0.453717121000163870f,	0.891840709392342830f,	-0.452349587233770720f,	0.892533555402764360f,	-0.450980989045104360f,
	0.893224301195515100f,	-0.449611329654606980f,	0.893912945145203140f,	-0.448240612285220160f,	0.894599485631382580f,	-0.446868840162374270f,
	0.895283921038557580f,	-0.445496016513981690f,	0.895966249756184880f,	-0.444122144570429810f,	0.896646470178679930f,	-0.442747227564570470f,
	0.897324580705418100f,	-0.441371268731717000f,	0.898000579740739770f,	-0.439994271309633420f,	0.898674465693953820f,	-0.438616238538527660f,
	0.899346236979341570f,	-0.437237173661043980f,	0.900015892016159950f,	-0.435857079922256030f,	0.900683429228646750f,	-0.434475960569656090f,
	0.901348847046021920f,	-0.433093818853152180f,	0.902012143902493180f,	-0.431710658025057370f,	0.902673318237258830f,	-0.430326481340082560f,
	0.903332368494511600f,	-0.428941292055330100f,	0.903989293123443120f,	-0.427555093430282530f,	0.904644090578246020f,	-0.426167888726799950f,
	0.905296759318118700f,	-0.424779681209108970f,	0.905947297807268460f,	-0.423390474143796050f,	0.906595704514915450f,	-0.422000270799799570f,
	0.907241977915295590f,	-0.420609074448403060f,	0.907886116487666040f,	-0.419216888363224350f,	0.908528118716306010f,	-0.417823715820212550f,
	0.909167983090522380f,	-0.416429560097637260f,	0.909805708104652220f,	-0.415034424476081580f,	0.910441292258066910f,	-0.413638312238435170f,
	0.911074734055176140f,	-0.412241226669883330f,	0.911706032005429660f,	-0.410843171057904240f,	0.912335184623322640f,	-0.409444148692257810f,
	0.912962190428398100f,	-0.408044162864978690f,	0.913587047945250920f,	-0.406643216870368860f,	0.914209755703530470f,	-0.405241314004990420f,
	0.914830312237945980f,	-0.403838457567654520f,	0.915448716088267720f,	-0.402434650859418710f,	0.916064965799331610f,	-0.401029897183575730f,
	0.916679059921042700f,	-0.399624199845646790f,	0.917290997008377680f,	-0.398217562153374230f,	0.917900775621390270f,	-0.396809987416710810f,
	0.918508394325212140f,	-0.395401478947816630f,	0.919113851690057660f,	-0.393992040061048270f,	0.919717146291227360f,	-0.392581674072951530f,
	0.920318276709110590f,	-0.391170384302253760f,	0.920917241529189190f,	-0.389758174069857020f,	0.921514039342041790f,	-0.388345046698826690f,
	0.922108668743344960f,	-0.386931005514388860f,	0.922701128333878520f,	-0.385516053843918960f,	0.923291416719527640f,	-0.384100195016934990f,
	0.923879532511286520f,	-0.382683432365090390f,	0.924465474325262490f,	-0.381265769222162820f,	0.925049240782677470f,	-0.379847208924051490f,
	0.925630830509872720f,	-0.378427754808765780f,	0.926210242138311380f,	-0.377007410216418260f,	0.926787474304581860f,	-0.375586178489217100f,
	0.927362525650400890f,	-0.374164062971458540f,	0.927935394822617670f,	-0.372741067009516200f,	0.928506080473215480f,	-0.371317193951837820f,
	0.929074581259315750f,	-0.369892447148934210f,	0.929640895843181330f,	-0.368466829953372270f,	0.930205022892219180f,	-0.367040345719767010f,
	0.930766961078983490f,	-0.365612997804774350f,	0.931326709081180320f,	-0.364184789567080220f,	0.931884265581668040f,	-0.362755724367397390f,
	0.932439629268462360f,	-0.361325805568454280f,	0.932992798834738960f,	-0.359895036534988000f,	0.933543772978835950f,	-0.358463420633737100f,
	0.934092550404258760f,	-0.357030961233430420f,	0.934639129819680670f,	-0.355597661704784130f,	0.935183509938947500f,	-0.354163525420490510f,
	0.935725689481080370f,	-0.352728555755210670f,	0.936265667170278370f,	-0.351292756085566930f,	0.936803441735921450f,	-0.349856129790135420f,
	0.937339011912574850f,	-0.348418680249434900f,	0.937872376439989770f,	-0.346980410845923850f,	0.938403534063108060f,	-0.345541324963989090f,
	0.938932483532064600f,	-0.344101425989938700f,	0.939459223602189700f,	-0.342660717311994930f,	0.939983753034013820f,	-0.341219202320282800f,
	0.940506070593268180f,	-0.339776884406827130f,	0.941026175050889260f,	-0.338333766965541290f,	0.941544065183020810f,	-0.336889853392220000f,
	0.942059739771017420f,	-0.335445147084531440f,	0.942573197601446760f,	-0.333999651442009880f,	0.943084437466093380f,	-0.332553369866044560f,
	0.943593458161960270f,	-0.331106305759876600f,	0.944100258491272660f,	-0.329658462528587550f,	0.944604837261480260f,	-0.328209843579092390f,
	0.945107193285260380f,	-0.326760452320132340f,	0.945607325380521170f,	-0.325310292162263370f,	0.946105232370403340f,	-0.323859366517853130f,
	0.946600913083283530f,	-0.322407678801069960f,	0.947094366352777220f,	-0.320955232427875210f,	0.947585591017741200f,	-0.319502030816015470f,
	0.948074585922276110f,	-0.318048077385015450f,	0.948561349915730160f,	-0.316593375556166230f,	0.949045881852700560f,	-0.315137928752522610f,
	0.949528180593036670f,	-0.313681740398891520f,	0.950008245001843110f,	-0.312224813921824830f,	0.950486073949481480f,	-0.310767152749612090f,
	0.950961666311574970f,	-0.309308760312269170f,	0.951435020969008340f,	-0.307849640041535140f,	0.951906136807932230f,	-0.306389795370861080f,
	0.952375012719765880f,	-0.304929229735402370f,	0.952841647601198720f,	-0.303467946572011090f,	0.953306040354193640f,	-0.302005949319228580f,
	0.953768189885990210f,	-0.300543241417273790f,	0.954228095109105560f,	-0.299079826308040640f,	0.954685754941338340f,	-0.297615707435086250f,
	0.955141168305770780f,	-0.296150888243623730f,	0.955594334130770880f,	-0.294685372180514940f,	0.956045251349996290f,	-0.293219162694259070f,
	0.956493918902394990f,	-0.291752263234989540f,	0.956940335732208820f,	-0.290284677254462500f,	0.957384500788975860f,	-0.288816408206049420f,
	0.957826413027532910f,	-0.287347459544729340f,	0.958266071408017450f,	-0.285877834727081110f,	0.958703474895871490f,	-0.284407537211272210f,
	0.959138622461841890f,	-0.282936570457055560f,	0.959571513081984520f,	-0.281464937925758050f,	0.960002145737665960f,	-0.279992643080273110f,
	0.960430519415565680f,	-0.278519689385053670f,	0.960856633107679550f,	-0.277046080306100340f,	0.961280485811320530f,	-0.275571819310958420f,
	0.961702076529122540f,	-0.274096909868706500f,	0.962121404269041580f,	-0.272621355449948980f,	0.962538468044359160f,	-0.271145159526807840f,
	0.962953266873683770f,	-0.269668325572915590f,	0.963365799780953940f,	-0.268190857063403560f,	0.963776065795439840f,	-0.266712757474898590f,
	0.964184063951745720f,	-0.265234030285511850f,	0.964589793289812760f,	-0.263754678974831290f,	0.964993252854920210f,	-0.262274707023914200f,
	0.965394441697689290f,	-0.260794117915275960f,	0.965793358874083570f,	-0.259312915132886510f,	0.966190003445412500f,	-0.257831102162159150f,
	0.966584374478333120f,	-0.256348682489942860f,	0.966976471044852180f,	-0.254865659604514410f,	0.967366292222328390f,	-0.253382036995570660f,
	0.967753837093475400f,	-0.251897818154217300f,	0.968139104746362330f,	-0.250413006572965450f,	0.968522094274417270f,	-0.248927605745720200f,
	0.968902804776428870f,	-0.247441619167773190f,	0.969281235356548310f,	-0.245955050335795210f,	0.969657385124292340f,	-0.244467902747824590f,
	0.970031253194543970f,	-0.242980179903264180f,	0.970402838687555390f,	-0.241491885302869470f,	0.970772140728950350f,	-0.240003022448741440f,
	0.971139158449725200f,	-0.238513594844318250f,	0.971503890986251670f,	-0.237023605994367730f,	0.971866337480279290f,	-0.235533059404975880f,
	0.972226497078936270f,	-0.234041958583543620f,	0.972584368934732210f,	-0.232550307038775300f,	0.972939952205560180f,	-0.231058108280671000f,
	0.973293246054698140f,	-0.229565365820519480f,	0.973644249650811870f,	-0.228072083170886180f,	0.973992962167955830f,	-0.226578263845610280f,
	0.974339382785575860f,	-0.225083911359792970f,	0.974683510688510670f,	-0.223589029229789960f,	0.975025345066994230f,	-0.222093620973203340f,
	0.975364885116656870f,	-0.220597690108874060f,	0.975702130038528460f,	-0.219101240156870160f,	0.976037079039039020f,	-0.217604274638483840f,
	0.976369731330021140f,	-0.216106797076219550f,	0.976700086128711840f,	-0.214608810993786680f,	0.977028142657754170f,	-0.213110319916091970f,
	0.977353900145199960f,	-0.211611327369228020f,	0.977677357824509930f,	-0.210111836880469920f,	0.977998514934557030f,	-0.208611851978263620f,
	0.978317370719627650f,	-0.207111376192218530f,	0.978633924429423210f,	-0.205610413053099070f,	0.978948175319062090f,	-0.204108966092817400f,
	0.979260122649082020f,	-0.202607038844421490f,	0.979569765685440520f,	-0.201104634842092120f,	0.979877103699517640f,	-0.199601757621131030f,
	0.980182135968117430f,	-0.198098410717953480f,	0.980484861773469270f,	-0.196594597670080830f,	0.980785280403230320f,	-0.195090322016128720f,
	0.981083391150486590f,	-0.193585587295803910f,	0.981379193313754560f,	-0.192080397049892580f,	0.981672686196983110f,	-0.190574754820252740f,
	0.981963869109555350f,	-0.189068664149806030f,	0.982252741366289260f,	-0.187562128582530130f,	0.982539302287441130f,	-0.186055151663447020f,
	0.982823551198705240f,	-0.184547736938619840f,	0.983105487431216290f,	-0.183039887955141010f,	0.983385110321551180f,	-0.181531608261124880f,
	0.983662419211730140f,	-0.180022901405700130f,	0.983937413449218810f,	-0.178513770938997980f,	0.984210092386929030f,	-0.177004220412149050f,
	0.984480455383220930f,	-0.175494253377271560f,	0.984748501801904210f,	-0.173983873387463790f,	0.985014231012239840f,	-0.172473083996795780f,
	0.985277642388941110f,	-0.170961888760301770f,	0.985538735312175950f,	-0.169450291233968350f,	0.985797509167567370f,	-0.167938294974731390f,
	0.986053963346195440f,	-0.166425903540464190f,	0.986308097244598670f,	-0.164913120489969810f,	0.986559910264775300f,	-0.163399949382973860f,
	0.986809401814185420f,	-0.161886393780112300f,	0.987056571305750970f,	-0.160372457242928560f,	0.987301418157858320f,	-0.158858143333861580f,
	0.987543941794359230f,	-0.157343455616238250f,	0.987784141644572180f,	-0.155828397654265070f,	0.988022017143283530f,	-0.154312973013020630f,
	0.988257567730749460f,	-0.152797185258443800f,	0.988490792852696590f,	-0.151281037957330440f,	0.988721691960323780f,	-0.149764534677321590f,
	0.988950264510302990f,	-0.148247678986895950f,	0.989176509964780900f,	-0.146730474455362390f,	0.989400427791380270f,	-0.145212924652847940f,
	0.989622017463200780f,	-0.143695033150294750f,	0.989841278458820530f,	-0.142176803519448200f,	0.990058210262297120f,	-0.140658239332849210f,
	0.990272812363169110f,	-0.139139344163826030f,	0.990485084256456980f,	-0.137620121586486590f,	0.990695025442664630f,	-0.136100575175706590f,
	0.990902635427780010f,	-0.134580708507126420f,	0.991107913723276780f,	-0.133060525157139120f,	0.991310859846115440f,	-0.131540028702883010f,
	0.991511473318743900f,	-0.130019222722233980f,	0.991709753669099410f,	-0.128498110793793640f,	0.991905700430609330f,	-0.126976696496886170f,
	0.992099313142191690f,	-0.125454983411546370f,	0.992290591348257370f,	-0.123932975118512160f,	0.992479534598709970f,	-0.122410675199216030f,
	0.992666142448947910f,	-0.120888087235777620f,	0.992850414459864990f,	-0.119365214810991760f,	0.993032350197851410f,	-0.117842061508325200f,
	0.993211949234794500f,	-0.116318630911904840f,	0.993389211148080650f,	-0.114794926606509990f,	0.993564135520595300f,	-0.113270952177564970f,
	0.993736721940724600f,	-0.111746711211127060f,	0.993906970002356060f,	-0.110222207293883360f,	0.994074879304879370f,	-0.108697444013138870f,
	0.994240449453187900f,	-0.107172424956808840f,	0.994403680057679100f,	-0.105647153713410440f,	0.994564570734255420f,	-0.104121633872055130f,
	0.994723121104325700f,	-0.102595869022436680f,	0.994879330794805620f,	-0.101069862754828060f,	0.995033199438118630f,	-0.099543618660069402f,
	0.995184726672196930f,	-0.098017140329560506f,	0.995333912140482170f,	-0.096490431355253231f,	0.995480755491926940f,	-0.094963495329639477f,
	0.995625256380994310f,	-0.093436335845748092f,	0.995767414467659820f,	-0.091908956497132877f,	0.995907229417411720f,	-0.090381360877864969f,
	0.996044700901251970f,	-0.088853552582524420f,	0.996179828595696870f,	-0.087325535206192628f,	0.996312612182778000f,	-0.085797312344440282f,
	0.996443051350042630f,	-0.084268887593324307f,	0.996571145790554840f,	-0.082740264549375761f,	0.996696895202896060f,	-0.081211446809592344f,
	0.996820299291165670f,	-0.079682437971430750f,	0.996941357764982050f,	-0.078153241632794718f,	0.997060070339482960f,	-0.076623861392031811f,
	0.997176436735326080f,	-0.075094300847921472f,	0.997290456678690210f,	-0.073564563599667412f,	0.997402129901275300f,	-0.072034653246889152f,
	0.997511456140303450f,	-0.070504573389614425f,	0.997618435138519550f,	-0.068974327628267135f,	0.997723066644191640f,	-0.067443919563664287f,
	0.997825350411111640f,	-0.065913352797003888f,	0.997925286198596000f,	-0.064382630929857368f,	0.998022873771486130f,	-0.062851757564162059f,
	0.998118112900149180f,	-0.061320736302209057f,	0.998211003360478190f,	-0.059789570746640187f,	0.998301544933892780f,	-0.058258264500435912f,
	0.998389737407340160f,	-0.056726821166907741f,	0.998475580573294770f,	-0.055195244349689775f,	0.998559074229759310f,	-0.053663537652731082f,
	0.998640218180265160f,	-0.052131704680283719f,	0.998719012233872940f,	-0.050599749036899518f,	0.998795456205172410f,	-0.049067674327418091f,
	0.998869549914283560f,	-0.047535484156959219f,	0.998941293186856870f,	-0.046003182130915268f,	0.999010685854073270f,	-0.044470771854939146f,
	0.999077727752645360f,	-0.042938256934941139f,	0.999142418724816910f,	-0.041405640977076899f,	0.999204758618363890f,	-0.039872927587739811f,
	0.999264747286594420f,	-0.038340120373552534f,	0.999322384588349430f,	-0.036807222941359394f,	0.999377670388002850f,	-0.035274238898214350f,
	0.999430604555461730f,	-0.033741171851377823f,	0.999481186966166950f,	-0.032208025408304662f,	0.999529417501093140f,	-0.030674803176636543f,
	0.999575296046749220f,	-0.029141508764194368f,	0.999618822495178640f,	-0.027608145778966225f,	0.999659996743959220f,	-0.026074717829104220f,
	0.999698818696204250f,	-0.024541228522912448f,	0.999735288260561680f,	-0.023007681468839372f,	0.999769405351215280f,	-0.021474080275469345f,
	0.999801169887884260f,	-0.019940428551515003f,	0.999830581795823400f,	-0.018406729905805226f,	0.999857641005823860f,	-0.016872987947281957f,
	0.999882347454212560f,	-0.015339206284988182f,	0.999904701082852900f,	-0.013805388528060311f,	0.999924701839144500f,	-0.012271538285720572f,
	0.999942349676023910f,	-0.010737659167264976f,	0.999957644551963900f,	-0.009203754782060144f,	0.999970586430974140f,	-0.007669828739531261f,
	0.999981175282601110f,	-0.006135884649154477f,	0.999989411081928400f,	-0.004601926120448411f,	0.999995293809576190f,	-0.003067956762966544f,
	0.999998823451701880f,	-0.001533980186285172f,
  };

/**   
* @brief  Initialization function for the floating-point CFFT/CIFFT.  
* @param[in,out] *S             points to an instance of the floating-point CFFT/CIFFT structure.  
* @param[in]     fftLen         length of the FFT.  
* @param[in]     ifftFlag       flag that selects forward (ifftFlag=0) or inverse (ifftFlag=1) transform.  
* @param[in]     bitReverseFlag flag that enables (bitReverseFlag=1) or disables (bitReverseFlag=0) bit reversal of output.  
* @return        The function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLen</code> is not a supported value.  
*   
* \par Description:  
* \par   
* The parameter <code>ifftFlag</code> controls whether a forward or inverse transform is computed.   
* Set(=1) ifftFlag for calculation of CIFFT otherwise  CFFT is calculated  
* \par   
* The parameter <code>bitReverseFlag</code> controls whether output is in normal order or bit reversed order.   
* Set(=1) bitReverseFlag for output to be in normal order otherwise output is in bit reversed order.   
* \par   
* The parameter <code>fftLen</code>	Specifies length of CFFT/CIFFT process. Supported FFT Lengths are 16, 64, 256, 1024.   
* \par   
* This Function also initializes Twiddle factor table pointer and Bit reversal table pointer.   
*/

arm_status arm_cfft_radix4_init_f32(
  arm_cfft_radix4_instance_f32 * S,
  uint16_t fftLen,
  uint8_t ifftFlag,
  uint8_t bitReverseFlag)
{
  /*  Initialise the default arm status */
  arm_status status = ARM_MATH_SUCCESS;

  /*  Initialise the FFT length */
  S->fftLen = fftLen;

  /*  Initialise the Twiddle coefficient pointer */
  S->pTwiddle = (float32_t *) twiddleCoef;

  /*  Initialise the Flag for selection of CFFT or CIFFT */
  S->ifftFlag = ifftFlag;

  /*  Initialise the Flag for calculation Bit reversal or not */
  S->bitReverseFlag = bitReverseFlag;

  /*  Initializations of structure parameters depending on the FFT length */
  switch (S->fftLen)
  {

  case 4096u:
    /*  Initializations of structure parameters for 4096 point FFT */

    /*  Initialise the twiddle coef modifier value */
    S->twidCoefModifier = 1u;
    /*  Initialise the bit reversal table modifier */
    S->bitRevFactor = 1u;
    /*  Initialise the bit reversal table pointer */
    S->pBitRevTable = armBitRevTable;
    /*  Initialise the 1/N Value */
    S->onebyfftLen = 0.000244140625;
    break;

  case 1024u:
    /*  Initializations of structure parameters for 1024 point FFT */

    /*  Initialise the twiddle coef modifier value */
    S->twidCoefModifier = 4u;
    /*  Initialise the bit reversal table modifier */
    S->bitRevFactor = 4u;
    /*  Initialise the bit reversal table pointer */
    S->pBitRevTable = &armBitRevTable[3];
    /*  Initialise the 1/fftLen Value */
    S->onebyfftLen = 0.0009765625f;
    break;


  case 256u:
    /*  Initializations of structure parameters for 256 point FFT */
    S->twidCoefModifier = 16u;
    S->bitRevFactor = 16u;
    S->pBitRevTable = &armBitRevTable[15];
    S->onebyfftLen = 0.00390625f;
    break;

  case 64u:
    /*  Initializations of structure parameters for 64 point FFT */
    S->twidCoefModifier = 64u;
    S->bitRevFactor = 64u;
    S->pBitRevTable = &armBitRevTable[63];
    S->onebyfftLen = 0.015625f;
    break;

  case 16u:
    /*  Initializations of structure parameters for 16 point FFT */
    S->twidCoefModifier = 256u;
    S->bitRevFactor = 256u;
    S->pBitRevTable = &armBitRevTable[255];
    S->onebyfftLen = 0.0625f;
    break;


  default:
    /*  Reporting argument error if fftSize is not valid value */
    status = ARM_MATH_ARGUMENT_ERROR;
    break;
  }

  return (status);
}

/**   
 * @} end of CFFT_CIFFT group   
 */
