/* ----------------------------------------------------------------------   
* Copyright (C) 2010 ARM Limited. All rights reserved.   
*   
* $Date:        15. July 2011  
* $Revision: 	V1.0.10  
*   
* Project: 	    CMSIS DSP Library   
* Title:	    arm_cfft_radix4_init_q31.c   
*   
* Description:	Radix-4 Decimation in Frequency Q31 FFT & IFFT initialization function   
*   
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
* 
** Version 1.0.11 2011/08/17 
*    Updated to support 4096 CFFT length.  
*
* Version 1.0.10 2011/7/15 
*    Big Endian support added and Merged M0 and M3/M4 Source code.  
*   
* Version 1.0.3 2010/11/29  
*    Re-organized the CMSIS folders and updated documentation.   
*    
* Version 1.0.2 2010/11/11   
*    Documentation updated.    
*   
* Version 1.0.1 2010/10/05    
*    Production release and review comments incorporated.   
*   
* Version 1.0.0 2010/09/20    
*    Production release and review comments incorporated.   
*   
* Version 0.0.5  2010/04/26    
* 	 incorporated review comments and updated with latest CMSIS layer   
*   
* Version 0.0.3  2010/03/10    
*    Initial version   
* -------------------------------------------------------------------- */

#include "arm_math.h"
#include "arm_common_tables.h"

/**   
 * @ingroup groupTransforms   
 */

/**   
 * @addtogroup CFFT_CIFFT   
 * @{   
 */

/*   
* @brief  Twiddle factors Table   
*/

/**   
* \par  
* Example code for Q31 Twiddle factors Generation::   
* \par   
* <pre>for(i = 0; i< N; i++)   
* {   
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);   
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);   
* } </pre>   
* \par   
* where N = 1024	and PI = 3.14159265358979   
* \par   
* Cos and Sin values are interleaved fashion   
* \par   
* Convert Floating point to Q31(Fixed point 1.31):   
*	round(twiddleCoefQ31(i) * pow(2, 31))   
*   
*/

static const q31_t twiddleCoefQ31[4096*2] = {
  0x7fffffff,	0x0,	0x7ffff621,	0x3243f5,	0x7fffd886,	0x6487e3,	0x7fffa72c,	0x96cbc1,
	0x7fff6216,	0xc90f88,	0x7fff0943,	0xfb5330,	0x7ffe9cb2,	0x12d96b1,	0x7ffe1c65,	0x15fda03,
	0x7ffd885a,	0x1921d20,	0x7ffce093,	0x1c45ffe,	0x7ffc250f,	0x1f6a297,	0x7ffb55ce,	0x228e4e2,
	0x7ffa72d1,	0x25b26d7,	0x7ff97c18,	0x28d6870,	0x7ff871a2,	0x2bfa9a4,	0x7ff75370,	0x2f1ea6c,
	0x7ff62182,	0x3242abf,	0x7ff4dbd9,	0x3566a96,	0x7ff38274,	0x388a9ea,	0x7ff21553,	0x3bae8b2,
	0x7ff09478,	0x3ed26e6,	0x7feeffe1,	0x41f6480,	0x7fed5791,	0x451a177,	0x7feb9b85,	0x483ddc3,
	0x7fe9cbc0,	0x4b6195d,	0x7fe7e841,	0x4e8543e,	0x7fe5f108,	0x51a8e5c,	0x7fe3e616,	0x54cc7b1,
	0x7fe1c76b,	0x57f0035,	0x7fdf9508,	0x5b137df,	0x7fdd4eec,	0x5e36ea9,	0x7fdaf519,	0x615a48b,
	0x7fd8878e,	0x647d97c,	0x7fd6064c,	0x67a0d76,	0x7fd37153,	0x6ac406f,	0x7fd0c8a3,	0x6de7262,
	0x7fce0c3e,	0x710a345,	0x7fcb3c23,	0x742d311,	0x7fc85854,	0x77501be,	0x7fc560cf,	0x7a72f45,
	0x7fc25596,	0x7d95b9e,	0x7fbf36aa,	0x80b86c2,	0x7fbc040a,	0x83db0a7,	0x7fb8bdb8,	0x86fd947,
	0x7fb563b3,	0x8a2009a,	0x7fb1f5fc,	0x8d42699,	0x7fae7495,	0x9064b3a,	0x7faadf7c,	0x9386e78,
	0x7fa736b4,	0x96a9049,	0x7fa37a3c,	0x99cb0a7,	0x7f9faa15,	0x9cecf89,	0x7f9bc640,	0xa00ece8,
	0x7f97cebd,	0xa3308bd,	0x7f93c38c,	0xa6522fe,	0x7f8fa4b0,	0xa973ba5,	0x7f8b7227,	0xac952aa,
	0x7f872bf3,	0xafb6805,	0x7f82d214,	0xb2d7baf,	0x7f7e648c,	0xb5f8d9f,	0x7f79e35a,	0xb919dcf,
	0x7f754e80,	0xbc3ac35,	0x7f70a5fe,	0xbf5b8cb,	0x7f6be9d4,	0xc27c389,	0x7f671a05,	0xc59cc68,
	0x7f62368f,	0xc8bd35e,	0x7f5d3f75,	0xcbdd865,	0x7f5834b7,	0xcefdb76,	0x7f531655,	0xd21dc87,
	0x7f4de451,	0xd53db92,	0x7f489eaa,	0xd85d88f,	0x7f434563,	0xdb7d376,	0x7f3dd87c,	0xde9cc40,
	0x7f3857f6,	0xe1bc2e4,	0x7f32c3d1,	0xe4db75b,	0x7f2d1c0e,	0xe7fa99e,	0x7f2760af,	0xeb199a4,
	0x7f2191b4,	0xee38766,	0x7f1baf1e,	0xf1572dc,	0x7f15b8ee,	0xf475bff,	0x7f0faf25,	0xf7942c7,
	0x7f0991c4,	0xfab272b,	0x7f0360cb,	0xfdd0926,	0x7efd1c3c,	0x100ee8ad,	0x7ef6c418,	0x1040c5bb,
	0x7ef05860,	0x1072a048,	0x7ee9d914,	0x10a4784b,	0x7ee34636,	0x10d64dbd,	0x7edc9fc6,	0x11082096,
	0x7ed5e5c6,	0x1139f0cf,	0x7ecf1837,	0x116bbe60,	0x7ec8371a,	0x119d8941,	0x7ec14270,	0x11cf516a,
	0x7eba3a39,	0x120116d5,	0x7eb31e78,	0x1232d979,	0x7eabef2c,	0x1264994e,	0x7ea4ac58,	0x1296564d,
	0x7e9d55fc,	0x12c8106f,	0x7e95ec1a,	0x12f9c7aa,	0x7e8e6eb2,	0x132b7bf9,	0x7e86ddc6,	0x135d2d53,
	0x7e7f3957,	0x138edbb1,	0x7e778166,	0x13c0870a,	0x7e6fb5f4,	0x13f22f58,	0x7e67d703,	0x1423d492,
	0x7e5fe493,	0x145576b1,	0x7e57dea7,	0x148715ae,	0x7e4fc53e,	0x14b8b17f,	0x7e47985b,	0x14ea4a1f,
	0x7e3f57ff,	0x151bdf86,	0x7e37042a,	0x154d71aa,	0x7e2e9cdf,	0x157f0086,	0x7e26221f,	0x15b08c12,
	0x7e1d93ea,	0x15e21445,	0x7e14f242,	0x16139918,	0x7e0c3d29,	0x16451a83,	0x7e0374a0,	0x1676987f,
	0x7dfa98a8,	0x16a81305,	0x7df1a942,	0x16d98a0c,	0x7de8a670,	0x170afd8d,	0x7ddf9034,	0x173c6d80,
	0x7dd6668f,	0x176dd9de,	0x7dcd2981,	0x179f429f,	0x7dc3d90d,	0x17d0a7bc,	0x7dba7534,	0x1802092c,
	0x7db0fdf8,	0x183366e9,	0x7da77359,	0x1864c0ea,	0x7d9dd55a,	0x18961728,	0x7d9423fc,	0x18c7699b,
	0x7d8a5f40,	0x18f8b83c,	0x7d808728,	0x192a0304,	0x7d769bb5,	0x195b49ea,	0x7d6c9ce9,	0x198c8ce7,
	0x7d628ac6,	0x19bdcbf3,	0x7d58654d,	0x19ef0707,	0x7d4e2c7f,	0x1a203e1b,	0x7d43e05e,	0x1a517128,
	0x7d3980ec,	0x1a82a026,	0x7d2f0e2b,	0x1ab3cb0d,	0x7d24881b,	0x1ae4f1d6,	0x7d19eebf,	0x1b161479,
	0x7d0f4218,	0x1b4732ef,	0x7d048228,	0x1b784d30,	0x7cf9aef0,	0x1ba96335,	0x7ceec873,	0x1bda74f6,
	0x7ce3ceb2,	0x1c0b826a,	0x7cd8c1ae,	0x1c3c8b8c,	0x7ccda169,	0x1c6d9053,	0x7cc26de5,	0x1c9e90b8,
	0x7cb72724,	0x1ccf8cb3,	0x7cabcd28,	0x1d00843d,	0x7ca05ff1,	0x1d31774d,	0x7c94df83,	0x1d6265dd,
	0x7c894bde,	0x1d934fe5,	0x7c7da505,	0x1dc4355e,	0x7c71eaf9,	0x1df5163f,	0x7c661dbc,	0x1e25f282,
	0x7c5a3d50,	0x1e56ca1e,	0x7c4e49b7,	0x1e879d0d,	0x7c4242f2,	0x1eb86b46,	0x7c362904,	0x1ee934c3,
	0x7c29fbee,	0x1f19f97b,	0x7c1dbbb3,	0x1f4ab968,	0x7c116853,	0x1f7b7481,	0x7c0501d2,	0x1fac2abf,
	0x7bf88830,	0x1fdcdc1b,	0x7bebfb70,	0x200d888d,	0x7bdf5b94,	0x203e300d,	0x7bd2a89e,	0x206ed295,
	0x7bc5e290,	0x209f701c,	0x7bb9096b,	0x20d0089c,	0x7bac1d31,	0x21009c0c,	0x7b9f1de6,	0x21312a65,
	0x7b920b89,	0x2161b3a0,	0x7b84e61f,	0x219237b5,	0x7b77ada8,	0x21c2b69c,	0x7b6a6227,	0x21f3304f,
	0x7b5d039e,	0x2223a4c5,	0x7b4f920e,	0x225413f8,	0x7b420d7a,	0x22847de0,	0x7b3475e5,	0x22b4e274,
	0x7b26cb4f,	0x22e541af,	0x7b190dbc,	0x23159b88,	0x7b0b3d2c,	0x2345eff8,	0x7afd59a4,	0x23763ef7,
	0x7aef6323,	0x23a6887f,	0x7ae159ae,	0x23d6cc87,	0x7ad33d45,	0x24070b08,	0x7ac50dec,	0x243743fa,
	0x7ab6cba4,	0x24677758,	0x7aa8766f,	0x2497a517,	0x7a9a0e50,	0x24c7cd33,	0x7a8b9348,	0x24f7efa2,
	0x7a7d055b,	0x25280c5e,	0x7a6e648a,	0x2558235f,	0x7a5fb0d8,	0x2588349d,	0x7a50ea47,	0x25b84012,
	0x7a4210d8,	0x25e845b6,	0x7a332490,	0x26184581,	0x7a24256f,	0x26483f6c,	0x7a151378,	0x26783370,
	0x7a05eead,	0x26a82186,	0x79f6b711,	0x26d809a5,	0x79e76ca7,	0x2707ebc7,	0x79d80f6f,	0x2737c7e3,
	0x79c89f6e,	0x27679df4,	0x79b91ca4,	0x27976df1,	0x79a98715,	0x27c737d3,	0x7999dec4,	0x27f6fb92,
	0x798a23b1,	0x2826b928,	0x797a55e0,	0x2856708d,	0x796a7554,	0x288621b9,	0x795a820e,	0x28b5cca5,
	0x794a7c12,	0x28e5714b,	0x793a6361,	0x29150fa1,	0x792a37fe,	0x2944a7a2,	0x7919f9ec,	0x29743946,
	0x7909a92d,	0x29a3c485,	0x78f945c3,	0x29d34958,	0x78e8cfb2,	0x2a02c7b8,	0x78d846fb,	0x2a323f9e,
	0x78c7aba2,	0x2a61b101,	0x78b6fda8,	0x2a911bdc,	0x78a63d11,	0x2ac08026,	0x789569df,	0x2aefddd8,
	0x78848414,	0x2b1f34eb,	0x78738bb3,	0x2b4e8558,	0x786280bf,	0x2b7dcf17,	0x7851633b,	0x2bad1221,
	0x78403329,	0x2bdc4e6f,	0x782ef08b,	0x2c0b83fa,	0x781d9b65,	0x2c3ab2b9,	0x780c33b8,	0x2c69daa6,
	0x77fab989,	0x2c98fbba,	0x77e92cd9,	0x2cc815ee,	0x77d78daa,	0x2cf72939,	0x77c5dc01,	0x2d263596,
	0x77b417df,	0x2d553afc,	0x77a24148,	0x2d843964,	0x7790583e,	0x2db330c7,	0x777e5cc3,	0x2de2211e,
	0x776c4edb,	0x2e110a62,	0x775a2e89,	0x2e3fec8b,	0x7747fbce,	0x2e6ec792,	0x7735b6af,	0x2e9d9b70,
	0x77235f2d,	0x2ecc681e,	0x7710f54c,	0x2efb2d95,	0x76fe790e,	0x2f29ebcc,	0x76ebea77,	0x2f58a2be,
	0x76d94989,	0x2f875262,	0x76c69647,	0x2fb5fab2,	0x76b3d0b4,	0x2fe49ba7,	0x76a0f8d2,	0x30133539,
	0x768e0ea6,	0x3041c761,	0x767b1231,	0x30705217,	0x76680376,	0x309ed556,	0x7654e279,	0x30cd5115,
	0x7641af3d,	0x30fbc54d,	0x762e69c4,	0x312a31f8,	0x761b1211,	0x3158970e,	0x7607a828,	0x3186f487,
	0x75f42c0b,	0x31b54a5e,	0x75e09dbd,	0x31e39889,	0x75ccfd42,	0x3211df04,	0x75b94a9c,	0x32401dc6,
	0x75a585cf,	0x326e54c7,	0x7591aedd,	0x329c8402,	0x757dc5ca,	0x32caab6f,	0x7569ca99,	0x32f8cb07,
	0x7555bd4c,	0x3326e2c3,	0x75419de7,	0x3354f29b,	0x752d6c6c,	0x3382fa88,	0x751928e0,	0x33b0fa84,
	0x7504d345,	0x33def287,	0x74f06b9e,	0x340ce28b,	0x74dbf1ef,	0x343aca87,	0x74c7663a,	0x3468aa76,
	0x74b2c884,	0x34968250,	0x749e18cd,	0x34c4520d,	0x7489571c,	0x34f219a8,	0x74748371,	0x351fd918,
	0x745f9dd1,	0x354d9057,	0x744aa63f,	0x357b3f5d,	0x74359cbd,	0x35a8e625,	0x74208150,	0x35d684a6,
	0x740b53fb,	0x36041ad9,	0x73f614c0,	0x3631a8b8,	0x73e0c3a3,	0x365f2e3b,	0x73cb60a8,	0x368cab5c,
	0x73b5ebd1,	0x36ba2014,	0x73a06522,	0x36e78c5b,	0x738acc9e,	0x3714f02a,	0x73752249,	0x37424b7b,
	0x735f6626,	0x376f9e46,	0x73499838,	0x379ce885,	0x7333b883,	0x37ca2a30,	0x731dc70a,	0x37f76341,
	0x7307c3d0,	0x382493b0,	0x72f1aed9,	0x3851bb77,	0x72db8828,	0x387eda8e,	0x72c54fc1,	0x38abf0ef,
	0x72af05a7,	0x38d8fe93,	0x7298a9dd,	0x39060373,	0x72823c67,	0x3932ff87,	0x726bbd48,	0x395ff2c9,
	0x72552c85,	0x398cdd32,	0x723e8a20,	0x39b9bebc,	0x7227d61c,	0x39e6975e,	0x7211107e,	0x3a136712,
	0x71fa3949,	0x3a402dd2,	0x71e35080,	0x3a6ceb96,	0x71cc5626,	0x3a99a057,	0x71b54a41,	0x3ac64c0f,
	0x719e2cd2,	0x3af2eeb7,	0x7186fdde,	0x3b1f8848,	0x716fbd68,	0x3b4c18ba,	0x71586b74,	0x3b78a007,
	0x71410805,	0x3ba51e29,	0x7129931f,	0x3bd19318,	0x71120cc5,	0x3bfdfecd,	0x70fa74fc,	0x3c2a6142,
	0x70e2cbc6,	0x3c56ba70,	0x70cb1128,	0x3c830a50,	0x70b34525,	0x3caf50da,	0x709b67c0,	0x3cdb8e09,
	0x708378ff,	0x3d07c1d6,	0x706b78e3,	0x3d33ec39,	0x70536771,	0x3d600d2c,	0x703b44ad,	0x3d8c24a8,
	0x7023109a,	0x3db832a6,	0x700acb3c,	0x3de4371f,	0x6ff27497,	0x3e10320d,	0x6fda0cae,	0x3e3c2369,
	0x6fc19385,	0x3e680b2c,	0x6fa90921,	0x3e93e950,	0x6f906d84,	0x3ebfbdcd,	0x6f77c0b3,	0x3eeb889c,
	0x6f5f02b2,	0x3f1749b8,	0x6f463383,	0x3f430119,	0x6f2d532c,	0x3f6eaeb8,	0x6f1461b0,	0x3f9a5290,
	0x6efb5f12,	0x3fc5ec98,	0x6ee24b57,	0x3ff17cca,	0x6ec92683,	0x401d0321,	0x6eaff099,	0x40487f94,
	0x6e96a99d,	0x4073f21d,	0x6e7d5193,	0x409f5ab6,	0x6e63e87f,	0x40cab958,	0x6e4a6e66,	0x40f60dfb,
	0x6e30e34a,	0x4121589b,	0x6e174730,	0x414c992f,	0x6dfd9a1c,	0x4177cfb1,	0x6de3dc11,	0x41a2fc1a,
	0x6dca0d14,	0x41ce1e65,	0x6db02d29,	0x41f93689,	0x6d963c54,	0x42244481,	0x6d7c3a98,	0x424f4845,
	0x6d6227fa,	0x427a41d0,	0x6d48047e,	0x42a5311b,	0x6d2dd027,	0x42d0161e,	0x6d138afb,	0x42faf0d4,
	0x6cf934fc,	0x4325c135,	0x6cdece2f,	0x4350873c,	0x6cc45698,	0x437b42e1,	0x6ca9ce3b,	0x43a5f41e,
	0x6c8f351c,	0x43d09aed,	0x6c748b3f,	0x43fb3746,	0x6c59d0a9,	0x4425c923,	0x6c3f055d,	0x4450507e,
	0x6c242960,	0x447acd50,	0x6c093cb6,	0x44a53f93,	0x6bee3f62,	0x44cfa740,	0x6bd3316a,	0x44fa0450,
	0x6bb812d1,	0x452456bd,	0x6b9ce39b,	0x454e9e80,	0x6b81a3cd,	0x4578db93,	0x6b66536b,	0x45a30df0,
	0x6b4af279,	0x45cd358f,	0x6b2f80fb,	0x45f7526b,	0x6b13fef5,	0x4621647d,	0x6af86c6c,	0x464b6bbe,
	0x6adcc964,	0x46756828,	0x6ac115e2,	0x469f59b4,	0x6aa551e9,	0x46c9405c,	0x6a897d7d,	0x46f31c1a,
	0x6a6d98a4,	0x471cece7,	0x6a51a361,	0x4746b2bc,	0x6a359db9,	0x47706d93,	0x6a1987b0,	0x479a1d67,
	0x69fd614a,	0x47c3c22f,	0x69e12a8c,	0x47ed5be6,	0x69c4e37a,	0x4816ea86,	0x69a88c19,	0x48406e08,
	0x698c246c,	0x4869e665,	0x696fac78,	0x48935397,	0x69532442,	0x48bcb599,	0x69368bce,	0x48e60c62,
	0x6919e320,	0x490f57ee,	0x68fd2a3d,	0x49389836,	0x68e06129,	0x4961cd33,	0x68c387e9,	0x498af6df,
	0x68a69e81,	0x49b41533,	0x6889a4f6,	0x49dd282a,	0x686c9b4b,	0x4a062fbd,	0x684f8186,	0x4a2f2be6,
	0x683257ab,	0x4a581c9e,	0x68151dbe,	0x4a8101de,	0x67f7d3c5,	0x4aa9dba2,	0x67da79c3,	0x4ad2a9e2,
	0x67bd0fbd,	0x4afb6c98,	0x679f95b7,	0x4b2423be,	0x67820bb7,	0x4b4ccf4d,	0x676471c0,	0x4b756f40,
	0x6746c7d8,	0x4b9e0390,	0x67290e02,	0x4bc68c36,	0x670b4444,	0x4bef092d,	0x66ed6aa1,	0x4c177a6e,
	0x66cf8120,	0x4c3fdff4,	0x66b187c3,	0x4c6839b7,	0x66937e91,	0x4c9087b1,	0x6675658c,	0x4cb8c9dd,
	0x66573cbb,	0x4ce10034,	0x66390422,	0x4d092ab0,	0x661abbc5,	0x4d31494b,	0x65fc63a9,	0x4d595bfe,
	0x65ddfbd3,	0x4d8162c4,	0x65bf8447,	0x4da95d96,	0x65a0fd0b,	0x4dd14c6e,	0x65826622,	0x4df92f46,
	0x6563bf92,	0x4e210617,	0x6545095f,	0x4e48d0dd,	0x6526438f,	0x4e708f8f,	0x65076e25,	0x4e984229,
	0x64e88926,	0x4ebfe8a5,	0x64c99498,	0x4ee782fb,	0x64aa907f,	0x4f0f1126,	0x648b7ce0,	0x4f369320,
	0x646c59bf,	0x4f5e08e3,	0x644d2722,	0x4f857269,	0x642de50d,	0x4faccfab,	0x640e9386,	0x4fd420a4,
	0x63ef3290,	0x4ffb654d,	0x63cfc231,	0x50229da1,	0x63b0426d,	0x5049c999,	0x6390b34a,	0x5070e92f,
	0x637114cc,	0x5097fc5e,	0x635166f9,	0x50bf031f,	0x6331a9d4,	0x50e5fd6d,	0x6311dd64,	0x510ceb40,
	0x62f201ac,	0x5133cc94,	0x62d216b3,	0x515aa162,	0x62b21c7b,	0x518169a5,	0x6292130c,	0x51a82555,
	0x6271fa69,	0x51ced46e,	0x6251d298,	0x51f576ea,	0x62319b9d,	0x521c0cc2,	0x6211557e,	0x524295f0,
	0x61f1003f,	0x5269126e,	0x61d09be5,	0x528f8238,	0x61b02876,	0x52b5e546,	0x618fa5f7,	0x52dc3b92,
	0x616f146c,	0x53028518,	0x614e73da,	0x5328c1d0,	0x612dc447,	0x534ef1b5,	0x610d05b7,	0x537514c2,
	0x60ec3830,	0x539b2af0,	0x60cb5bb7,	0x53c13439,	0x60aa7050,	0x53e73097,	0x60897601,	0x540d2005,
	0x60686ccf,	0x5433027d,	0x604754bf,	0x5458d7f9,	0x60262dd6,	0x547ea073,	0x6004f819,	0x54a45be6,
	0x5fe3b38d,	0x54ca0a4b,	0x5fc26038,	0x54efab9c,	0x5fa0fe1f,	0x55153fd4,	0x5f7f8d46,	0x553ac6ee,
	0x5f5e0db3,	0x556040e2,	0x5f3c7f6b,	0x5585adad,	0x5f1ae274,	0x55ab0d46,	0x5ef936d1,	0x55d05faa,
	0x5ed77c8a,	0x55f5a4d2,	0x5eb5b3a2,	0x561adcb9,	0x5e93dc1f,	0x56400758,	0x5e71f606,	0x566524aa,
	0x5e50015d,	0x568a34a9,	0x5e2dfe29,	0x56af3750,	0x5e0bec6e,	0x56d42c99,	0x5de9cc33,	0x56f9147e,
	0x5dc79d7c,	0x571deefa,	0x5da5604f,	0x5742bc06,	0x5d8314b1,	0x57677b9d,	0x5d60baa7,	0x578c2dba,
	0x5d3e5237,	0x57b0d256,	0x5d1bdb65,	0x57d5696d,	0x5cf95638,	0x57f9f2f8,	0x5cd6c2b5,	0x581e6ef1,
	0x5cb420e0,	0x5842dd54,	0x5c9170bf,	0x58673e1b,	0x5c6eb258,	0x588b9140,	0x5c4be5b0,	0x58afd6bd,
	0x5c290acc,	0x58d40e8c,	0x5c0621b2,	0x58f838a9,	0x5be32a67,	0x591c550e,	0x5bc024f0,	0x594063b5,
	0x5b9d1154,	0x59646498,	0x5b79ef96,	0x598857b2,	0x5b56bfbd,	0x59ac3cfd,	0x5b3381ce,	0x59d01475,
	0x5b1035cf,	0x59f3de12,	0x5aecdbc5,	0x5a1799d1,	0x5ac973b5,	0x5a3b47ab,	0x5aa5fda5,	0x5a5ee79a,
	0x5a82799a,	0x5a82799a,	0x5a5ee79a,	0x5aa5fda5,	0x5a3b47ab,	0x5ac973b5,	0x5a1799d1,	0x5aecdbc5,
	0x59f3de12,	0x5b1035cf,	0x59d01475,	0x5b3381ce,	0x59ac3cfd,	0x5b56bfbd,	0x598857b2,	0x5b79ef96,
	0x59646498,	0x5b9d1154,	0x594063b5,	0x5bc024f0,	0x591c550e,	0x5be32a67,	0x58f838a9,	0x5c0621b2,
	0x58d40e8c,	0x5c290acc,	0x58afd6bd,	0x5c4be5b0,	0x588b9140,	0x5c6eb258,	0x58673e1b,	0x5c9170bf,
	0x5842dd54,	0x5cb420e0,	0x581e6ef1,	0x5cd6c2b5,	0x57f9f2f8,	0x5cf95638,	0x57d5696d,	0x5d1bdb65,
	0x57b0d256,	0x5d3e5237,	0x578c2dba,	0x5d60baa7,	0x57677b9d,	0x5d8314b1,	0x5742bc06,	0x5da5604f,
	0x571deefa,	0x5dc79d7c,	0x56f9147e,	0x5de9cc33,	0x56d42c99,	0x5e0bec6e,	0x56af3750,	0x5e2dfe29,
	0x568a34a9,	0x5e50015d,	0x566524aa,	0x5e71f606,	0x56400758,	0x5e93dc1f,	0x561adcb9,	0x5eb5b3a2,
	0x55f5a4d2,	0x5ed77c8a,	0x55d05faa,	0x5ef936d1,	0x55ab0d46,	0x5f1ae274,	0x5585adad,	0x5f3c7f6b,
	0x556040e2,	0x5f5e0db3,	0x553ac6ee,	0x5f7f8d46,	0x55153fd4,	0x5fa0fe1f,	0x54efab9c,	0x5fc26038,
	0x54ca0a4b,	0x5fe3b38d,	0x54a45be6,	0x6004f819,	0x547ea073,	0x60262dd6,	0x5458d7f9,	0x604754bf,
	0x5433027d,	0x60686ccf,	0x540d2005,	0x60897601,	0x53e73097,	0x60aa7050,	0x53c13439,	0x60cb5bb7,
	0x539b2af0,	0x60ec3830,	0x537514c2,	0x610d05b7,	0x534ef1b5,	0x612dc447,	0x5328c1d0,	0x614e73da,
	0x53028518,	0x616f146c,	0x52dc3b92,	0x618fa5f7,	0x52b5e546,	0x61b02876,	0x528f8238,	0x61d09be5,
	0x5269126e,	0x61f1003f,	0x524295f0,	0x6211557e,	0x521c0cc2,	0x62319b9d,	0x51f576ea,	0x6251d298,
	0x51ced46e,	0x6271fa69,	0x51a82555,	0x6292130c,	0x518169a5,	0x62b21c7b,	0x515aa162,	0x62d216b3,
	0x5133cc94,	0x62f201ac,	0x510ceb40,	0x6311dd64,	0x50e5fd6d,	0x6331a9d4,	0x50bf031f,	0x635166f9,
	0x5097fc5e,	0x637114cc,	0x5070e92f,	0x6390b34a,	0x5049c999,	0x63b0426d,	0x50229da1,	0x63cfc231,
	0x4ffb654d,	0x63ef3290,	0x4fd420a4,	0x640e9386,	0x4faccfab,	0x642de50d,	0x4f857269,	0x644d2722,
	0x4f5e08e3,	0x646c59bf,	0x4f369320,	0x648b7ce0,	0x4f0f1126,	0x64aa907f,	0x4ee782fb,	0x64c99498,
	0x4ebfe8a5,	0x64e88926,	0x4e984229,	0x65076e25,	0x4e708f8f,	0x6526438f,	0x4e48d0dd,	0x6545095f,
	0x4e210617,	0x6563bf92,	0x4df92f46,	0x65826622,	0x4dd14c6e,	0x65a0fd0b,	0x4da95d96,	0x65bf8447,
	0x4d8162c4,	0x65ddfbd3,	0x4d595bfe,	0x65fc63a9,	0x4d31494b,	0x661abbc5,	0x4d092ab0,	0x66390422,
	0x4ce10034,	0x66573cbb,	0x4cb8c9dd,	0x6675658c,	0x4c9087b1,	0x66937e91,	0x4c6839b7,	0x66b187c3,
	0x4c3fdff4,	0x66cf8120,	0x4c177a6e,	0x66ed6aa1,	0x4bef092d,	0x670b4444,	0x4bc68c36,	0x67290e02,
	0x4b9e0390,	0x6746c7d8,	0x4b756f40,	0x676471c0,	0x4b4ccf4d,	0x67820bb7,	0x4b2423be,	0x679f95b7,
	0x4afb6c98,	0x67bd0fbd,	0x4ad2a9e2,	0x67da79c3,	0x4aa9dba2,	0x67f7d3c5,	0x4a8101de,	0x68151dbe,
	0x4a581c9e,	0x683257ab,	0x4a2f2be6,	0x684f8186,	0x4a062fbd,	0x686c9b4b,	0x49dd282a,	0x6889a4f6,
	0x49b41533,	0x68a69e81,	0x498af6df,	0x68c387e9,	0x4961cd33,	0x68e06129,	0x49389836,	0x68fd2a3d,
	0x490f57ee,	0x6919e320,	0x48e60c62,	0x69368bce,	0x48bcb599,	0x69532442,	0x48935397,	0x696fac78,
	0x4869e665,	0x698c246c,	0x48406e08,	0x69a88c19,	0x4816ea86,	0x69c4e37a,	0x47ed5be6,	0x69e12a8c,
	0x47c3c22f,	0x69fd614a,	0x479a1d67,	0x6a1987b0,	0x47706d93,	0x6a359db9,	0x4746b2bc,	0x6a51a361,
	0x471cece7,	0x6a6d98a4,	0x46f31c1a,	0x6a897d7d,	0x46c9405c,	0x6aa551e9,	0x469f59b4,	0x6ac115e2,
	0x46756828,	0x6adcc964,	0x464b6bbe,	0x6af86c6c,	0x4621647d,	0x6b13fef5,	0x45f7526b,	0x6b2f80fb,
	0x45cd358f,	0x6b4af279,	0x45a30df0,	0x6b66536b,	0x4578db93,	0x6b81a3cd,	0x454e9e80,	0x6b9ce39b,
	0x452456bd,	0x6bb812d1,	0x44fa0450,	0x6bd3316a,	0x44cfa740,	0x6bee3f62,	0x44a53f93,	0x6c093cb6,
	0x447acd50,	0x6c242960,	0x4450507e,	0x6c3f055d,	0x4425c923,	0x6c59d0a9,	0x43fb3746,	0x6c748b3f,
	0x43d09aed,	0x6c8f351c,	0x43a5f41e,	0x6ca9ce3b,	0x437b42e1,	0x6cc45698,	0x4350873c,	0x6cdece2f,
	0x4325c135,	0x6cf934fc,	0x42faf0d4,	0x6d138afb,	0x42d0161e,	0x6d2dd027,	0x42a5311b,	0x6d48047e,
	0x427a41d0,	0x6d6227fa,	0x424f4845,	0x6d7c3a98,	0x42244481,	0x6d963c54,	0x41f93689,	0x6db02d29,
	0x41ce1e65,	0x6dca0d14,	0x41a2fc1a,	0x6de3dc11,	0x4177cfb1,	0x6dfd9a1c,	0x414c992f,	0x6e174730,
	0x4121589b,	0x6e30e34a,	0x40f60dfb,	0x6e4a6e66,	0x40cab958,	0x6e63e87f,	0x409f5ab6,	0x6e7d5193,
	0x4073f21d,	0x6e96a99d,	0x40487f94,	0x6eaff099,	0x401d0321,	0x6ec92683,	0x3ff17cca,	0x6ee24b57,
	0x3fc5ec98,	0x6efb5f12,	0x3f9a5290,	0x6f1461b0,	0x3f6eaeb8,	0x6f2d532c,	0x3f430119,	0x6f463383,
	0x3f1749b8,	0x6f5f02b2,	0x3eeb889c,	0x6f77c0b3,	0x3ebfbdcd,	0x6f906d84,	0x3e93e950,	0x6fa90921,
	0x3e680b2c,	0x6fc19385,	0x3e3c2369,	0x6fda0cae,	0x3e10320d,	0x6ff27497,	0x3de4371f,	0x700acb3c,
	0x3db832a6,	0x7023109a,	0x3d8c24a8,	0x703b44ad,	0x3d600d2c,	0x70536771,	0x3d33ec39,	0x706b78e3,
	0x3d07c1d6,	0x708378ff,	0x3cdb8e09,	0x709b67c0,	0x3caf50da,	0x70b34525,	0x3c830a50,	0x70cb1128,
	0x3c56ba70,	0x70e2cbc6,	0x3c2a6142,	0x70fa74fc,	0x3bfdfecd,	0x71120cc5,	0x3bd19318,	0x7129931f,
	0x3ba51e29,	0x71410805,	0x3b78a007,	0x71586b74,	0x3b4c18ba,	0x716fbd68,	0x3b1f8848,	0x7186fdde,
	0x3af2eeb7,	0x719e2cd2,	0x3ac64c0f,	0x71b54a41,	0x3a99a057,	0x71cc5626,	0x3a6ceb96,	0x71e35080,
	0x3a402dd2,	0x71fa3949,	0x3a136712,	0x7211107e,	0x39e6975e,	0x7227d61c,	0x39b9bebc,	0x723e8a20,
	0x398cdd32,	0x72552c85,	0x395ff2c9,	0x726bbd48,	0x3932ff87,	0x72823c67,	0x39060373,	0x7298a9dd,
	0x38d8fe93,	0x72af05a7,	0x38abf0ef,	0x72c54fc1,	0x387eda8e,	0x72db8828,	0x3851bb77,	0x72f1aed9,
	0x382493b0,	0x7307c3d0,	0x37f76341,	0x731dc70a,	0x37ca2a30,	0x7333b883,	0x379ce885,	0x73499838,
	0x376f9e46,	0x735f6626,	0x37424b7b,	0x73752249,	0x3714f02a,	0x738acc9e,	0x36e78c5b,	0x73a06522,
	0x36ba2014,	0x73b5ebd1,	0x368cab5c,	0x73cb60a8,	0x365f2e3b,	0x73e0c3a3,	0x3631a8b8,	0x73f614c0,
	0x36041ad9,	0x740b53fb,	0x35d684a6,	0x74208150,	0x35a8e625,	0x74359cbd,	0x357b3f5d,	0x744aa63f,
	0x354d9057,	0x745f9dd1,	0x351fd918,	0x74748371,	0x34f219a8,	0x7489571c,	0x34c4520d,	0x749e18cd,
	0x34968250,	0x74b2c884,	0x3468aa76,	0x74c7663a,	0x343aca87,	0x74dbf1ef,	0x340ce28b,	0x74f06b9e,
	0x33def287,	0x7504d345,	0x33b0fa84,	0x751928e0,	0x3382fa88,	0x752d6c6c,	0x3354f29b,	0x75419de7,
	0x3326e2c3,	0x7555bd4c,	0x32f8cb07,	0x7569ca99,	0x32caab6f,	0x757dc5ca,	0x329c8402,	0x7591aedd,
	0x326e54c7,	0x75a585cf,	0x32401dc6,	0x75b94a9c,	0x3211df04,	0x75ccfd42,	0x31e39889,	0x75e09dbd,
	0x31b54a5e,	0x75f42c0b,	0x3186f487,	0x7607a828,	0x3158970e,	0x761b1211,	0x312a31f8,	0x762e69c4,
	0x30fbc54d,	0x7641af3d,	0x30cd5115,	0x7654e279,	0x309ed556,	0x76680376,	0x30705217,	0x767b1231,
	0x3041c761,	0x768e0ea6,	0x30133539,	0x76a0f8d2,	0x2fe49ba7,	0x76b3d0b4,	0x2fb5fab2,	0x76c69647,
	0x2f875262,	0x76d94989,	0x2f58a2be,	0x76ebea77,	0x2f29ebcc,	0x76fe790e,	0x2efb2d95,	0x7710f54c,
	0x2ecc681e,	0x77235f2d,	0x2e9d9b70,	0x7735b6af,	0x2e6ec792,	0x7747fbce,	0x2e3fec8b,	0x775a2e89,
	0x2e110a62,	0x776c4edb,	0x2de2211e,	0x777e5cc3,	0x2db330c7,	0x7790583e,	0x2d843964,	0x77a24148,
	0x2d553afc,	0x77b417df,	0x2d263596,	0x77c5dc01,	0x2cf72939,	0x77d78daa,	0x2cc815ee,	0x77e92cd9,
	0x2c98fbba,	0x77fab989,	0x2c69daa6,	0x780c33b8,	0x2c3ab2b9,	0x781d9b65,	0x2c0b83fa,	0x782ef08b,
	0x2bdc4e6f,	0x78403329,	0x2bad1221,	0x7851633b,	0x2b7dcf17,	0x786280bf,	0x2b4e8558,	0x78738bb3,
	0x2b1f34eb,	0x78848414,	0x2aefddd8,	0x789569df,	0x2ac08026,	0x78a63d11,	0x2a911bdc,	0x78b6fda8,
	0x2a61b101,	0x78c7aba2,	0x2a323f9e,	0x78d846fb,	0x2a02c7b8,	0x78e8cfb2,	0x29d34958,	0x78f945c3,
	0x29a3c485,	0x7909a92d,	0x29743946,	0x7919f9ec,	0x2944a7a2,	0x792a37fe,	0x29150fa1,	0x793a6361,
	0x28e5714b,	0x794a7c12,	0x28b5cca5,	0x795a820e,	0x288621b9,	0x796a7554,	0x2856708d,	0x797a55e0,
	0x2826b928,	0x798a23b1,	0x27f6fb92,	0x7999dec4,	0x27c737d3,	0x79a98715,	0x27976df1,	0x79b91ca4,
	0x27679df4,	0x79c89f6e,	0x2737c7e3,	0x79d80f6f,	0x2707ebc7,	0x79e76ca7,	0x26d809a5,	0x79f6b711,
	0x26a82186,	0x7a05eead,	0x26783370,	0x7a151378,	0x26483f6c,	0x7a24256f,	0x26184581,	0x7a332490,
	0x25e845b6,	0x7a4210d8,	0x25b84012,	0x7a50ea47,	0x2588349d,	0x7a5fb0d8,	0x2558235f,	0x7a6e648a,
	0x25280c5e,	0x7a7d055b,	0x24f7efa2,	0x7a8b9348,	0x24c7cd33,	0x7a9a0e50,	0x2497a517,	0x7aa8766f,
	0x24677758,	0x7ab6cba4,	0x243743fa,	0x7ac50dec,	0x24070b08,	0x7ad33d45,	0x23d6cc87,	0x7ae159ae,
	0x23a6887f,	0x7aef6323,	0x23763ef7,	0x7afd59a4,	0x2345eff8,	0x7b0b3d2c,	0x23159b88,	0x7b190dbc,
	0x22e541af,	0x7b26cb4f,	0x22b4e274,	0x7b3475e5,	0x22847de0,	0x7b420d7a,	0x225413f8,	0x7b4f920e,
	0x2223a4c5,	0x7b5d039e,	0x21f3304f,	0x7b6a6227,	0x21c2b69c,	0x7b77ada8,	0x219237b5,	0x7b84e61f,
	0x2161b3a0,	0x7b920b89,	0x21312a65,	0x7b9f1de6,	0x21009c0c,	0x7bac1d31,	0x20d0089c,	0x7bb9096b,
	0x209f701c,	0x7bc5e290,	0x206ed295,	0x7bd2a89e,	0x203e300d,	0x7bdf5b94,	0x200d888d,	0x7bebfb70,
	0x1fdcdc1b,	0x7bf88830,	0x1fac2abf,	0x7c0501d2,	0x1f7b7481,	0x7c116853,	0x1f4ab968,	0x7c1dbbb3,
	0x1f19f97b,	0x7c29fbee,	0x1ee934c3,	0x7c362904,	0x1eb86b46,	0x7c4242f2,	0x1e879d0d,	0x7c4e49b7,
	0x1e56ca1e,	0x7c5a3d50,	0x1e25f282,	0x7c661dbc,	0x1df5163f,	0x7c71eaf9,	0x1dc4355e,	0x7c7da505,
	0x1d934fe5,	0x7c894bde,	0x1d6265dd,	0x7c94df83,	0x1d31774d,	0x7ca05ff1,	0x1d00843d,	0x7cabcd28,
	0x1ccf8cb3,	0x7cb72724,	0x1c9e90b8,	0x7cc26de5,	0x1c6d9053,	0x7ccda169,	0x1c3c8b8c,	0x7cd8c1ae,
	0x1c0b826a,	0x7ce3ceb2,	0x1bda74f6,	0x7ceec873,	0x1ba96335,	0x7cf9aef0,	0x1b784d30,	0x7d048228,
	0x1b4732ef,	0x7d0f4218,	0x1b161479,	0x7d19eebf,	0x1ae4f1d6,	0x7d24881b,	0x1ab3cb0d,	0x7d2f0e2b,
	0x1a82a026,	0x7d3980ec,	0x1a517128,	0x7d43e05e,	0x1a203e1b,	0x7d4e2c7f,	0x19ef0707,	0x7d58654d,
	0x19bdcbf3,	0x7d628ac6,	0x198c8ce7,	0x7d6c9ce9,	0x195b49ea,	0x7d769bb5,	0x192a0304,	0x7d808728,
	0x18f8b83c,	0x7d8a5f40,	0x18c7699b,	0x7d9423fc,	0x18961728,	0x7d9dd55a,	0x1864c0ea,	0x7da77359,
	0x183366e9,	0x7db0fdf8,	0x1802092c,	0x7dba7534,	0x17d0a7bc,	0x7dc3d90d,	0x179f429f,	0x7dcd2981,
	0x176dd9de,	0x7dd6668f,	0x173c6d80,	0x7ddf9034,	0x170afd8d,	0x7de8a670,	0x16d98a0c,	0x7df1a942,
	0x16a81305,	0x7dfa98a8,	0x1676987f,	0x7e0374a0,	0x16451a83,	0x7e0c3d29,	0x16139918,	0x7e14f242,
	0x15e21445,	0x7e1d93ea,	0x15b08c12,	0x7e26221f,	0x157f0086,	0x7e2e9cdf,	0x154d71aa,	0x7e37042a,
	0x151bdf86,	0x7e3f57ff,	0x14ea4a1f,	0x7e47985b,	0x14b8b17f,	0x7e4fc53e,	0x148715ae,	0x7e57dea7,
	0x145576b1,	0x7e5fe493,	0x1423d492,	0x7e67d703,	0x13f22f58,	0x7e6fb5f4,	0x13c0870a,	0x7e778166,
	0x138edbb1,	0x7e7f3957,	0x135d2d53,	0x7e86ddc6,	0x132b7bf9,	0x7e8e6eb2,	0x12f9c7aa,	0x7e95ec1a,
	0x12c8106f,	0x7e9d55fc,	0x1296564d,	0x7ea4ac58,	0x1264994e,	0x7eabef2c,	0x1232d979,	0x7eb31e78,
	0x120116d5,	0x7eba3a39,	0x11cf516a,	0x7ec14270,	0x119d8941,	0x7ec8371a,	0x116bbe60,	0x7ecf1837,
	0x1139f0cf,	0x7ed5e5c6,	0x11082096,	0x7edc9fc6,	0x10d64dbd,	0x7ee34636,	0x10a4784b,	0x7ee9d914,
	0x1072a048,	0x7ef05860,	0x1040c5bb,	0x7ef6c418,	0x100ee8ad,	0x7efd1c3c,	0xfdd0926,	0x7f0360cb,
	0xfab272b,	0x7f0991c4,	0xf7942c7,	0x7f0faf25,	0xf475bff,	0x7f15b8ee,	0xf1572dc,	0x7f1baf1e,
	0xee38766,	0x7f2191b4,	0xeb199a4,	0x7f2760af,	0xe7fa99e,	0x7f2d1c0e,	0xe4db75b,	0x7f32c3d1,
	0xe1bc2e4,	0x7f3857f6,	0xde9cc40,	0x7f3dd87c,	0xdb7d376,	0x7f434563,	0xd85d88f,	0x7f489eaa,
	0xd53db92,	0x7f4de451,	0xd21dc87,	0x7f531655,	0xcefdb76,	0x7f5834b7,	0xcbdd865,	0x7f5d3f75,
	0xc8bd35e,	0x7f62368f,	0xc59cc68,	0x7f671a05,	0xc27c389,	0x7f6be9d4,	0xbf5b8cb,	0x7f70a5fe,
	0xbc3ac35,	0x7f754e80,	0xb919dcf,	0x7f79e35a,	0xb5f8d9f,	0x7f7e648c,	0xb2d7baf,	0x7f82d214,
	0xafb6805,	0x7f872bf3,	0xac952aa,	0x7f8b7227,	0xa973ba5,	0x7f8fa4b0,	0xa6522fe,	0x7f93c38c,
	0xa3308bd,	0x7f97cebd,	0xa00ece8,	0x7f9bc640,	0x9cecf89,	0x7f9faa15,	0x99cb0a7,	0x7fa37a3c,
	0x96a9049,	0x7fa736b4,	0x9386e78,	0x7faadf7c,	0x9064b3a,	0x7fae7495,	0x8d42699,	0x7fb1f5fc,
	0x8a2009a,	0x7fb563b3,	0x86fd947,	0x7fb8bdb8,	0x83db0a7,	0x7fbc040a,	0x80b86c2,	0x7fbf36aa,
	0x7d95b9e,	0x7fc25596,	0x7a72f45,	0x7fc560cf,	0x77501be,	0x7fc85854,	0x742d311,	0x7fcb3c23,
	0x710a345,	0x7fce0c3e,	0x6de7262,	0x7fd0c8a3,	0x6ac406f,	0x7fd37153,	0x67a0d76,	0x7fd6064c,
	0x647d97c,	0x7fd8878e,	0x615a48b,	0x7fdaf519,	0x5e36ea9,	0x7fdd4eec,	0x5b137df,	0x7fdf9508,
	0x57f0035,	0x7fe1c76b,	0x54cc7b1,	0x7fe3e616,	0x51a8e5c,	0x7fe5f108,	0x4e8543e,	0x7fe7e841,
	0x4b6195d,	0x7fe9cbc0,	0x483ddc3,	0x7feb9b85,	0x451a177,	0x7fed5791,	0x41f6480,	0x7feeffe1,
	0x3ed26e6,	0x7ff09478,	0x3bae8b2,	0x7ff21553,	0x388a9ea,	0x7ff38274,	0x3566a96,	0x7ff4dbd9,
	0x3242abf,	0x7ff62182,	0x2f1ea6c,	0x7ff75370,	0x2bfa9a4,	0x7ff871a2,	0x28d6870,	0x7ff97c18,
	0x25b26d7,	0x7ffa72d1,	0x228e4e2,	0x7ffb55ce,	0x1f6a297,	0x7ffc250f,	0x1c45ffe,	0x7ffce093,
	0x1921d20,	0x7ffd885a,	0x15fda03,	0x7ffe1c65,	0x12d96b1,	0x7ffe9cb2,	0xfb5330,	0x7fff0943,
	0xc90f88,	0x7fff6216,	0x96cbc1,	0x7fffa72c,	0x6487e3,	0x7fffd886,	0x3243f5,	0x7ffff621,
	0x0,	0x7fffffff,	0xffcdbc0b,	0x7ffff621,	0xff9b781d,	0x7fffd886,	0xff69343f,	0x7fffa72c,
	0xff36f078,	0x7fff6216,	0xff04acd0,	0x7fff0943,	0xfed2694f,	0x7ffe9cb2,	0xfea025fd,	0x7ffe1c65,
	0xfe6de2e0,	0x7ffd885a,	0xfe3ba002,	0x7ffce093,	0xfe095d69,	0x7ffc250f,	0xfdd71b1e,	0x7ffb55ce,
	0xfda4d929,	0x7ffa72d1,	0xfd729790,	0x7ff97c18,	0xfd40565c,	0x7ff871a2,	0xfd0e1594,	0x7ff75370,
	0xfcdbd541,	0x7ff62182,	0xfca9956a,	0x7ff4dbd9,	0xfc775616,	0x7ff38274,	0xfc45174e,	0x7ff21553,
	0xfc12d91a,	0x7ff09478,	0xfbe09b80,	0x7feeffe1,	0xfbae5e89,	0x7fed5791,	0xfb7c223d,	0x7feb9b85,
	0xfb49e6a3,	0x7fe9cbc0,	0xfb17abc2,	0x7fe7e841,	0xfae571a4,	0x7fe5f108,	0xfab3384f,	0x7fe3e616,
	0xfa80ffcb,	0x7fe1c76b,	0xfa4ec821,	0x7fdf9508,	0xfa1c9157,	0x7fdd4eec,	0xf9ea5b75,	0x7fdaf519,
	0xf9b82684,	0x7fd8878e,	0xf985f28a,	0x7fd6064c,	0xf953bf91,	0x7fd37153,	0xf9218d9e,	0x7fd0c8a3,
	0xf8ef5cbb,	0x7fce0c3e,	0xf8bd2cef,	0x7fcb3c23,	0xf88afe42,	0x7fc85854,	0xf858d0bb,	0x7fc560cf,
	0xf826a462,	0x7fc25596,	0xf7f4793e,	0x7fbf36aa,	0xf7c24f59,	0x7fbc040a,	0xf79026b9,	0x7fb8bdb8,
	0xf75dff66,	0x7fb563b3,	0xf72bd967,	0x7fb1f5fc,	0xf6f9b4c6,	0x7fae7495,	0xf6c79188,	0x7faadf7c,
	0xf6956fb7,	0x7fa736b4,	0xf6634f59,	0x7fa37a3c,	0xf6313077,	0x7f9faa15,	0xf5ff1318,	0x7f9bc640,
	0xf5ccf743,	0x7f97cebd,	0xf59add02,	0x7f93c38c,	0xf568c45b,	0x7f8fa4b0,	0xf536ad56,	0x7f8b7227,
	0xf50497fb,	0x7f872bf3,	0xf4d28451,	0x7f82d214,	0xf4a07261,	0x7f7e648c,	0xf46e6231,	0x7f79e35a,
	0xf43c53cb,	0x7f754e80,	0xf40a4735,	0x7f70a5fe,	0xf3d83c77,	0x7f6be9d4,	0xf3a63398,	0x7f671a05,
	0xf3742ca2,	0x7f62368f,	0xf342279b,	0x7f5d3f75,	0xf310248a,	0x7f5834b7,	0xf2de2379,	0x7f531655,
	0xf2ac246e,	0x7f4de451,	0xf27a2771,	0x7f489eaa,	0xf2482c8a,	0x7f434563,	0xf21633c0,	0x7f3dd87c,
	0xf1e43d1c,	0x7f3857f6,	0xf1b248a5,	0x7f32c3d1,	0xf1805662,	0x7f2d1c0e,	0xf14e665c,	0x7f2760af,
	0xf11c789a,	0x7f2191b4,	0xf0ea8d24,	0x7f1baf1e,	0xf0b8a401,	0x7f15b8ee,	0xf086bd39,	0x7f0faf25,
	0xf054d8d5,	0x7f0991c4,	0xf022f6da,	0x7f0360cb,	0xeff11753,	0x7efd1c3c,	0xefbf3a45,	0x7ef6c418,
	0xef8d5fb8,	0x7ef05860,	0xef5b87b5,	0x7ee9d914,	0xef29b243,	0x7ee34636,	0xeef7df6a,	0x7edc9fc6,
	0xeec60f31,	0x7ed5e5c6,	0xee9441a0,	0x7ecf1837,	0xee6276bf,	0x7ec8371a,	0xee30ae96,	0x7ec14270,
	0xedfee92b,	0x7eba3a39,	0xedcd2687,	0x7eb31e78,	0xed9b66b2,	0x7eabef2c,	0xed69a9b3,	0x7ea4ac58,
	0xed37ef91,	0x7e9d55fc,	0xed063856,	0x7e95ec1a,	0xecd48407,	0x7e8e6eb2,	0xeca2d2ad,	0x7e86ddc6,
	0xec71244f,	0x7e7f3957,	0xec3f78f6,	0x7e778166,	0xec0dd0a8,	0x7e6fb5f4,	0xebdc2b6e,	0x7e67d703,
	0xebaa894f,	0x7e5fe493,	0xeb78ea52,	0x7e57dea7,	0xeb474e81,	0x7e4fc53e,	0xeb15b5e1,	0x7e47985b,
	0xeae4207a,	0x7e3f57ff,	0xeab28e56,	0x7e37042a,	0xea80ff7a,	0x7e2e9cdf,	0xea4f73ee,	0x7e26221f,
	0xea1debbb,	0x7e1d93ea,	0xe9ec66e8,	0x7e14f242,	0xe9bae57d,	0x7e0c3d29,	0xe9896781,	0x7e0374a0,
	0xe957ecfb,	0x7dfa98a8,	0xe92675f4,	0x7df1a942,	0xe8f50273,	0x7de8a670,	0xe8c39280,	0x7ddf9034,
	0xe8922622,	0x7dd6668f,	0xe860bd61,	0x7dcd2981,	0xe82f5844,	0x7dc3d90d,	0xe7fdf6d4,	0x7dba7534,
	0xe7cc9917,	0x7db0fdf8,	0xe79b3f16,	0x7da77359,	0xe769e8d8,	0x7d9dd55a,	0xe7389665,	0x7d9423fc,
	0xe70747c4,	0x7d8a5f40,	0xe6d5fcfc,	0x7d808728,	0xe6a4b616,	0x7d769bb5,	0xe6737319,	0x7d6c9ce9,
	0xe642340d,	0x7d628ac6,	0xe610f8f9,	0x7d58654d,	0xe5dfc1e5,	0x7d4e2c7f,	0xe5ae8ed8,	0x7d43e05e,
	0xe57d5fda,	0x7d3980ec,	0xe54c34f3,	0x7d2f0e2b,	0xe51b0e2a,	0x7d24881b,	0xe4e9eb87,	0x7d19eebf,
	0xe4b8cd11,	0x7d0f4218,	0xe487b2d0,	0x7d048228,	0xe4569ccb,	0x7cf9aef0,	0xe4258b0a,	0x7ceec873,
	0xe3f47d96,	0x7ce3ceb2,	0xe3c37474,	0x7cd8c1ae,	0xe3926fad,	0x7ccda169,	0xe3616f48,	0x7cc26de5,
	0xe330734d,	0x7cb72724,	0xe2ff7bc3,	0x7cabcd28,	0xe2ce88b3,	0x7ca05ff1,	0xe29d9a23,	0x7c94df83,
	0xe26cb01b,	0x7c894bde,	0xe23bcaa2,	0x7c7da505,	0xe20ae9c1,	0x7c71eaf9,	0xe1da0d7e,	0x7c661dbc,
	0xe1a935e2,	0x7c5a3d50,	0xe17862f3,	0x7c4e49b7,	0xe14794ba,	0x7c4242f2,	0xe116cb3d,	0x7c362904,
	0xe0e60685,	0x7c29fbee,	0xe0b54698,	0x7c1dbbb3,	0xe0848b7f,	0x7c116853,	0xe053d541,	0x7c0501d2,
	0xe02323e5,	0x7bf88830,	0xdff27773,	0x7bebfb70,	0xdfc1cff3,	0x7bdf5b94,	0xdf912d6b,	0x7bd2a89e,
	0xdf608fe4,	0x7bc5e290,	0xdf2ff764,	0x7bb9096b,	0xdeff63f4,	0x7bac1d31,	0xdeced59b,	0x7b9f1de6,
	0xde9e4c60,	0x7b920b89,	0xde6dc84b,	0x7b84e61f,	0xde3d4964,	0x7b77ada8,	0xde0ccfb1,	0x7b6a6227,
	0xdddc5b3b,	0x7b5d039e,	0xddabec08,	0x7b4f920e,	0xdd7b8220,	0x7b420d7a,	0xdd4b1d8c,	0x7b3475e5,
	0xdd1abe51,	0x7b26cb4f,	0xdcea6478,	0x7b190dbc,	0xdcba1008,	0x7b0b3d2c,	0xdc89c109,	0x7afd59a4,
	0xdc597781,	0x7aef6323,	0xdc293379,	0x7ae159ae,	0xdbf8f4f8,	0x7ad33d45,	0xdbc8bc06,	0x7ac50dec,
	0xdb9888a8,	0x7ab6cba4,	0xdb685ae9,	0x7aa8766f,	0xdb3832cd,	0x7a9a0e50,	0xdb08105e,	0x7a8b9348,
	0xdad7f3a2,	0x7a7d055b,	0xdaa7dca1,	0x7a6e648a,	0xda77cb63,	0x7a5fb0d8,	0xda47bfee,	0x7a50ea47,
	0xda17ba4a,	0x7a4210d8,	0xd9e7ba7f,	0x7a332490,	0xd9b7c094,	0x7a24256f,	0xd987cc90,	0x7a151378,
	0xd957de7a,	0x7a05eead,	0xd927f65b,	0x79f6b711,	0xd8f81439,	0x79e76ca7,	0xd8c8381d,	0x79d80f6f,
	0xd898620c,	0x79c89f6e,	0xd868920f,	0x79b91ca4,	0xd838c82d,	0x79a98715,	0xd809046e,	0x7999dec4,
	0xd7d946d8,	0x798a23b1,	0xd7a98f73,	0x797a55e0,	0xd779de47,	0x796a7554,	0xd74a335b,	0x795a820e,
	0xd71a8eb5,	0x794a7c12,	0xd6eaf05f,	0x793a6361,	0xd6bb585e,	0x792a37fe,	0xd68bc6ba,	0x7919f9ec,
	0xd65c3b7b,	0x7909a92d,	0xd62cb6a8,	0x78f945c3,	0xd5fd3848,	0x78e8cfb2,	0xd5cdc062,	0x78d846fb,
	0xd59e4eff,	0x78c7aba2,	0xd56ee424,	0x78b6fda8,	0xd53f7fda,	0x78a63d11,	0xd5102228,	0x789569df,
	0xd4e0cb15,	0x78848414,	0xd4b17aa8,	0x78738bb3,	0xd48230e9,	0x786280bf,	0xd452eddf,	0x7851633b,
	0xd423b191,	0x78403329,	0xd3f47c06,	0x782ef08b,	0xd3c54d47,	0x781d9b65,	0xd396255a,	0x780c33b8,
	0xd3670446,	0x77fab989,	0xd337ea12,	0x77e92cd9,	0xd308d6c7,	0x77d78daa,	0xd2d9ca6a,	0x77c5dc01,
	0xd2aac504,	0x77b417df,	0xd27bc69c,	0x77a24148,	0xd24ccf39,	0x7790583e,	0xd21ddee2,	0x777e5cc3,
	0xd1eef59e,	0x776c4edb,	0xd1c01375,	0x775a2e89,	0xd191386e,	0x7747fbce,	0xd1626490,	0x7735b6af,
	0xd13397e2,	0x77235f2d,	0xd104d26b,	0x7710f54c,	0xd0d61434,	0x76fe790e,	0xd0a75d42,	0x76ebea77,
	0xd078ad9e,	0x76d94989,	0xd04a054e,	0x76c69647,	0xd01b6459,	0x76b3d0b4,	0xcfeccac7,	0x76a0f8d2,
	0xcfbe389f,	0x768e0ea6,	0xcf8fade9,	0x767b1231,	0xcf612aaa,	0x76680376,	0xcf32aeeb,	0x7654e279,
	0xcf043ab3,	0x7641af3d,	0xced5ce08,	0x762e69c4,	0xcea768f2,	0x761b1211,	0xce790b79,	0x7607a828,
	0xce4ab5a2,	0x75f42c0b,	0xce1c6777,	0x75e09dbd,	0xcdee20fc,	0x75ccfd42,	0xcdbfe23a,	0x75b94a9c,
	0xcd91ab39,	0x75a585cf,	0xcd637bfe,	0x7591aedd,	0xcd355491,	0x757dc5ca,	0xcd0734f9,	0x7569ca99,
	0xccd91d3d,	0x7555bd4c,	0xccab0d65,	0x75419de7,	0xcc7d0578,	0x752d6c6c,	0xcc4f057c,	0x751928e0,
	0xcc210d79,	0x7504d345,	0xcbf31d75,	0x74f06b9e,	0xcbc53579,	0x74dbf1ef,	0xcb97558a,	0x74c7663a,
	0xcb697db0,	0x74b2c884,	0xcb3badf3,	0x749e18cd,	0xcb0de658,	0x7489571c,	0xcae026e8,	0x74748371,
	0xcab26fa9,	0x745f9dd1,	0xca84c0a3,	0x744aa63f,	0xca5719db,	0x74359cbd,	0xca297b5a,	0x74208150,
	0xc9fbe527,	0x740b53fb,	0xc9ce5748,	0x73f614c0,	0xc9a0d1c5,	0x73e0c3a3,	0xc97354a4,	0x73cb60a8,
	0xc945dfec,	0x73b5ebd1,	0xc91873a5,	0x73a06522,	0xc8eb0fd6,	0x738acc9e,	0xc8bdb485,	0x73752249,
	0xc89061ba,	0x735f6626,	0xc863177b,	0x73499838,	0xc835d5d0,	0x7333b883,	0xc8089cbf,	0x731dc70a,
	0xc7db6c50,	0x7307c3d0,	0xc7ae4489,	0x72f1aed9,	0xc7812572,	0x72db8828,	0xc7540f11,	0x72c54fc1,
	0xc727016d,	0x72af05a7,	0xc6f9fc8d,	0x7298a9dd,	0xc6cd0079,	0x72823c67,	0xc6a00d37,	0x726bbd48,
	0xc67322ce,	0x72552c85,	0xc6464144,	0x723e8a20,	0xc61968a2,	0x7227d61c,	0xc5ec98ee,	0x7211107e,
	0xc5bfd22e,	0x71fa3949,	0xc593146a,	0x71e35080,	0xc5665fa9,	0x71cc5626,	0xc539b3f1,	0x71b54a41,
	0xc50d1149,	0x719e2cd2,	0xc4e077b8,	0x7186fdde,	0xc4b3e746,	0x716fbd68,	0xc4875ff9,	0x71586b74,
	0xc45ae1d7,	0x71410805,	0xc42e6ce8,	0x7129931f,	0xc4020133,	0x71120cc5,	0xc3d59ebe,	0x70fa74fc,
	0xc3a94590,	0x70e2cbc6,	0xc37cf5b0,	0x70cb1128,	0xc350af26,	0x70b34525,	0xc32471f7,	0x709b67c0,
	0xc2f83e2a,	0x708378ff,	0xc2cc13c7,	0x706b78e3,	0xc29ff2d4,	0x70536771,	0xc273db58,	0x703b44ad,
	0xc247cd5a,	0x7023109a,	0xc21bc8e1,	0x700acb3c,	0xc1efcdf3,	0x6ff27497,	0xc1c3dc97,	0x6fda0cae,
	0xc197f4d4,	0x6fc19385,	0xc16c16b0,	0x6fa90921,	0xc1404233,	0x6f906d84,	0xc1147764,	0x6f77c0b3,
	0xc0e8b648,	0x6f5f02b2,	0xc0bcfee7,	0x6f463383,	0xc0915148,	0x6f2d532c,	0xc065ad70,	0x6f1461b0,
	0xc03a1368,	0x6efb5f12,	0xc00e8336,	0x6ee24b57,	0xbfe2fcdf,	0x6ec92683,	0xbfb7806c,	0x6eaff099,
	0xbf8c0de3,	0x6e96a99d,	0xbf60a54a,	0x6e7d5193,	0xbf3546a8,	0x6e63e87f,	0xbf09f205,	0x6e4a6e66,
	0xbedea765,	0x6e30e34a,	0xbeb366d1,	0x6e174730,	0xbe88304f,	0x6dfd9a1c,	0xbe5d03e6,	0x6de3dc11,
	0xbe31e19b,	0x6dca0d14,	0xbe06c977,	0x6db02d29,	0xbddbbb7f,	0x6d963c54,	0xbdb0b7bb,	0x6d7c3a98,
	0xbd85be30,	0x6d6227fa,	0xbd5acee5,	0x6d48047e,	0xbd2fe9e2,	0x6d2dd027,	0xbd050f2c,	0x6d138afb,
	0xbcda3ecb,	0x6cf934fc,	0xbcaf78c4,	0x6cdece2f,	0xbc84bd1f,	0x6cc45698,	0xbc5a0be2,	0x6ca9ce3b,
	0xbc2f6513,	0x6c8f351c,	0xbc04c8ba,	0x6c748b3f,	0xbbda36dd,	0x6c59d0a9,	0xbbafaf82,	0x6c3f055d,
	0xbb8532b0,	0x6c242960,	0xbb5ac06d,	0x6c093cb6,	0xbb3058c0,	0x6bee3f62,	0xbb05fbb0,	0x6bd3316a,
	0xbadba943,	0x6bb812d1,	0xbab16180,	0x6b9ce39b,	0xba87246d,	0x6b81a3cd,	0xba5cf210,	0x6b66536b,
	0xba32ca71,	0x6b4af279,	0xba08ad95,	0x6b2f80fb,	0xb9de9b83,	0x6b13fef5,	0xb9b49442,	0x6af86c6c,
	0xb98a97d8,	0x6adcc964,	0xb960a64c,	0x6ac115e2,	0xb936bfa4,	0x6aa551e9,	0xb90ce3e6,	0x6a897d7d,
	0xb8e31319,	0x6a6d98a4,	0xb8b94d44,	0x6a51a361,	0xb88f926d,	0x6a359db9,	0xb865e299,	0x6a1987b0,
	0xb83c3dd1,	0x69fd614a,	0xb812a41a,	0x69e12a8c,	0xb7e9157a,	0x69c4e37a,	0xb7bf91f8,	0x69a88c19,
	0xb796199b,	0x698c246c,	0xb76cac69,	0x696fac78,	0xb7434a67,	0x69532442,	0xb719f39e,	0x69368bce,
	0xb6f0a812,	0x6919e320,	0xb6c767ca,	0x68fd2a3d,	0xb69e32cd,	0x68e06129,	0xb6750921,	0x68c387e9,
	0xb64beacd,	0x68a69e81,	0xb622d7d6,	0x6889a4f6,	0xb5f9d043,	0x686c9b4b,	0xb5d0d41a,	0x684f8186,
	0xb5a7e362,	0x683257ab,	0xb57efe22,	0x68151dbe,	0xb556245e,	0x67f7d3c5,	0xb52d561e,	0x67da79c3,
	0xb5049368,	0x67bd0fbd,	0xb4dbdc42,	0x679f95b7,	0xb4b330b3,	0x67820bb7,	0xb48a90c0,	0x676471c0,
	0xb461fc70,	0x6746c7d8,	0xb43973ca,	0x67290e02,	0xb410f6d3,	0x670b4444,	0xb3e88592,	0x66ed6aa1,
	0xb3c0200c,	0x66cf8120,	0xb397c649,	0x66b187c3,	0xb36f784f,	0x66937e91,	0xb3473623,	0x6675658c,
	0xb31effcc,	0x66573cbb,	0xb2f6d550,	0x66390422,	0xb2ceb6b5,	0x661abbc5,	0xb2a6a402,	0x65fc63a9,
	0xb27e9d3c,	0x65ddfbd3,	0xb256a26a,	0x65bf8447,	0xb22eb392,	0x65a0fd0b,	0xb206d0ba,	0x65826622,
	0xb1def9e9,	0x6563bf92,	0xb1b72f23,	0x6545095f,	0xb18f7071,	0x6526438f,	0xb167bdd7,	0x65076e25,
	0xb140175b,	0x64e88926,	0xb1187d05,	0x64c99498,	0xb0f0eeda,	0x64aa907f,	0xb0c96ce0,	0x648b7ce0,
	0xb0a1f71d,	0x646c59bf,	0xb07a8d97,	0x644d2722,	0xb0533055,	0x642de50d,	0xb02bdf5c,	0x640e9386,
	0xb0049ab3,	0x63ef3290,	0xafdd625f,	0x63cfc231,	0xafb63667,	0x63b0426d,	0xaf8f16d1,	0x6390b34a,
	0xaf6803a2,	0x637114cc,	0xaf40fce1,	0x635166f9,	0xaf1a0293,	0x6331a9d4,	0xaef314c0,	0x6311dd64,
	0xaecc336c,	0x62f201ac,	0xaea55e9e,	0x62d216b3,	0xae7e965b,	0x62b21c7b,	0xae57daab,	0x6292130c,
	0xae312b92,	0x6271fa69,	0xae0a8916,	0x6251d298,	0xade3f33e,	0x62319b9d,	0xadbd6a10,	0x6211557e,
	0xad96ed92,	0x61f1003f,	0xad707dc8,	0x61d09be5,	0xad4a1aba,	0x61b02876,	0xad23c46e,	0x618fa5f7,
	0xacfd7ae8,	0x616f146c,	0xacd73e30,	0x614e73da,	0xacb10e4b,	0x612dc447,	0xac8aeb3e,	0x610d05b7,
	0xac64d510,	0x60ec3830,	0xac3ecbc7,	0x60cb5bb7,	0xac18cf69,	0x60aa7050,	0xabf2dffb,	0x60897601,
	0xabccfd83,	0x60686ccf,	0xaba72807,	0x604754bf,	0xab815f8d,	0x60262dd6,	0xab5ba41a,	0x6004f819,
	0xab35f5b5,	0x5fe3b38d,	0xab105464,	0x5fc26038,	0xaaeac02c,	0x5fa0fe1f,	0xaac53912,	0x5f7f8d46,
	0xaa9fbf1e,	0x5f5e0db3,	0xaa7a5253,	0x5f3c7f6b,	0xaa54f2ba,	0x5f1ae274,	0xaa2fa056,	0x5ef936d1,
	0xaa0a5b2e,	0x5ed77c8a,	0xa9e52347,	0x5eb5b3a2,	0xa9bff8a8,	0x5e93dc1f,	0xa99adb56,	0x5e71f606,
	0xa975cb57,	0x5e50015d,	0xa950c8b0,	0x5e2dfe29,	0xa92bd367,	0x5e0bec6e,	0xa906eb82,	0x5de9cc33,
	0xa8e21106,	0x5dc79d7c,	0xa8bd43fa,	0x5da5604f,	0xa8988463,	0x5d8314b1,	0xa873d246,	0x5d60baa7,
	0xa84f2daa,	0x5d3e5237,	0xa82a9693,	0x5d1bdb65,	0xa8060d08,	0x5cf95638,	0xa7e1910f,	0x5cd6c2b5,
	0xa7bd22ac,	0x5cb420e0,	0xa798c1e5,	0x5c9170bf,	0xa7746ec0,	0x5c6eb258,	0xa7502943,	0x5c4be5b0,
	0xa72bf174,	0x5c290acc,	0xa707c757,	0x5c0621b2,	0xa6e3aaf2,	0x5be32a67,	0xa6bf9c4b,	0x5bc024f0,
	0xa69b9b68,	0x5b9d1154,	0xa677a84e,	0x5b79ef96,	0xa653c303,	0x5b56bfbd,	0xa62feb8b,	0x5b3381ce,
	0xa60c21ee,	0x5b1035cf,	0xa5e8662f,	0x5aecdbc5,	0xa5c4b855,	0x5ac973b5,	0xa5a11866,	0x5aa5fda5,
	0xa57d8666,	0x5a82799a,	0xa55a025b,	0x5a5ee79a,	0xa5368c4b,	0x5a3b47ab,	0xa513243b,	0x5a1799d1,
	0xa4efca31,	0x59f3de12,	0xa4cc7e32,	0x59d01475,	0xa4a94043,	0x59ac3cfd,	0xa486106a,	0x598857b2,
	0xa462eeac,	0x59646498,	0xa43fdb10,	0x594063b5,	0xa41cd599,	0x591c550e,	0xa3f9de4e,	0x58f838a9,
	0xa3d6f534,	0x58d40e8c,	0xa3b41a50,	0x58afd6bd,	0xa3914da8,	0x588b9140,	0xa36e8f41,	0x58673e1b,
	0xa34bdf20,	0x5842dd54,	0xa3293d4b,	0x581e6ef1,	0xa306a9c8,	0x57f9f2f8,	0xa2e4249b,	0x57d5696d,
	0xa2c1adc9,	0x57b0d256,	0xa29f4559,	0x578c2dba,	0xa27ceb4f,	0x57677b9d,	0xa25a9fb1,	0x5742bc06,
	0xa2386284,	0x571deefa,	0xa21633cd,	0x56f9147e,	0xa1f41392,	0x56d42c99,	0xa1d201d7,	0x56af3750,
	0xa1affea3,	0x568a34a9,	0xa18e09fa,	0x566524aa,	0xa16c23e1,	0x56400758,	0xa14a4c5e,	0x561adcb9,
	0xa1288376,	0x55f5a4d2,	0xa106c92f,	0x55d05faa,	0xa0e51d8c,	0x55ab0d46,	0xa0c38095,	0x5585adad,
	0xa0a1f24d,	0x556040e2,	0xa08072ba,	0x553ac6ee,	0xa05f01e1,	0x55153fd4,	0xa03d9fc8,	0x54efab9c,
	0xa01c4c73,	0x54ca0a4b,	0x9ffb07e7,	0x54a45be6,	0x9fd9d22a,	0x547ea073,	0x9fb8ab41,	0x5458d7f9,
	0x9f979331,	0x5433027d,	0x9f7689ff,	0x540d2005,	0x9f558fb0,	0x53e73097,	0x9f34a449,	0x53c13439,
	0x9f13c7d0,	0x539b2af0,	0x9ef2fa49,	0x537514c2,	0x9ed23bb9,	0x534ef1b5,	0x9eb18c26,	0x5328c1d0,
	0x9e90eb94,	0x53028518,	0x9e705a09,	0x52dc3b92,	0x9e4fd78a,	0x52b5e546,	0x9e2f641b,	0x528f8238,
	0x9e0effc1,	0x5269126e,	0x9deeaa82,	0x524295f0,	0x9dce6463,	0x521c0cc2,	0x9dae2d68,	0x51f576ea,
	0x9d8e0597,	0x51ced46e,	0x9d6decf4,	0x51a82555,	0x9d4de385,	0x518169a5,	0x9d2de94d,	0x515aa162,
	0x9d0dfe54,	0x5133cc94,	0x9cee229c,	0x510ceb40,	0x9cce562c,	0x50e5fd6d,	0x9cae9907,	0x50bf031f,
	0x9c8eeb34,	0x5097fc5e,	0x9c6f4cb6,	0x5070e92f,	0x9c4fbd93,	0x5049c999,	0x9c303dcf,	0x50229da1,
	0x9c10cd70,	0x4ffb654d,	0x9bf16c7a,	0x4fd420a4,	0x9bd21af3,	0x4faccfab,	0x9bb2d8de,	0x4f857269,
	0x9b93a641,	0x4f5e08e3,	0x9b748320,	0x4f369320,	0x9b556f81,	0x4f0f1126,	0x9b366b68,	0x4ee782fb,
	0x9b1776da,	0x4ebfe8a5,	0x9af891db,	0x4e984229,	0x9ad9bc71,	0x4e708f8f,	0x9abaf6a1,	0x4e48d0dd,
	0x9a9c406e,	0x4e210617,	0x9a7d99de,	0x4df92f46,	0x9a5f02f5,	0x4dd14c6e,	0x9a407bb9,	0x4da95d96,
	0x9a22042d,	0x4d8162c4,	0x9a039c57,	0x4d595bfe,	0x99e5443b,	0x4d31494b,	0x99c6fbde,	0x4d092ab0,
	0x99a8c345,	0x4ce10034,	0x998a9a74,	0x4cb8c9dd,	0x996c816f,	0x4c9087b1,	0x994e783d,	0x4c6839b7,
	0x99307ee0,	0x4c3fdff4,	0x9912955f,	0x4c177a6e,	0x98f4bbbc,	0x4bef092d,	0x98d6f1fe,	0x4bc68c36,
	0x98b93828,	0x4b9e0390,	0x989b8e40,	0x4b756f40,	0x987df449,	0x4b4ccf4d,	0x98606a49,	0x4b2423be,
	0x9842f043,	0x4afb6c98,	0x9825863d,	0x4ad2a9e2,	0x98082c3b,	0x4aa9dba2,	0x97eae242,	0x4a8101de,
	0x97cda855,	0x4a581c9e,	0x97b07e7a,	0x4a2f2be6,	0x979364b5,	0x4a062fbd,	0x97765b0a,	0x49dd282a,
	0x9759617f,	0x49b41533,	0x973c7817,	0x498af6df,	0x971f9ed7,	0x4961cd33,	0x9702d5c3,	0x49389836,
	0x96e61ce0,	0x490f57ee,	0x96c97432,	0x48e60c62,	0x96acdbbe,	0x48bcb599,	0x96905388,	0x48935397,
	0x9673db94,	0x4869e665,	0x965773e7,	0x48406e08,	0x963b1c86,	0x4816ea86,	0x961ed574,	0x47ed5be6,
	0x96029eb6,	0x47c3c22f,	0x95e67850,	0x479a1d67,	0x95ca6247,	0x47706d93,	0x95ae5c9f,	0x4746b2bc,
	0x9592675c,	0x471cece7,	0x95768283,	0x46f31c1a,	0x955aae17,	0x46c9405c,	0x953eea1e,	0x469f59b4,
	0x9523369c,	0x46756828,	0x95079394,	0x464b6bbe,	0x94ec010b,	0x4621647d,	0x94d07f05,	0x45f7526b,
	0x94b50d87,	0x45cd358f,	0x9499ac95,	0x45a30df0,	0x947e5c33,	0x4578db93,	0x94631c65,	0x454e9e80,
	0x9447ed2f,	0x452456bd,	0x942cce96,	0x44fa0450,	0x9411c09e,	0x44cfa740,	0x93f6c34a,	0x44a53f93,
	0x93dbd6a0,	0x447acd50,	0x93c0faa3,	0x4450507e,	0x93a62f57,	0x4425c923,	0x938b74c1,	0x43fb3746,
	0x9370cae4,	0x43d09aed,	0x935631c5,	0x43a5f41e,	0x933ba968,	0x437b42e1,	0x932131d1,	0x4350873c,
	0x9306cb04,	0x4325c135,	0x92ec7505,	0x42faf0d4,	0x92d22fd9,	0x42d0161e,	0x92b7fb82,	0x42a5311b,
	0x929dd806,	0x427a41d0,	0x9283c568,	0x424f4845,	0x9269c3ac,	0x42244481,	0x924fd2d7,	0x41f93689,
	0x9235f2ec,	0x41ce1e65,	0x921c23ef,	0x41a2fc1a,	0x920265e4,	0x4177cfb1,	0x91e8b8d0,	0x414c992f,
	0x91cf1cb6,	0x4121589b,	0x91b5919a,	0x40f60dfb,	0x919c1781,	0x40cab958,	0x9182ae6d,	0x409f5ab6,
	0x91695663,	0x4073f21d,	0x91500f67,	0x40487f94,	0x9136d97d,	0x401d0321,	0x911db4a9,	0x3ff17cca,
	0x9104a0ee,	0x3fc5ec98,	0x90eb9e50,	0x3f9a5290,	0x90d2acd4,	0x3f6eaeb8,	0x90b9cc7d,	0x3f430119,
	0x90a0fd4e,	0x3f1749b8,	0x90883f4d,	0x3eeb889c,	0x906f927c,	0x3ebfbdcd,	0x9056f6df,	0x3e93e950,
	0x903e6c7b,	0x3e680b2c,	0x9025f352,	0x3e3c2369,	0x900d8b69,	0x3e10320d,	0x8ff534c4,	0x3de4371f,
	0x8fdcef66,	0x3db832a6,	0x8fc4bb53,	0x3d8c24a8,	0x8fac988f,	0x3d600d2c,	0x8f94871d,	0x3d33ec39,
	0x8f7c8701,	0x3d07c1d6,	0x8f649840,	0x3cdb8e09,	0x8f4cbadb,	0x3caf50da,	0x8f34eed8,	0x3c830a50,
	0x8f1d343a,	0x3c56ba70,	0x8f058b04,	0x3c2a6142,	0x8eedf33b,	0x3bfdfecd,	0x8ed66ce1,	0x3bd19318,
	0x8ebef7fb,	0x3ba51e29,	0x8ea7948c,	0x3b78a007,	0x8e904298,	0x3b4c18ba,	0x8e790222,	0x3b1f8848,
	0x8e61d32e,	0x3af2eeb7,	0x8e4ab5bf,	0x3ac64c0f,	0x8e33a9da,	0x3a99a057,	0x8e1caf80,	0x3a6ceb96,
	0x8e05c6b7,	0x3a402dd2,	0x8deeef82,	0x3a136712,	0x8dd829e4,	0x39e6975e,	0x8dc175e0,	0x39b9bebc,
	0x8daad37b,	0x398cdd32,	0x8d9442b8,	0x395ff2c9,	0x8d7dc399,	0x3932ff87,	0x8d675623,	0x39060373,
	0x8d50fa59,	0x38d8fe93,	0x8d3ab03f,	0x38abf0ef,	0x8d2477d8,	0x387eda8e,	0x8d0e5127,	0x3851bb77,
	0x8cf83c30,	0x382493b0,	0x8ce238f6,	0x37f76341,	0x8ccc477d,	0x37ca2a30,	0x8cb667c8,	0x379ce885,
	0x8ca099da,	0x376f9e46,	0x8c8addb7,	0x37424b7b,	0x8c753362,	0x3714f02a,	0x8c5f9ade,	0x36e78c5b,
	0x8c4a142f,	0x36ba2014,	0x8c349f58,	0x368cab5c,	0x8c1f3c5d,	0x365f2e3b,	0x8c09eb40,	0x3631a8b8,
	0x8bf4ac05,	0x36041ad9,	0x8bdf7eb0,	0x35d684a6,	0x8bca6343,	0x35a8e625,	0x8bb559c1,	0x357b3f5d,
	0x8ba0622f,	0x354d9057,	0x8b8b7c8f,	0x351fd918,	0x8b76a8e4,	0x34f219a8,	0x8b61e733,	0x34c4520d,
	0x8b4d377c,	0x34968250,	0x8b3899c6,	0x3468aa76,	0x8b240e11,	0x343aca87,	0x8b0f9462,	0x340ce28b,
	0x8afb2cbb,	0x33def287,	0x8ae6d720,	0x33b0fa84,	0x8ad29394,	0x3382fa88,	0x8abe6219,	0x3354f29b,
	0x8aaa42b4,	0x3326e2c3,	0x8a963567,	0x32f8cb07,	0x8a823a36,	0x32caab6f,	0x8a6e5123,	0x329c8402,
	0x8a5a7a31,	0x326e54c7,	0x8a46b564,	0x32401dc6,	0x8a3302be,	0x3211df04,	0x8a1f6243,	0x31e39889,
	0x8a0bd3f5,	0x31b54a5e,	0x89f857d8,	0x3186f487,	0x89e4edef,	0x3158970e,	0x89d1963c,	0x312a31f8,
	0x89be50c3,	0x30fbc54d,	0x89ab1d87,	0x30cd5115,	0x8997fc8a,	0x309ed556,	0x8984edcf,	0x30705217,
	0x8971f15a,	0x3041c761,	0x895f072e,	0x30133539,	0x894c2f4c,	0x2fe49ba7,	0x893969b9,	0x2fb5fab2,
	0x8926b677,	0x2f875262,	0x89141589,	0x2f58a2be,	0x890186f2,	0x2f29ebcc,	0x88ef0ab4,	0x2efb2d95,
	0x88dca0d3,	0x2ecc681e,	0x88ca4951,	0x2e9d9b70,	0x88b80432,	0x2e6ec792,	0x88a5d177,	0x2e3fec8b,
	0x8893b125,	0x2e110a62,	0x8881a33d,	0x2de2211e,	0x886fa7c2,	0x2db330c7,	0x885dbeb8,	0x2d843964,
	0x884be821,	0x2d553afc,	0x883a23ff,	0x2d263596,	0x88287256,	0x2cf72939,	0x8816d327,	0x2cc815ee,
	0x88054677,	0x2c98fbba,	0x87f3cc48,	0x2c69daa6,	0x87e2649b,	0x2c3ab2b9,	0x87d10f75,	0x2c0b83fa,
	0x87bfccd7,	0x2bdc4e6f,	0x87ae9cc5,	0x2bad1221,	0x879d7f41,	0x2b7dcf17,	0x878c744d,	0x2b4e8558,
	0x877b7bec,	0x2b1f34eb,	0x876a9621,	0x2aefddd8,	0x8759c2ef,	0x2ac08026,	0x87490258,	0x2a911bdc,
	0x8738545e,	0x2a61b101,	0x8727b905,	0x2a323f9e,	0x8717304e,	0x2a02c7b8,	0x8706ba3d,	0x29d34958,
	0x86f656d3,	0x29a3c485,	0x86e60614,	0x29743946,	0x86d5c802,	0x2944a7a2,	0x86c59c9f,	0x29150fa1,
	0x86b583ee,	0x28e5714b,	0x86a57df2,	0x28b5cca5,	0x86958aac,	0x288621b9,	0x8685aa20,	0x2856708d,
	0x8675dc4f,	0x2826b928,	0x8666213c,	0x27f6fb92,	0x865678eb,	0x27c737d3,	0x8646e35c,	0x27976df1,
	0x86376092,	0x27679df4,	0x8627f091,	0x2737c7e3,	0x86189359,	0x2707ebc7,	0x860948ef,	0x26d809a5,
	0x85fa1153,	0x26a82186,	0x85eaec88,	0x26783370,	0x85dbda91,	0x26483f6c,	0x85ccdb70,	0x26184581,
	0x85bdef28,	0x25e845b6,	0x85af15b9,	0x25b84012,	0x85a04f28,	0x2588349d,	0x85919b76,	0x2558235f,
	0x8582faa5,	0x25280c5e,	0x85746cb8,	0x24f7efa2,	0x8565f1b0,	0x24c7cd33,	0x85578991,	0x2497a517,
	0x8549345c,	0x24677758,	0x853af214,	0x243743fa,	0x852cc2bb,	0x24070b08,	0x851ea652,	0x23d6cc87,
	0x85109cdd,	0x23a6887f,	0x8502a65c,	0x23763ef7,	0x84f4c2d4,	0x2345eff8,	0x84e6f244,	0x23159b88,
	0x84d934b1,	0x22e541af,	0x84cb8a1b,	0x22b4e274,	0x84bdf286,	0x22847de0,	0x84b06df2,	0x225413f8,
	0x84a2fc62,	0x2223a4c5,	0x84959dd9,	0x21f3304f,	0x84885258,	0x21c2b69c,	0x847b19e1,	0x219237b5,
	0x846df477,	0x2161b3a0,	0x8460e21a,	0x21312a65,	0x8453e2cf,	0x21009c0c,	0x8446f695,	0x20d0089c,
	0x843a1d70,	0x209f701c,	0x842d5762,	0x206ed295,	0x8420a46c,	0x203e300d,	0x84140490,	0x200d888d,
	0x840777d0,	0x1fdcdc1b,	0x83fafe2e,	0x1fac2abf,	0x83ee97ad,	0x1f7b7481,	0x83e2444d,	0x1f4ab968,
	0x83d60412,	0x1f19f97b,	0x83c9d6fc,	0x1ee934c3,	0x83bdbd0e,	0x1eb86b46,	0x83b1b649,	0x1e879d0d,
	0x83a5c2b0,	0x1e56ca1e,	0x8399e244,	0x1e25f282,	0x838e1507,	0x1df5163f,	0x83825afb,	0x1dc4355e,
	0x8376b422,	0x1d934fe5,	0x836b207d,	0x1d6265dd,	0x835fa00f,	0x1d31774d,	0x835432d8,	0x1d00843d,
	0x8348d8dc,	0x1ccf8cb3,	0x833d921b,	0x1c9e90b8,	0x83325e97,	0x1c6d9053,	0x83273e52,	0x1c3c8b8c,
	0x831c314e,	0x1c0b826a,	0x8311378d,	0x1bda74f6,	0x83065110,	0x1ba96335,	0x82fb7dd8,	0x1b784d30,
	0x82f0bde8,	0x1b4732ef,	0x82e61141,	0x1b161479,	0x82db77e5,	0x1ae4f1d6,	0x82d0f1d5,	0x1ab3cb0d,
	0x82c67f14,	0x1a82a026,	0x82bc1fa2,	0x1a517128,	0x82b1d381,	0x1a203e1b,	0x82a79ab3,	0x19ef0707,
	0x829d753a,	0x19bdcbf3,	0x82936317,	0x198c8ce7,	0x8289644b,	0x195b49ea,	0x827f78d8,	0x192a0304,
	0x8275a0c0,	0x18f8b83c,	0x826bdc04,	0x18c7699b,	0x82622aa6,	0x18961728,	0x82588ca7,	0x1864c0ea,
	0x824f0208,	0x183366e9,	0x82458acc,	0x1802092c,	0x823c26f3,	0x17d0a7bc,	0x8232d67f,	0x179f429f,
	0x82299971,	0x176dd9de,	0x82206fcc,	0x173c6d80,	0x82175990,	0x170afd8d,	0x820e56be,	0x16d98a0c,
	0x82056758,	0x16a81305,	0x81fc8b60,	0x1676987f,	0x81f3c2d7,	0x16451a83,	0x81eb0dbe,	0x16139918,
	0x81e26c16,	0x15e21445,	0x81d9dde1,	0x15b08c12,	0x81d16321,	0x157f0086,	0x81c8fbd6,	0x154d71aa,
	0x81c0a801,	0x151bdf86,	0x81b867a5,	0x14ea4a1f,	0x81b03ac2,	0x14b8b17f,	0x81a82159,	0x148715ae,
	0x81a01b6d,	0x145576b1,	0x819828fd,	0x1423d492,	0x81904a0c,	0x13f22f58,	0x81887e9a,	0x13c0870a,
	0x8180c6a9,	0x138edbb1,	0x8179223a,	0x135d2d53,	0x8171914e,	0x132b7bf9,	0x816a13e6,	0x12f9c7aa,
	0x8162aa04,	0x12c8106f,	0x815b53a8,	0x1296564d,	0x815410d4,	0x1264994e,	0x814ce188,	0x1232d979,
	0x8145c5c7,	0x120116d5,	0x813ebd90,	0x11cf516a,	0x8137c8e6,	0x119d8941,	0x8130e7c9,	0x116bbe60,
	0x812a1a3a,	0x1139f0cf,	0x8123603a,	0x11082096,	0x811cb9ca,	0x10d64dbd,	0x811626ec,	0x10a4784b,
	0x810fa7a0,	0x1072a048,	0x81093be8,	0x1040c5bb,	0x8102e3c4,	0x100ee8ad,	0x80fc9f35,	0xfdd0926,
	0x80f66e3c,	0xfab272b,	0x80f050db,	0xf7942c7,	0x80ea4712,	0xf475bff,	0x80e450e2,	0xf1572dc,
	0x80de6e4c,	0xee38766,	0x80d89f51,	0xeb199a4,	0x80d2e3f2,	0xe7fa99e,	0x80cd3c2f,	0xe4db75b,
	0x80c7a80a,	0xe1bc2e4,	0x80c22784,	0xde9cc40,	0x80bcba9d,	0xdb7d376,	0x80b76156,	0xd85d88f,
	0x80b21baf,	0xd53db92,	0x80ace9ab,	0xd21dc87,	0x80a7cb49,	0xcefdb76,	0x80a2c08b,	0xcbdd865,
	0x809dc971,	0xc8bd35e,	0x8098e5fb,	0xc59cc68,	0x8094162c,	0xc27c389,	0x808f5a02,	0xbf5b8cb,
	0x808ab180,	0xbc3ac35,	0x80861ca6,	0xb919dcf,	0x80819b74,	0xb5f8d9f,	0x807d2dec,	0xb2d7baf,
	0x8078d40d,	0xafb6805,	0x80748dd9,	0xac952aa,	0x80705b50,	0xa973ba5,	0x806c3c74,	0xa6522fe,
	0x80683143,	0xa3308bd,	0x806439c0,	0xa00ece8,	0x806055eb,	0x9cecf89,	0x805c85c4,	0x99cb0a7,
	0x8058c94c,	0x96a9049,	0x80552084,	0x9386e78,	0x80518b6b,	0x9064b3a,	0x804e0a04,	0x8d42699,
	0x804a9c4d,	0x8a2009a,	0x80474248,	0x86fd947,	0x8043fbf6,	0x83db0a7,	0x8040c956,	0x80b86c2,
	0x803daa6a,	0x7d95b9e,	0x803a9f31,	0x7a72f45,	0x8037a7ac,	0x77501be,	0x8034c3dd,	0x742d311,
	0x8031f3c2,	0x710a345,	0x802f375d,	0x6de7262,	0x802c8ead,	0x6ac406f,	0x8029f9b4,	0x67a0d76,
	0x80277872,	0x647d97c,	0x80250ae7,	0x615a48b,	0x8022b114,	0x5e36ea9,	0x80206af8,	0x5b137df,
	0x801e3895,	0x57f0035,	0x801c19ea,	0x54cc7b1,	0x801a0ef8,	0x51a8e5c,	0x801817bf,	0x4e8543e,
	0x80163440,	0x4b6195d,	0x8014647b,	0x483ddc3,	0x8012a86f,	0x451a177,	0x8011001f,	0x41f6480,
	0x800f6b88,	0x3ed26e6,	0x800deaad,	0x3bae8b2,	0x800c7d8c,	0x388a9ea,	0x800b2427,	0x3566a96,
	0x8009de7e,	0x3242abf,	0x8008ac90,	0x2f1ea6c,	0x80078e5e,	0x2bfa9a4,	0x800683e8,	0x28d6870,
	0x80058d2f,	0x25b26d7,	0x8004aa32,	0x228e4e2,	0x8003daf1,	0x1f6a297,	0x80031f6d,	0x1c45ffe,
	0x800277a6,	0x1921d20,	0x8001e39b,	0x15fda03,	0x8001634e,	0x12d96b1,	0x8000f6bd,	0xfb5330,
	0x80009dea,	0xc90f88,	0x800058d4,	0x96cbc1,	0x8000277a,	0x6487e3,	0x800009df,	0x3243f5,
	0x80000000,	0x0,	0x800009df,	0xffcdbc0b,	0x8000277a,	0xff9b781d,	0x800058d4,	0xff69343f,
	0x80009dea,	0xff36f078,	0x8000f6bd,	0xff04acd0,	0x8001634e,	0xfed2694f,	0x8001e39b,	0xfea025fd,
	0x800277a6,	0xfe6de2e0,	0x80031f6d,	0xfe3ba002,	0x8003daf1,	0xfe095d69,	0x8004aa32,	0xfdd71b1e,
	0x80058d2f,	0xfda4d929,	0x800683e8,	0xfd729790,	0x80078e5e,	0xfd40565c,	0x8008ac90,	0xfd0e1594,
	0x8009de7e,	0xfcdbd541,	0x800b2427,	0xfca9956a,	0x800c7d8c,	0xfc775616,	0x800deaad,	0xfc45174e,
	0x800f6b88,	0xfc12d91a,	0x8011001f,	0xfbe09b80,	0x8012a86f,	0xfbae5e89,	0x8014647b,	0xfb7c223d,
	0x80163440,	0xfb49e6a3,	0x801817bf,	0xfb17abc2,	0x801a0ef8,	0xfae571a4,	0x801c19ea,	0xfab3384f,
	0x801e3895,	0xfa80ffcb,	0x80206af8,	0xfa4ec821,	0x8022b114,	0xfa1c9157,	0x80250ae7,	0xf9ea5b75,
	0x80277872,	0xf9b82684,	0x8029f9b4,	0xf985f28a,	0x802c8ead,	0xf953bf91,	0x802f375d,	0xf9218d9e,
	0x8031f3c2,	0xf8ef5cbb,	0x8034c3dd,	0xf8bd2cef,	0x8037a7ac,	0xf88afe42,	0x803a9f31,	0xf858d0bb,
	0x803daa6a,	0xf826a462,	0x8040c956,	0xf7f4793e,	0x8043fbf6,	0xf7c24f59,	0x80474248,	0xf79026b9,
	0x804a9c4d,	0xf75dff66,	0x804e0a04,	0xf72bd967,	0x80518b6b,	0xf6f9b4c6,	0x80552084,	0xf6c79188,
	0x8058c94c,	0xf6956fb7,	0x805c85c4,	0xf6634f59,	0x806055eb,	0xf6313077,	0x806439c0,	0xf5ff1318,
	0x80683143,	0xf5ccf743,	0x806c3c74,	0xf59add02,	0x80705b50,	0xf568c45b,	0x80748dd9,	0xf536ad56,
	0x8078d40d,	0xf50497fb,	0x807d2dec,	0xf4d28451,	0x80819b74,	0xf4a07261,	0x80861ca6,	0xf46e6231,
	0x808ab180,	0xf43c53cb,	0x808f5a02,	0xf40a4735,	0x8094162c,	0xf3d83c77,	0x8098e5fb,	0xf3a63398,
	0x809dc971,	0xf3742ca2,	0x80a2c08b,	0xf342279b,	0x80a7cb49,	0xf310248a,	0x80ace9ab,	0xf2de2379,
	0x80b21baf,	0xf2ac246e,	0x80b76156,	0xf27a2771,	0x80bcba9d,	0xf2482c8a,	0x80c22784,	0xf21633c0,
	0x80c7a80a,	0xf1e43d1c,	0x80cd3c2f,	0xf1b248a5,	0x80d2e3f2,	0xf1805662,	0x80d89f51,	0xf14e665c,
	0x80de6e4c,	0xf11c789a,	0x80e450e2,	0xf0ea8d24,	0x80ea4712,	0xf0b8a401,	0x80f050db,	0xf086bd39,
	0x80f66e3c,	0xf054d8d5,	0x80fc9f35,	0xf022f6da,	0x8102e3c4,	0xeff11753,	0x81093be8,	0xefbf3a45,
	0x810fa7a0,	0xef8d5fb8,	0x811626ec,	0xef5b87b5,	0x811cb9ca,	0xef29b243,	0x8123603a,	0xeef7df6a,
	0x812a1a3a,	0xeec60f31,	0x8130e7c9,	0xee9441a0,	0x8137c8e6,	0xee6276bf,	0x813ebd90,	0xee30ae96,
	0x8145c5c7,	0xedfee92b,	0x814ce188,	0xedcd2687,	0x815410d4,	0xed9b66b2,	0x815b53a8,	0xed69a9b3,
	0x8162aa04,	0xed37ef91,	0x816a13e6,	0xed063856,	0x8171914e,	0xecd48407,	0x8179223a,	0xeca2d2ad,
	0x8180c6a9,	0xec71244f,	0x81887e9a,	0xec3f78f6,	0x81904a0c,	0xec0dd0a8,	0x819828fd,	0xebdc2b6e,
	0x81a01b6d,	0xebaa894f,	0x81a82159,	0xeb78ea52,	0x81b03ac2,	0xeb474e81,	0x81b867a5,	0xeb15b5e1,
	0x81c0a801,	0xeae4207a,	0x81c8fbd6,	0xeab28e56,	0x81d16321,	0xea80ff7a,	0x81d9dde1,	0xea4f73ee,
	0x81e26c16,	0xea1debbb,	0x81eb0dbe,	0xe9ec66e8,	0x81f3c2d7,	0xe9bae57d,	0x81fc8b60,	0xe9896781,
	0x82056758,	0xe957ecfb,	0x820e56be,	0xe92675f4,	0x82175990,	0xe8f50273,	0x82206fcc,	0xe8c39280,
	0x82299971,	0xe8922622,	0x8232d67f,	0xe860bd61,	0x823c26f3,	0xe82f5844,	0x82458acc,	0xe7fdf6d4,
	0x824f0208,	0xe7cc9917,	0x82588ca7,	0xe79b3f16,	0x82622aa6,	0xe769e8d8,	0x826bdc04,	0xe7389665,
	0x8275a0c0,	0xe70747c4,	0x827f78d8,	0xe6d5fcfc,	0x8289644b,	0xe6a4b616,	0x82936317,	0xe6737319,
	0x829d753a,	0xe642340d,	0x82a79ab3,	0xe610f8f9,	0x82b1d381,	0xe5dfc1e5,	0x82bc1fa2,	0xe5ae8ed8,
	0x82c67f14,	0xe57d5fda,	0x82d0f1d5,	0xe54c34f3,	0x82db77e5,	0xe51b0e2a,	0x82e61141,	0xe4e9eb87,
	0x82f0bde8,	0xe4b8cd11,	0x82fb7dd8,	0xe487b2d0,	0x83065110,	0xe4569ccb,	0x8311378d,	0xe4258b0a,
	0x831c314e,	0xe3f47d96,	0x83273e52,	0xe3c37474,	0x83325e97,	0xe3926fad,	0x833d921b,	0xe3616f48,
	0x8348d8dc,	0xe330734d,	0x835432d8,	0xe2ff7bc3,	0x835fa00f,	0xe2ce88b3,	0x836b207d,	0xe29d9a23,
	0x8376b422,	0xe26cb01b,	0x83825afb,	0xe23bcaa2,	0x838e1507,	0xe20ae9c1,	0x8399e244,	0xe1da0d7e,
	0x83a5c2b0,	0xe1a935e2,	0x83b1b649,	0xe17862f3,	0x83bdbd0e,	0xe14794ba,	0x83c9d6fc,	0xe116cb3d,
	0x83d60412,	0xe0e60685,	0x83e2444d,	0xe0b54698,	0x83ee97ad,	0xe0848b7f,	0x83fafe2e,	0xe053d541,
	0x840777d0,	0xe02323e5,	0x84140490,	0xdff27773,	0x8420a46c,	0xdfc1cff3,	0x842d5762,	0xdf912d6b,
	0x843a1d70,	0xdf608fe4,	0x8446f695,	0xdf2ff764,	0x8453e2cf,	0xdeff63f4,	0x8460e21a,	0xdeced59b,
	0x846df477,	0xde9e4c60,	0x847b19e1,	0xde6dc84b,	0x84885258,	0xde3d4964,	0x84959dd9,	0xde0ccfb1,
	0x84a2fc62,	0xdddc5b3b,	0x84b06df2,	0xddabec08,	0x84bdf286,	0xdd7b8220,	0x84cb8a1b,	0xdd4b1d8c,
	0x84d934b1,	0xdd1abe51,	0x84e6f244,	0xdcea6478,	0x84f4c2d4,	0xdcba1008,	0x8502a65c,	0xdc89c109,
	0x85109cdd,	0xdc597781,	0x851ea652,	0xdc293379,	0x852cc2bb,	0xdbf8f4f8,	0x853af214,	0xdbc8bc06,
	0x8549345c,	0xdb9888a8,	0x85578991,	0xdb685ae9,	0x8565f1b0,	0xdb3832cd,	0x85746cb8,	0xdb08105e,
	0x8582faa5,	0xdad7f3a2,	0x85919b76,	0xdaa7dca1,	0x85a04f28,	0xda77cb63,	0x85af15b9,	0xda47bfee,
	0x85bdef28,	0xda17ba4a,	0x85ccdb70,	0xd9e7ba7f,	0x85dbda91,	0xd9b7c094,	0x85eaec88,	0xd987cc90,
	0x85fa1153,	0xd957de7a,	0x860948ef,	0xd927f65b,	0x86189359,	0xd8f81439,	0x8627f091,	0xd8c8381d,
	0x86376092,	0xd898620c,	0x8646e35c,	0xd868920f,	0x865678eb,	0xd838c82d,	0x8666213c,	0xd809046e,
	0x8675dc4f,	0xd7d946d8,	0x8685aa20,	0xd7a98f73,	0x86958aac,	0xd779de47,	0x86a57df2,	0xd74a335b,
	0x86b583ee,	0xd71a8eb5,	0x86c59c9f,	0xd6eaf05f,	0x86d5c802,	0xd6bb585e,	0x86e60614,	0xd68bc6ba,
	0x86f656d3,	0xd65c3b7b,	0x8706ba3d,	0xd62cb6a8,	0x8717304e,	0xd5fd3848,	0x8727b905,	0xd5cdc062,
	0x8738545e,	0xd59e4eff,	0x87490258,	0xd56ee424,	0x8759c2ef,	0xd53f7fda,	0x876a9621,	0xd5102228,
	0x877b7bec,	0xd4e0cb15,	0x878c744d,	0xd4b17aa8,	0x879d7f41,	0xd48230e9,	0x87ae9cc5,	0xd452eddf,
	0x87bfccd7,	0xd423b191,	0x87d10f75,	0xd3f47c06,	0x87e2649b,	0xd3c54d47,	0x87f3cc48,	0xd396255a,
	0x88054677,	0xd3670446,	0x8816d327,	0xd337ea12,	0x88287256,	0xd308d6c7,	0x883a23ff,	0xd2d9ca6a,
	0x884be821,	0xd2aac504,	0x885dbeb8,	0xd27bc69c,	0x886fa7c2,	0xd24ccf39,	0x8881a33d,	0xd21ddee2,
	0x8893b125,	0xd1eef59e,	0x88a5d177,	0xd1c01375,	0x88b80432,	0xd191386e,	0x88ca4951,	0xd1626490,
	0x88dca0d3,	0xd13397e2,	0x88ef0ab4,	0xd104d26b,	0x890186f2,	0xd0d61434,	0x89141589,	0xd0a75d42,
	0x8926b677,	0xd078ad9e,	0x893969b9,	0xd04a054e,	0x894c2f4c,	0xd01b6459,	0x895f072e,	0xcfeccac7,
	0x8971f15a,	0xcfbe389f,	0x8984edcf,	0xcf8fade9,	0x8997fc8a,	0xcf612aaa,	0x89ab1d87,	0xcf32aeeb,
	0x89be50c3,	0xcf043ab3,	0x89d1963c,	0xced5ce08,	0x89e4edef,	0xcea768f2,	0x89f857d8,	0xce790b79,
	0x8a0bd3f5,	0xce4ab5a2,	0x8a1f6243,	0xce1c6777,	0x8a3302be,	0xcdee20fc,	0x8a46b564,	0xcdbfe23a,
	0x8a5a7a31,	0xcd91ab39,	0x8a6e5123,	0xcd637bfe,	0x8a823a36,	0xcd355491,	0x8a963567,	0xcd0734f9,
	0x8aaa42b4,	0xccd91d3d,	0x8abe6219,	0xccab0d65,	0x8ad29394,	0xcc7d0578,	0x8ae6d720,	0xcc4f057c,
	0x8afb2cbb,	0xcc210d79,	0x8b0f9462,	0xcbf31d75,	0x8b240e11,	0xcbc53579,	0x8b3899c6,	0xcb97558a,
	0x8b4d377c,	0xcb697db0,	0x8b61e733,	0xcb3badf3,	0x8b76a8e4,	0xcb0de658,	0x8b8b7c8f,	0xcae026e8,
	0x8ba0622f,	0xcab26fa9,	0x8bb559c1,	0xca84c0a3,	0x8bca6343,	0xca5719db,	0x8bdf7eb0,	0xca297b5a,
	0x8bf4ac05,	0xc9fbe527,	0x8c09eb40,	0xc9ce5748,	0x8c1f3c5d,	0xc9a0d1c5,	0x8c349f58,	0xc97354a4,
	0x8c4a142f,	0xc945dfec,	0x8c5f9ade,	0xc91873a5,	0x8c753362,	0xc8eb0fd6,	0x8c8addb7,	0xc8bdb485,
	0x8ca099da,	0xc89061ba,	0x8cb667c8,	0xc863177b,	0x8ccc477d,	0xc835d5d0,	0x8ce238f6,	0xc8089cbf,
	0x8cf83c30,	0xc7db6c50,	0x8d0e5127,	0xc7ae4489,	0x8d2477d8,	0xc7812572,	0x8d3ab03f,	0xc7540f11,
	0x8d50fa59,	0xc727016d,	0x8d675623,	0xc6f9fc8d,	0x8d7dc399,	0xc6cd0079,	0x8d9442b8,	0xc6a00d37,
	0x8daad37b,	0xc67322ce,	0x8dc175e0,	0xc6464144,	0x8dd829e4,	0xc61968a2,	0x8deeef82,	0xc5ec98ee,
	0x8e05c6b7,	0xc5bfd22e,	0x8e1caf80,	0xc593146a,	0x8e33a9da,	0xc5665fa9,	0x8e4ab5bf,	0xc539b3f1,
	0x8e61d32e,	0xc50d1149,	0x8e790222,	0xc4e077b8,	0x8e904298,	0xc4b3e746,	0x8ea7948c,	0xc4875ff9,
	0x8ebef7fb,	0xc45ae1d7,	0x8ed66ce1,	0xc42e6ce8,	0x8eedf33b,	0xc4020133,	0x8f058b04,	0xc3d59ebe,
	0x8f1d343a,	0xc3a94590,	0x8f34eed8,	0xc37cf5b0,	0x8f4cbadb,	0xc350af26,	0x8f649840,	0xc32471f7,
	0x8f7c8701,	0xc2f83e2a,	0x8f94871d,	0xc2cc13c7,	0x8fac988f,	0xc29ff2d4,	0x8fc4bb53,	0xc273db58,
	0x8fdcef66,	0xc247cd5a,	0x8ff534c4,	0xc21bc8e1,	0x900d8b69,	0xc1efcdf3,	0x9025f352,	0xc1c3dc97,
	0x903e6c7b,	0xc197f4d4,	0x9056f6df,	0xc16c16b0,	0x906f927c,	0xc1404233,	0x90883f4d,	0xc1147764,
	0x90a0fd4e,	0xc0e8b648,	0x90b9cc7d,	0xc0bcfee7,	0x90d2acd4,	0xc0915148,	0x90eb9e50,	0xc065ad70,
	0x9104a0ee,	0xc03a1368,	0x911db4a9,	0xc00e8336,	0x9136d97d,	0xbfe2fcdf,	0x91500f67,	0xbfb7806c,
	0x91695663,	0xbf8c0de3,	0x9182ae6d,	0xbf60a54a,	0x919c1781,	0xbf3546a8,	0x91b5919a,	0xbf09f205,
	0x91cf1cb6,	0xbedea765,	0x91e8b8d0,	0xbeb366d1,	0x920265e4,	0xbe88304f,	0x921c23ef,	0xbe5d03e6,
	0x9235f2ec,	0xbe31e19b,	0x924fd2d7,	0xbe06c977,	0x9269c3ac,	0xbddbbb7f,	0x9283c568,	0xbdb0b7bb,
	0x929dd806,	0xbd85be30,	0x92b7fb82,	0xbd5acee5,	0x92d22fd9,	0xbd2fe9e2,	0x92ec7505,	0xbd050f2c,
	0x9306cb04,	0xbcda3ecb,	0x932131d1,	0xbcaf78c4,	0x933ba968,	0xbc84bd1f,	0x935631c5,	0xbc5a0be2,
	0x9370cae4,	0xbc2f6513,	0x938b74c1,	0xbc04c8ba,	0x93a62f57,	0xbbda36dd,	0x93c0faa3,	0xbbafaf82,
	0x93dbd6a0,	0xbb8532b0,	0x93f6c34a,	0xbb5ac06d,	0x9411c09e,	0xbb3058c0,	0x942cce96,	0xbb05fbb0,
	0x9447ed2f,	0xbadba943,	0x94631c65,	0xbab16180,	0x947e5c33,	0xba87246d,	0x9499ac95,	0xba5cf210,
	0x94b50d87,	0xba32ca71,	0x94d07f05,	0xba08ad95,	0x94ec010b,	0xb9de9b83,	0x95079394,	0xb9b49442,
	0x9523369c,	0xb98a97d8,	0x953eea1e,	0xb960a64c,	0x955aae17,	0xb936bfa4,	0x95768283,	0xb90ce3e6,
	0x9592675c,	0xb8e31319,	0x95ae5c9f,	0xb8b94d44,	0x95ca6247,	0xb88f926d,	0x95e67850,	0xb865e299,
	0x96029eb6,	0xb83c3dd1,	0x961ed574,	0xb812a41a,	0x963b1c86,	0xb7e9157a,	0x965773e7,	0xb7bf91f8,
	0x9673db94,	0xb796199b,	0x96905388,	0xb76cac69,	0x96acdbbe,	0xb7434a67,	0x96c97432,	0xb719f39e,
	0x96e61ce0,	0xb6f0a812,	0x9702d5c3,	0xb6c767ca,	0x971f9ed7,	0xb69e32cd,	0x973c7817,	0xb6750921,
	0x9759617f,	0xb64beacd,	0x97765b0a,	0xb622d7d6,	0x979364b5,	0xb5f9d043,	0x97b07e7a,	0xb5d0d41a,
	0x97cda855,	0xb5a7e362,	0x97eae242,	0xb57efe22,	0x98082c3b,	0xb556245e,	0x9825863d,	0xb52d561e,
	0x9842f043,	0xb5049368,	0x98606a49,	0xb4dbdc42,	0x987df449,	0xb4b330b3,	0x989b8e40,	0xb48a90c0,
	0x98b93828,	0xb461fc70,	0x98d6f1fe,	0xb43973ca,	0x98f4bbbc,	0xb410f6d3,	0x9912955f,	0xb3e88592,
	0x99307ee0,	0xb3c0200c,	0x994e783d,	0xb397c649,	0x996c816f,	0xb36f784f,	0x998a9a74,	0xb3473623,
	0x99a8c345,	0xb31effcc,	0x99c6fbde,	0xb2f6d550,	0x99e5443b,	0xb2ceb6b5,	0x9a039c57,	0xb2a6a402,
	0x9a22042d,	0xb27e9d3c,	0x9a407bb9,	0xb256a26a,	0x9a5f02f5,	0xb22eb392,	0x9a7d99de,	0xb206d0ba,
	0x9a9c406e,	0xb1def9e9,	0x9abaf6a1,	0xb1b72f23,	0x9ad9bc71,	0xb18f7071,	0x9af891db,	0xb167bdd7,
	0x9b1776da,	0xb140175b,	0x9b366b68,	0xb1187d05,	0x9b556f81,	0xb0f0eeda,	0x9b748320,	0xb0c96ce0,
	0x9b93a641,	0xb0a1f71d,	0x9bb2d8de,	0xb07a8d97,	0x9bd21af3,	0xb0533055,	0x9bf16c7a,	0xb02bdf5c,
	0x9c10cd70,	0xb0049ab3,	0x9c303dcf,	0xafdd625f,	0x9c4fbd93,	0xafb63667,	0x9c6f4cb6,	0xaf8f16d1,
	0x9c8eeb34,	0xaf6803a2,	0x9cae9907,	0xaf40fce1,	0x9cce562c,	0xaf1a0293,	0x9cee229c,	0xaef314c0,
	0x9d0dfe54,	0xaecc336c,	0x9d2de94d,	0xaea55e9e,	0x9d4de385,	0xae7e965b,	0x9d6decf4,	0xae57daab,
	0x9d8e0597,	0xae312b92,	0x9dae2d68,	0xae0a8916,	0x9dce6463,	0xade3f33e,	0x9deeaa82,	0xadbd6a10,
	0x9e0effc1,	0xad96ed92,	0x9e2f641b,	0xad707dc8,	0x9e4fd78a,	0xad4a1aba,	0x9e705a09,	0xad23c46e,
	0x9e90eb94,	0xacfd7ae8,	0x9eb18c26,	0xacd73e30,	0x9ed23bb9,	0xacb10e4b,	0x9ef2fa49,	0xac8aeb3e,
	0x9f13c7d0,	0xac64d510,	0x9f34a449,	0xac3ecbc7,	0x9f558fb0,	0xac18cf69,	0x9f7689ff,	0xabf2dffb,
	0x9f979331,	0xabccfd83,	0x9fb8ab41,	0xaba72807,	0x9fd9d22a,	0xab815f8d,	0x9ffb07e7,	0xab5ba41a,
	0xa01c4c73,	0xab35f5b5,	0xa03d9fc8,	0xab105464,	0xa05f01e1,	0xaaeac02c,	0xa08072ba,	0xaac53912,
	0xa0a1f24d,	0xaa9fbf1e,	0xa0c38095,	0xaa7a5253,	0xa0e51d8c,	0xaa54f2ba,	0xa106c92f,	0xaa2fa056,
	0xa1288376,	0xaa0a5b2e,	0xa14a4c5e,	0xa9e52347,	0xa16c23e1,	0xa9bff8a8,	0xa18e09fa,	0xa99adb56,
	0xa1affea3,	0xa975cb57,	0xa1d201d7,	0xa950c8b0,	0xa1f41392,	0xa92bd367,	0xa21633cd,	0xa906eb82,
	0xa2386284,	0xa8e21106,	0xa25a9fb1,	0xa8bd43fa,	0xa27ceb4f,	0xa8988463,	0xa29f4559,	0xa873d246,
	0xa2c1adc9,	0xa84f2daa,	0xa2e4249b,	0xa82a9693,	0xa306a9c8,	0xa8060d08,	0xa3293d4b,	0xa7e1910f,
	0xa34bdf20,	0xa7bd22ac,	0xa36e8f41,	0xa798c1e5,	0xa3914da8,	0xa7746ec0,	0xa3b41a50,	0xa7502943,
	0xa3d6f534,	0xa72bf174,	0xa3f9de4e,	0xa707c757,	0xa41cd599,	0xa6e3aaf2,	0xa43fdb10,	0xa6bf9c4b,
	0xa462eeac,	0xa69b9b68,	0xa486106a,	0xa677a84e,	0xa4a94043,	0xa653c303,	0xa4cc7e32,	0xa62feb8b,
	0xa4efca31,	0xa60c21ee,	0xa513243b,	0xa5e8662f,	0xa5368c4b,	0xa5c4b855,	0xa55a025b,	0xa5a11866,
	0xa57d8666,	0xa57d8666,	0xa5a11866,	0xa55a025b,	0xa5c4b855,	0xa5368c4b,	0xa5e8662f,	0xa513243b,
	0xa60c21ee,	0xa4efca31,	0xa62feb8b,	0xa4cc7e32,	0xa653c303,	0xa4a94043,	0xa677a84e,	0xa486106a,
	0xa69b9b68,	0xa462eeac,	0xa6bf9c4b,	0xa43fdb10,	0xa6e3aaf2,	0xa41cd599,	0xa707c757,	0xa3f9de4e,
	0xa72bf174,	0xa3d6f534,	0xa7502943,	0xa3b41a50,	0xa7746ec0,	0xa3914da8,	0xa798c1e5,	0xa36e8f41,
	0xa7bd22ac,	0xa34bdf20,	0xa7e1910f,	0xa3293d4b,	0xa8060d08,	0xa306a9c8,	0xa82a9693,	0xa2e4249b,
	0xa84f2daa,	0xa2c1adc9,	0xa873d246,	0xa29f4559,	0xa8988463,	0xa27ceb4f,	0xa8bd43fa,	0xa25a9fb1,
	0xa8e21106,	0xa2386284,	0xa906eb82,	0xa21633cd,	0xa92bd367,	0xa1f41392,	0xa950c8b0,	0xa1d201d7,
	0xa975cb57,	0xa1affea3,	0xa99adb56,	0xa18e09fa,	0xa9bff8a8,	0xa16c23e1,	0xa9e52347,	0xa14a4c5e,
	0xaa0a5b2e,	0xa1288376,	0xaa2fa056,	0xa106c92f,	0xaa54f2ba,	0xa0e51d8c,	0xaa7a5253,	0xa0c38095,
	0xaa9fbf1e,	0xa0a1f24d,	0xaac53912,	0xa08072ba,	0xaaeac02c,	0xa05f01e1,	0xab105464,	0xa03d9fc8,
	0xab35f5b5,	0xa01c4c73,	0xab5ba41a,	0x9ffb07e7,	0xab815f8d,	0x9fd9d22a,	0xaba72807,	0x9fb8ab41,
	0xabccfd83,	0x9f979331,	0xabf2dffb,	0x9f7689ff,	0xac18cf69,	0x9f558fb0,	0xac3ecbc7,	0x9f34a449,
	0xac64d510,	0x9f13c7d0,	0xac8aeb3e,	0x9ef2fa49,	0xacb10e4b,	0x9ed23bb9,	0xacd73e30,	0x9eb18c26,
	0xacfd7ae8,	0x9e90eb94,	0xad23c46e,	0x9e705a09,	0xad4a1aba,	0x9e4fd78a,	0xad707dc8,	0x9e2f641b,
	0xad96ed92,	0x9e0effc1,	0xadbd6a10,	0x9deeaa82,	0xade3f33e,	0x9dce6463,	0xae0a8916,	0x9dae2d68,
	0xae312b92,	0x9d8e0597,	0xae57daab,	0x9d6decf4,	0xae7e965b,	0x9d4de385,	0xaea55e9e,	0x9d2de94d,
	0xaecc336c,	0x9d0dfe54,	0xaef314c0,	0x9cee229c,	0xaf1a0293,	0x9cce562c,	0xaf40fce1,	0x9cae9907,
	0xaf6803a2,	0x9c8eeb34,	0xaf8f16d1,	0x9c6f4cb6,	0xafb63667,	0x9c4fbd93,	0xafdd625f,	0x9c303dcf,
	0xb0049ab3,	0x9c10cd70,	0xb02bdf5c,	0x9bf16c7a,	0xb0533055,	0x9bd21af3,	0xb07a8d97,	0x9bb2d8de,
	0xb0a1f71d,	0x9b93a641,	0xb0c96ce0,	0x9b748320,	0xb0f0eeda,	0x9b556f81,	0xb1187d05,	0x9b366b68,
	0xb140175b,	0x9b1776da,	0xb167bdd7,	0x9af891db,	0xb18f7071,	0x9ad9bc71,	0xb1b72f23,	0x9abaf6a1,
	0xb1def9e9,	0x9a9c406e,	0xb206d0ba,	0x9a7d99de,	0xb22eb392,	0x9a5f02f5,	0xb256a26a,	0x9a407bb9,
	0xb27e9d3c,	0x9a22042d,	0xb2a6a402,	0x9a039c57,	0xb2ceb6b5,	0x99e5443b,	0xb2f6d550,	0x99c6fbde,
	0xb31effcc,	0x99a8c345,	0xb3473623,	0x998a9a74,	0xb36f784f,	0x996c816f,	0xb397c649,	0x994e783d,
	0xb3c0200c,	0x99307ee0,	0xb3e88592,	0x9912955f,	0xb410f6d3,	0x98f4bbbc,	0xb43973ca,	0x98d6f1fe,
	0xb461fc70,	0x98b93828,	0xb48a90c0,	0x989b8e40,	0xb4b330b3,	0x987df449,	0xb4dbdc42,	0x98606a49,
	0xb5049368,	0x9842f043,	0xb52d561e,	0x9825863d,	0xb556245e,	0x98082c3b,	0xb57efe22,	0x97eae242,
	0xb5a7e362,	0x97cda855,	0xb5d0d41a,	0x97b07e7a,	0xb5f9d043,	0x979364b5,	0xb622d7d6,	0x97765b0a,
	0xb64beacd,	0x9759617f,	0xb6750921,	0x973c7817,	0xb69e32cd,	0x971f9ed7,	0xb6c767ca,	0x9702d5c3,
	0xb6f0a812,	0x96e61ce0,	0xb719f39e,	0x96c97432,	0xb7434a67,	0x96acdbbe,	0xb76cac69,	0x96905388,
	0xb796199b,	0x9673db94,	0xb7bf91f8,	0x965773e7,	0xb7e9157a,	0x963b1c86,	0xb812a41a,	0x961ed574,
	0xb83c3dd1,	0x96029eb6,	0xb865e299,	0x95e67850,	0xb88f926d,	0x95ca6247,	0xb8b94d44,	0x95ae5c9f,
	0xb8e31319,	0x9592675c,	0xb90ce3e6,	0x95768283,	0xb936bfa4,	0x955aae17,	0xb960a64c,	0x953eea1e,
	0xb98a97d8,	0x9523369c,	0xb9b49442,	0x95079394,	0xb9de9b83,	0x94ec010b,	0xba08ad95,	0x94d07f05,
	0xba32ca71,	0x94b50d87,	0xba5cf210,	0x9499ac95,	0xba87246d,	0x947e5c33,	0xbab16180,	0x94631c65,
	0xbadba943,	0x9447ed2f,	0xbb05fbb0,	0x942cce96,	0xbb3058c0,	0x9411c09e,	0xbb5ac06d,	0x93f6c34a,
	0xbb8532b0,	0x93dbd6a0,	0xbbafaf82,	0x93c0faa3,	0xbbda36dd,	0x93a62f57,	0xbc04c8ba,	0x938b74c1,
	0xbc2f6513,	0x9370cae4,	0xbc5a0be2,	0x935631c5,	0xbc84bd1f,	0x933ba968,	0xbcaf78c4,	0x932131d1,
	0xbcda3ecb,	0x9306cb04,	0xbd050f2c,	0x92ec7505,	0xbd2fe9e2,	0x92d22fd9,	0xbd5acee5,	0x92b7fb82,
	0xbd85be30,	0x929dd806,	0xbdb0b7bb,	0x9283c568,	0xbddbbb7f,	0x9269c3ac,	0xbe06c977,	0x924fd2d7,
	0xbe31e19b,	0x9235f2ec,	0xbe5d03e6,	0x921c23ef,	0xbe88304f,	0x920265e4,	0xbeb366d1,	0x91e8b8d0,
	0xbedea765,	0x91cf1cb6,	0xbf09f205,	0x91b5919a,	0xbf3546a8,	0x919c1781,	0xbf60a54a,	0x9182ae6d,
	0xbf8c0de3,	0x91695663,	0xbfb7806c,	0x91500f67,	0xbfe2fcdf,	0x9136d97d,	0xc00e8336,	0x911db4a9,
	0xc03a1368,	0x9104a0ee,	0xc065ad70,	0x90eb9e50,	0xc0915148,	0x90d2acd4,	0xc0bcfee7,	0x90b9cc7d,
	0xc0e8b648,	0x90a0fd4e,	0xc1147764,	0x90883f4d,	0xc1404233,	0x906f927c,	0xc16c16b0,	0x9056f6df,
	0xc197f4d4,	0x903e6c7b,	0xc1c3dc97,	0x9025f352,	0xc1efcdf3,	0x900d8b69,	0xc21bc8e1,	0x8ff534c4,
	0xc247cd5a,	0x8fdcef66,	0xc273db58,	0x8fc4bb53,	0xc29ff2d4,	0x8fac988f,	0xc2cc13c7,	0x8f94871d,
	0xc2f83e2a,	0x8f7c8701,	0xc32471f7,	0x8f649840,	0xc350af26,	0x8f4cbadb,	0xc37cf5b0,	0x8f34eed8,
	0xc3a94590,	0x8f1d343a,	0xc3d59ebe,	0x8f058b04,	0xc4020133,	0x8eedf33b,	0xc42e6ce8,	0x8ed66ce1,
	0xc45ae1d7,	0x8ebef7fb,	0xc4875ff9,	0x8ea7948c,	0xc4b3e746,	0x8e904298,	0xc4e077b8,	0x8e790222,
	0xc50d1149,	0x8e61d32e,	0xc539b3f1,	0x8e4ab5bf,	0xc5665fa9,	0x8e33a9da,	0xc593146a,	0x8e1caf80,
	0xc5bfd22e,	0x8e05c6b7,	0xc5ec98ee,	0x8deeef82,	0xc61968a2,	0x8dd829e4,	0xc6464144,	0x8dc175e0,
	0xc67322ce,	0x8daad37b,	0xc6a00d37,	0x8d9442b8,	0xc6cd0079,	0x8d7dc399,	0xc6f9fc8d,	0x8d675623,
	0xc727016d,	0x8d50fa59,	0xc7540f11,	0x8d3ab03f,	0xc7812572,	0x8d2477d8,	0xc7ae4489,	0x8d0e5127,
	0xc7db6c50,	0x8cf83c30,	0xc8089cbf,	0x8ce238f6,	0xc835d5d0,	0x8ccc477d,	0xc863177b,	0x8cb667c8,
	0xc89061ba,	0x8ca099da,	0xc8bdb485,	0x8c8addb7,	0xc8eb0fd6,	0x8c753362,	0xc91873a5,	0x8c5f9ade,
	0xc945dfec,	0x8c4a142f,	0xc97354a4,	0x8c349f58,	0xc9a0d1c5,	0x8c1f3c5d,	0xc9ce5748,	0x8c09eb40,
	0xc9fbe527,	0x8bf4ac05,	0xca297b5a,	0x8bdf7eb0,	0xca5719db,	0x8bca6343,	0xca84c0a3,	0x8bb559c1,
	0xcab26fa9,	0x8ba0622f,	0xcae026e8,	0x8b8b7c8f,	0xcb0de658,	0x8b76a8e4,	0xcb3badf3,	0x8b61e733,
	0xcb697db0,	0x8b4d377c,	0xcb97558a,	0x8b3899c6,	0xcbc53579,	0x8b240e11,	0xcbf31d75,	0x8b0f9462,
	0xcc210d79,	0x8afb2cbb,	0xcc4f057c,	0x8ae6d720,	0xcc7d0578,	0x8ad29394,	0xccab0d65,	0x8abe6219,
	0xccd91d3d,	0x8aaa42b4,	0xcd0734f9,	0x8a963567,	0xcd355491,	0x8a823a36,	0xcd637bfe,	0x8a6e5123,
	0xcd91ab39,	0x8a5a7a31,	0xcdbfe23a,	0x8a46b564,	0xcdee20fc,	0x8a3302be,	0xce1c6777,	0x8a1f6243,
	0xce4ab5a2,	0x8a0bd3f5,	0xce790b79,	0x89f857d8,	0xcea768f2,	0x89e4edef,	0xced5ce08,	0x89d1963c,
	0xcf043ab3,	0x89be50c3,	0xcf32aeeb,	0x89ab1d87,	0xcf612aaa,	0x8997fc8a,	0xcf8fade9,	0x8984edcf,
	0xcfbe389f,	0x8971f15a,	0xcfeccac7,	0x895f072e,	0xd01b6459,	0x894c2f4c,	0xd04a054e,	0x893969b9,
	0xd078ad9e,	0x8926b677,	0xd0a75d42,	0x89141589,	0xd0d61434,	0x890186f2,	0xd104d26b,	0x88ef0ab4,
	0xd13397e2,	0x88dca0d3,	0xd1626490,	0x88ca4951,	0xd191386e,	0x88b80432,	0xd1c01375,	0x88a5d177,
	0xd1eef59e,	0x8893b125,	0xd21ddee2,	0x8881a33d,	0xd24ccf39,	0x886fa7c2,	0xd27bc69c,	0x885dbeb8,
	0xd2aac504,	0x884be821,	0xd2d9ca6a,	0x883a23ff,	0xd308d6c7,	0x88287256,	0xd337ea12,	0x8816d327,
	0xd3670446,	0x88054677,	0xd396255a,	0x87f3cc48,	0xd3c54d47,	0x87e2649b,	0xd3f47c06,	0x87d10f75,
	0xd423b191,	0x87bfccd7,	0xd452eddf,	0x87ae9cc5,	0xd48230e9,	0x879d7f41,	0xd4b17aa8,	0x878c744d,
	0xd4e0cb15,	0x877b7bec,	0xd5102228,	0x876a9621,	0xd53f7fda,	0x8759c2ef,	0xd56ee424,	0x87490258,
	0xd59e4eff,	0x8738545e,	0xd5cdc062,	0x8727b905,	0xd5fd3848,	0x8717304e,	0xd62cb6a8,	0x8706ba3d,
	0xd65c3b7b,	0x86f656d3,	0xd68bc6ba,	0x86e60614,	0xd6bb585e,	0x86d5c802,	0xd6eaf05f,	0x86c59c9f,
	0xd71a8eb5,	0x86b583ee,	0xd74a335b,	0x86a57df2,	0xd779de47,	0x86958aac,	0xd7a98f73,	0x8685aa20,
	0xd7d946d8,	0x8675dc4f,	0xd809046e,	0x8666213c,	0xd838c82d,	0x865678eb,	0xd868920f,	0x8646e35c,
	0xd898620c,	0x86376092,	0xd8c8381d,	0x8627f091,	0xd8f81439,	0x86189359,	0xd927f65b,	0x860948ef,
	0xd957de7a,	0x85fa1153,	0xd987cc90,	0x85eaec88,	0xd9b7c094,	0x85dbda91,	0xd9e7ba7f,	0x85ccdb70,
	0xda17ba4a,	0x85bdef28,	0xda47bfee,	0x85af15b9,	0xda77cb63,	0x85a04f28,	0xdaa7dca1,	0x85919b76,
	0xdad7f3a2,	0x8582faa5,	0xdb08105e,	0x85746cb8,	0xdb3832cd,	0x8565f1b0,	0xdb685ae9,	0x85578991,
	0xdb9888a8,	0x8549345c,	0xdbc8bc06,	0x853af214,	0xdbf8f4f8,	0x852cc2bb,	0xdc293379,	0x851ea652,
	0xdc597781,	0x85109cdd,	0xdc89c109,	0x8502a65c,	0xdcba1008,	0x84f4c2d4,	0xdcea6478,	0x84e6f244,
	0xdd1abe51,	0x84d934b1,	0xdd4b1d8c,	0x84cb8a1b,	0xdd7b8220,	0x84bdf286,	0xddabec08,	0x84b06df2,
	0xdddc5b3b,	0x84a2fc62,	0xde0ccfb1,	0x84959dd9,	0xde3d4964,	0x84885258,	0xde6dc84b,	0x847b19e1,
	0xde9e4c60,	0x846df477,	0xdeced59b,	0x8460e21a,	0xdeff63f4,	0x8453e2cf,	0xdf2ff764,	0x8446f695,
	0xdf608fe4,	0x843a1d70,	0xdf912d6b,	0x842d5762,	0xdfc1cff3,	0x8420a46c,	0xdff27773,	0x84140490,
	0xe02323e5,	0x840777d0,	0xe053d541,	0x83fafe2e,	0xe0848b7f,	0x83ee97ad,	0xe0b54698,	0x83e2444d,
	0xe0e60685,	0x83d60412,	0xe116cb3d,	0x83c9d6fc,	0xe14794ba,	0x83bdbd0e,	0xe17862f3,	0x83b1b649,
	0xe1a935e2,	0x83a5c2b0,	0xe1da0d7e,	0x8399e244,	0xe20ae9c1,	0x838e1507,	0xe23bcaa2,	0x83825afb,
	0xe26cb01b,	0x8376b422,	0xe29d9a23,	0x836b207d,	0xe2ce88b3,	0x835fa00f,	0xe2ff7bc3,	0x835432d8,
	0xe330734d,	0x8348d8dc,	0xe3616f48,	0x833d921b,	0xe3926fad,	0x83325e97,	0xe3c37474,	0x83273e52,
	0xe3f47d96,	0x831c314e,	0xe4258b0a,	0x8311378d,	0xe4569ccb,	0x83065110,	0xe487b2d0,	0x82fb7dd8,
	0xe4b8cd11,	0x82f0bde8,	0xe4e9eb87,	0x82e61141,	0xe51b0e2a,	0x82db77e5,	0xe54c34f3,	0x82d0f1d5,
	0xe57d5fda,	0x82c67f14,	0xe5ae8ed8,	0x82bc1fa2,	0xe5dfc1e5,	0x82b1d381,	0xe610f8f9,	0x82a79ab3,
	0xe642340d,	0x829d753a,	0xe6737319,	0x82936317,	0xe6a4b616,	0x8289644b,	0xe6d5fcfc,	0x827f78d8,
	0xe70747c4,	0x8275a0c0,	0xe7389665,	0x826bdc04,	0xe769e8d8,	0x82622aa6,	0xe79b3f16,	0x82588ca7,
	0xe7cc9917,	0x824f0208,	0xe7fdf6d4,	0x82458acc,	0xe82f5844,	0x823c26f3,	0xe860bd61,	0x8232d67f,
	0xe8922622,	0x82299971,	0xe8c39280,	0x82206fcc,	0xe8f50273,	0x82175990,	0xe92675f4,	0x820e56be,
	0xe957ecfb,	0x82056758,	0xe9896781,	0x81fc8b60,	0xe9bae57d,	0x81f3c2d7,	0xe9ec66e8,	0x81eb0dbe,
	0xea1debbb,	0x81e26c16,	0xea4f73ee,	0x81d9dde1,	0xea80ff7a,	0x81d16321,	0xeab28e56,	0x81c8fbd6,
	0xeae4207a,	0x81c0a801,	0xeb15b5e1,	0x81b867a5,	0xeb474e81,	0x81b03ac2,	0xeb78ea52,	0x81a82159,
	0xebaa894f,	0x81a01b6d,	0xebdc2b6e,	0x819828fd,	0xec0dd0a8,	0x81904a0c,	0xec3f78f6,	0x81887e9a,
	0xec71244f,	0x8180c6a9,	0xeca2d2ad,	0x8179223a,	0xecd48407,	0x8171914e,	0xed063856,	0x816a13e6,
	0xed37ef91,	0x8162aa04,	0xed69a9b3,	0x815b53a8,	0xed9b66b2,	0x815410d4,	0xedcd2687,	0x814ce188,
	0xedfee92b,	0x8145c5c7,	0xee30ae96,	0x813ebd90,	0xee6276bf,	0x8137c8e6,	0xee9441a0,	0x8130e7c9,
	0xeec60f31,	0x812a1a3a,	0xeef7df6a,	0x8123603a,	0xef29b243,	0x811cb9ca,	0xef5b87b5,	0x811626ec,
	0xef8d5fb8,	0x810fa7a0,	0xefbf3a45,	0x81093be8,	0xeff11753,	0x8102e3c4,	0xf022f6da,	0x80fc9f35,
	0xf054d8d5,	0x80f66e3c,	0xf086bd39,	0x80f050db,	0xf0b8a401,	0x80ea4712,	0xf0ea8d24,	0x80e450e2,
	0xf11c789a,	0x80de6e4c,	0xf14e665c,	0x80d89f51,	0xf1805662,	0x80d2e3f2,	0xf1b248a5,	0x80cd3c2f,
	0xf1e43d1c,	0x80c7a80a,	0xf21633c0,	0x80c22784,	0xf2482c8a,	0x80bcba9d,	0xf27a2771,	0x80b76156,
	0xf2ac246e,	0x80b21baf,	0xf2de2379,	0x80ace9ab,	0xf310248a,	0x80a7cb49,	0xf342279b,	0x80a2c08b,
	0xf3742ca2,	0x809dc971,	0xf3a63398,	0x8098e5fb,	0xf3d83c77,	0x8094162c,	0xf40a4735,	0x808f5a02,
	0xf43c53cb,	0x808ab180,	0xf46e6231,	0x80861ca6,	0xf4a07261,	0x80819b74,	0xf4d28451,	0x807d2dec,
	0xf50497fb,	0x8078d40d,	0xf536ad56,	0x80748dd9,	0xf568c45b,	0x80705b50,	0xf59add02,	0x806c3c74,
	0xf5ccf743,	0x80683143,	0xf5ff1318,	0x806439c0,	0xf6313077,	0x806055eb,	0xf6634f59,	0x805c85c4,
	0xf6956fb7,	0x8058c94c,	0xf6c79188,	0x80552084,	0xf6f9b4c6,	0x80518b6b,	0xf72bd967,	0x804e0a04,
	0xf75dff66,	0x804a9c4d,	0xf79026b9,	0x80474248,	0xf7c24f59,	0x8043fbf6,	0xf7f4793e,	0x8040c956,
	0xf826a462,	0x803daa6a,	0xf858d0bb,	0x803a9f31,	0xf88afe42,	0x8037a7ac,	0xf8bd2cef,	0x8034c3dd,
	0xf8ef5cbb,	0x8031f3c2,	0xf9218d9e,	0x802f375d,	0xf953bf91,	0x802c8ead,	0xf985f28a,	0x8029f9b4,
	0xf9b82684,	0x80277872,	0xf9ea5b75,	0x80250ae7,	0xfa1c9157,	0x8022b114,	0xfa4ec821,	0x80206af8,
	0xfa80ffcb,	0x801e3895,	0xfab3384f,	0x801c19ea,	0xfae571a4,	0x801a0ef8,	0xfb17abc2,	0x801817bf,
	0xfb49e6a3,	0x80163440,	0xfb7c223d,	0x8014647b,	0xfbae5e89,	0x8012a86f,	0xfbe09b80,	0x8011001f,
	0xfc12d91a,	0x800f6b88,	0xfc45174e,	0x800deaad,	0xfc775616,	0x800c7d8c,	0xfca9956a,	0x800b2427,
	0xfcdbd541,	0x8009de7e,	0xfd0e1594,	0x8008ac90,	0xfd40565c,	0x80078e5e,	0xfd729790,	0x800683e8,
	0xfda4d929,	0x80058d2f,	0xfdd71b1e,	0x8004aa32,	0xfe095d69,	0x8003daf1,	0xfe3ba002,	0x80031f6d,
	0xfe6de2e0,	0x800277a6,	0xfea025fd,	0x8001e39b,	0xfed2694f,	0x8001634e,	0xff04acd0,	0x8000f6bd,
	0xff36f078,	0x80009dea,	0xff69343f,	0x800058d4,	0xff9b781d,	0x8000277a,	0xffcdbc0b,	0x800009df,
	0x0,	0x80000000,	0x3243f5,	0x800009df,	0x6487e3,	0x8000277a,	0x96cbc1,	0x800058d4,
	0xc90f88,	0x80009dea,	0xfb5330,	0x8000f6bd,	0x12d96b1,	0x8001634e,	0x15fda03,	0x8001e39b,
	0x1921d20,	0x800277a6,	0x1c45ffe,	0x80031f6d,	0x1f6a297,	0x8003daf1,	0x228e4e2,	0x8004aa32,
	0x25b26d7,	0x80058d2f,	0x28d6870,	0x800683e8,	0x2bfa9a4,	0x80078e5e,	0x2f1ea6c,	0x8008ac90,
	0x3242abf,	0x8009de7e,	0x3566a96,	0x800b2427,	0x388a9ea,	0x800c7d8c,	0x3bae8b2,	0x800deaad,
	0x3ed26e6,	0x800f6b88,	0x41f6480,	0x8011001f,	0x451a177,	0x8012a86f,	0x483ddc3,	0x8014647b,
	0x4b6195d,	0x80163440,	0x4e8543e,	0x801817bf,	0x51a8e5c,	0x801a0ef8,	0x54cc7b1,	0x801c19ea,
	0x57f0035,	0x801e3895,	0x5b137df,	0x80206af8,	0x5e36ea9,	0x8022b114,	0x615a48b,	0x80250ae7,
	0x647d97c,	0x80277872,	0x67a0d76,	0x8029f9b4,	0x6ac406f,	0x802c8ead,	0x6de7262,	0x802f375d,
	0x710a345,	0x8031f3c2,	0x742d311,	0x8034c3dd,	0x77501be,	0x8037a7ac,	0x7a72f45,	0x803a9f31,
	0x7d95b9e,	0x803daa6a,	0x80b86c2,	0x8040c956,	0x83db0a7,	0x8043fbf6,	0x86fd947,	0x80474248,
	0x8a2009a,	0x804a9c4d,	0x8d42699,	0x804e0a04,	0x9064b3a,	0x80518b6b,	0x9386e78,	0x80552084,
	0x96a9049,	0x8058c94c,	0x99cb0a7,	0x805c85c4,	0x9cecf89,	0x806055eb,	0xa00ece8,	0x806439c0,
	0xa3308bd,	0x80683143,	0xa6522fe,	0x806c3c74,	0xa973ba5,	0x80705b50,	0xac952aa,	0x80748dd9,
	0xafb6805,	0x8078d40d,	0xb2d7baf,	0x807d2dec,	0xb5f8d9f,	0x80819b74,	0xb919dcf,	0x80861ca6,
	0xbc3ac35,	0x808ab180,	0xbf5b8cb,	0x808f5a02,	0xc27c389,	0x8094162c,	0xc59cc68,	0x8098e5fb,
	0xc8bd35e,	0x809dc971,	0xcbdd865,	0x80a2c08b,	0xcefdb76,	0x80a7cb49,	0xd21dc87,	0x80ace9ab,
	0xd53db92,	0x80b21baf,	0xd85d88f,	0x80b76156,	0xdb7d376,	0x80bcba9d,	0xde9cc40,	0x80c22784,
	0xe1bc2e4,	0x80c7a80a,	0xe4db75b,	0x80cd3c2f,	0xe7fa99e,	0x80d2e3f2,	0xeb199a4,	0x80d89f51,
	0xee38766,	0x80de6e4c,	0xf1572dc,	0x80e450e2,	0xf475bff,	0x80ea4712,	0xf7942c7,	0x80f050db,
	0xfab272b,	0x80f66e3c,	0xfdd0926,	0x80fc9f35,	0x100ee8ad,	0x8102e3c4,	0x1040c5bb,	0x81093be8,
	0x1072a048,	0x810fa7a0,	0x10a4784b,	0x811626ec,	0x10d64dbd,	0x811cb9ca,	0x11082096,	0x8123603a,
	0x1139f0cf,	0x812a1a3a,	0x116bbe60,	0x8130e7c9,	0x119d8941,	0x8137c8e6,	0x11cf516a,	0x813ebd90,
	0x120116d5,	0x8145c5c7,	0x1232d979,	0x814ce188,	0x1264994e,	0x815410d4,	0x1296564d,	0x815b53a8,
	0x12c8106f,	0x8162aa04,	0x12f9c7aa,	0x816a13e6,	0x132b7bf9,	0x8171914e,	0x135d2d53,	0x8179223a,
	0x138edbb1,	0x8180c6a9,	0x13c0870a,	0x81887e9a,	0x13f22f58,	0x81904a0c,	0x1423d492,	0x819828fd,
	0x145576b1,	0x81a01b6d,	0x148715ae,	0x81a82159,	0x14b8b17f,	0x81b03ac2,	0x14ea4a1f,	0x81b867a5,
	0x151bdf86,	0x81c0a801,	0x154d71aa,	0x81c8fbd6,	0x157f0086,	0x81d16321,	0x15b08c12,	0x81d9dde1,
	0x15e21445,	0x81e26c16,	0x16139918,	0x81eb0dbe,	0x16451a83,	0x81f3c2d7,	0x1676987f,	0x81fc8b60,
	0x16a81305,	0x82056758,	0x16d98a0c,	0x820e56be,	0x170afd8d,	0x82175990,	0x173c6d80,	0x82206fcc,
	0x176dd9de,	0x82299971,	0x179f429f,	0x8232d67f,	0x17d0a7bc,	0x823c26f3,	0x1802092c,	0x82458acc,
	0x183366e9,	0x824f0208,	0x1864c0ea,	0x82588ca7,	0x18961728,	0x82622aa6,	0x18c7699b,	0x826bdc04,
	0x18f8b83c,	0x8275a0c0,	0x192a0304,	0x827f78d8,	0x195b49ea,	0x8289644b,	0x198c8ce7,	0x82936317,
	0x19bdcbf3,	0x829d753a,	0x19ef0707,	0x82a79ab3,	0x1a203e1b,	0x82b1d381,	0x1a517128,	0x82bc1fa2,
	0x1a82a026,	0x82c67f14,	0x1ab3cb0d,	0x82d0f1d5,	0x1ae4f1d6,	0x82db77e5,	0x1b161479,	0x82e61141,
	0x1b4732ef,	0x82f0bde8,	0x1b784d30,	0x82fb7dd8,	0x1ba96335,	0x83065110,	0x1bda74f6,	0x8311378d,
	0x1c0b826a,	0x831c314e,	0x1c3c8b8c,	0x83273e52,	0x1c6d9053,	0x83325e97,	0x1c9e90b8,	0x833d921b,
	0x1ccf8cb3,	0x8348d8dc,	0x1d00843d,	0x835432d8,	0x1d31774d,	0x835fa00f,	0x1d6265dd,	0x836b207d,
	0x1d934fe5,	0x8376b422,	0x1dc4355e,	0x83825afb,	0x1df5163f,	0x838e1507,	0x1e25f282,	0x8399e244,
	0x1e56ca1e,	0x83a5c2b0,	0x1e879d0d,	0x83b1b649,	0x1eb86b46,	0x83bdbd0e,	0x1ee934c3,	0x83c9d6fc,
	0x1f19f97b,	0x83d60412,	0x1f4ab968,	0x83e2444d,	0x1f7b7481,	0x83ee97ad,	0x1fac2abf,	0x83fafe2e,
	0x1fdcdc1b,	0x840777d0,	0x200d888d,	0x84140490,	0x203e300d,	0x8420a46c,	0x206ed295,	0x842d5762,
	0x209f701c,	0x843a1d70,	0x20d0089c,	0x8446f695,	0x21009c0c,	0x8453e2cf,	0x21312a65,	0x8460e21a,
	0x2161b3a0,	0x846df477,	0x219237b5,	0x847b19e1,	0x21c2b69c,	0x84885258,	0x21f3304f,	0x84959dd9,
	0x2223a4c5,	0x84a2fc62,	0x225413f8,	0x84b06df2,	0x22847de0,	0x84bdf286,	0x22b4e274,	0x84cb8a1b,
	0x22e541af,	0x84d934b1,	0x23159b88,	0x84e6f244,	0x2345eff8,	0x84f4c2d4,	0x23763ef7,	0x8502a65c,
	0x23a6887f,	0x85109cdd,	0x23d6cc87,	0x851ea652,	0x24070b08,	0x852cc2bb,	0x243743fa,	0x853af214,
	0x24677758,	0x8549345c,	0x2497a517,	0x85578991,	0x24c7cd33,	0x8565f1b0,	0x24f7efa2,	0x85746cb8,
	0x25280c5e,	0x8582faa5,	0x2558235f,	0x85919b76,	0x2588349d,	0x85a04f28,	0x25b84012,	0x85af15b9,
	0x25e845b6,	0x85bdef28,	0x26184581,	0x85ccdb70,	0x26483f6c,	0x85dbda91,	0x26783370,	0x85eaec88,
	0x26a82186,	0x85fa1153,	0x26d809a5,	0x860948ef,	0x2707ebc7,	0x86189359,	0x2737c7e3,	0x8627f091,
	0x27679df4,	0x86376092,	0x27976df1,	0x8646e35c,	0x27c737d3,	0x865678eb,	0x27f6fb92,	0x8666213c,
	0x2826b928,	0x8675dc4f,	0x2856708d,	0x8685aa20,	0x288621b9,	0x86958aac,	0x28b5cca5,	0x86a57df2,
	0x28e5714b,	0x86b583ee,	0x29150fa1,	0x86c59c9f,	0x2944a7a2,	0x86d5c802,	0x29743946,	0x86e60614,
	0x29a3c485,	0x86f656d3,	0x29d34958,	0x8706ba3d,	0x2a02c7b8,	0x8717304e,	0x2a323f9e,	0x8727b905,
	0x2a61b101,	0x8738545e,	0x2a911bdc,	0x87490258,	0x2ac08026,	0x8759c2ef,	0x2aefddd8,	0x876a9621,
	0x2b1f34eb,	0x877b7bec,	0x2b4e8558,	0x878c744d,	0x2b7dcf17,	0x879d7f41,	0x2bad1221,	0x87ae9cc5,
	0x2bdc4e6f,	0x87bfccd7,	0x2c0b83fa,	0x87d10f75,	0x2c3ab2b9,	0x87e2649b,	0x2c69daa6,	0x87f3cc48,
	0x2c98fbba,	0x88054677,	0x2cc815ee,	0x8816d327,	0x2cf72939,	0x88287256,	0x2d263596,	0x883a23ff,
	0x2d553afc,	0x884be821,	0x2d843964,	0x885dbeb8,	0x2db330c7,	0x886fa7c2,	0x2de2211e,	0x8881a33d,
	0x2e110a62,	0x8893b125,	0x2e3fec8b,	0x88a5d177,	0x2e6ec792,	0x88b80432,	0x2e9d9b70,	0x88ca4951,
	0x2ecc681e,	0x88dca0d3,	0x2efb2d95,	0x88ef0ab4,	0x2f29ebcc,	0x890186f2,	0x2f58a2be,	0x89141589,
	0x2f875262,	0x8926b677,	0x2fb5fab2,	0x893969b9,	0x2fe49ba7,	0x894c2f4c,	0x30133539,	0x895f072e,
	0x3041c761,	0x8971f15a,	0x30705217,	0x8984edcf,	0x309ed556,	0x8997fc8a,	0x30cd5115,	0x89ab1d87,
	0x30fbc54d,	0x89be50c3,	0x312a31f8,	0x89d1963c,	0x3158970e,	0x89e4edef,	0x3186f487,	0x89f857d8,
	0x31b54a5e,	0x8a0bd3f5,	0x31e39889,	0x8a1f6243,	0x3211df04,	0x8a3302be,	0x32401dc6,	0x8a46b564,
	0x326e54c7,	0x8a5a7a31,	0x329c8402,	0x8a6e5123,	0x32caab6f,	0x8a823a36,	0x32f8cb07,	0x8a963567,
	0x3326e2c3,	0x8aaa42b4,	0x3354f29b,	0x8abe6219,	0x3382fa88,	0x8ad29394,	0x33b0fa84,	0x8ae6d720,
	0x33def287,	0x8afb2cbb,	0x340ce28b,	0x8b0f9462,	0x343aca87,	0x8b240e11,	0x3468aa76,	0x8b3899c6,
	0x34968250,	0x8b4d377c,	0x34c4520d,	0x8b61e733,	0x34f219a8,	0x8b76a8e4,	0x351fd918,	0x8b8b7c8f,
	0x354d9057,	0x8ba0622f,	0x357b3f5d,	0x8bb559c1,	0x35a8e625,	0x8bca6343,	0x35d684a6,	0x8bdf7eb0,
	0x36041ad9,	0x8bf4ac05,	0x3631a8b8,	0x8c09eb40,	0x365f2e3b,	0x8c1f3c5d,	0x368cab5c,	0x8c349f58,
	0x36ba2014,	0x8c4a142f,	0x36e78c5b,	0x8c5f9ade,	0x3714f02a,	0x8c753362,	0x37424b7b,	0x8c8addb7,
	0x376f9e46,	0x8ca099da,	0x379ce885,	0x8cb667c8,	0x37ca2a30,	0x8ccc477d,	0x37f76341,	0x8ce238f6,
	0x382493b0,	0x8cf83c30,	0x3851bb77,	0x8d0e5127,	0x387eda8e,	0x8d2477d8,	0x38abf0ef,	0x8d3ab03f,
	0x38d8fe93,	0x8d50fa59,	0x39060373,	0x8d675623,	0x3932ff87,	0x8d7dc399,	0x395ff2c9,	0x8d9442b8,
	0x398cdd32,	0x8daad37b,	0x39b9bebc,	0x8dc175e0,	0x39e6975e,	0x8dd829e4,	0x3a136712,	0x8deeef82,
	0x3a402dd2,	0x8e05c6b7,	0x3a6ceb96,	0x8e1caf80,	0x3a99a057,	0x8e33a9da,	0x3ac64c0f,	0x8e4ab5bf,
	0x3af2eeb7,	0x8e61d32e,	0x3b1f8848,	0x8e790222,	0x3b4c18ba,	0x8e904298,	0x3b78a007,	0x8ea7948c,
	0x3ba51e29,	0x8ebef7fb,	0x3bd19318,	0x8ed66ce1,	0x3bfdfecd,	0x8eedf33b,	0x3c2a6142,	0x8f058b04,
	0x3c56ba70,	0x8f1d343a,	0x3c830a50,	0x8f34eed8,	0x3caf50da,	0x8f4cbadb,	0x3cdb8e09,	0x8f649840,
	0x3d07c1d6,	0x8f7c8701,	0x3d33ec39,	0x8f94871d,	0x3d600d2c,	0x8fac988f,	0x3d8c24a8,	0x8fc4bb53,
	0x3db832a6,	0x8fdcef66,	0x3de4371f,	0x8ff534c4,	0x3e10320d,	0x900d8b69,	0x3e3c2369,	0x9025f352,
	0x3e680b2c,	0x903e6c7b,	0x3e93e950,	0x9056f6df,	0x3ebfbdcd,	0x906f927c,	0x3eeb889c,	0x90883f4d,
	0x3f1749b8,	0x90a0fd4e,	0x3f430119,	0x90b9cc7d,	0x3f6eaeb8,	0x90d2acd4,	0x3f9a5290,	0x90eb9e50,
	0x3fc5ec98,	0x9104a0ee,	0x3ff17cca,	0x911db4a9,	0x401d0321,	0x9136d97d,	0x40487f94,	0x91500f67,
	0x4073f21d,	0x91695663,	0x409f5ab6,	0x9182ae6d,	0x40cab958,	0x919c1781,	0x40f60dfb,	0x91b5919a,
	0x4121589b,	0x91cf1cb6,	0x414c992f,	0x91e8b8d0,	0x4177cfb1,	0x920265e4,	0x41a2fc1a,	0x921c23ef,
	0x41ce1e65,	0x9235f2ec,	0x41f93689,	0x924fd2d7,	0x42244481,	0x9269c3ac,	0x424f4845,	0x9283c568,
	0x427a41d0,	0x929dd806,	0x42a5311b,	0x92b7fb82,	0x42d0161e,	0x92d22fd9,	0x42faf0d4,	0x92ec7505,
	0x4325c135,	0x9306cb04,	0x4350873c,	0x932131d1,	0x437b42e1,	0x933ba968,	0x43a5f41e,	0x935631c5,
	0x43d09aed,	0x9370cae4,	0x43fb3746,	0x938b74c1,	0x4425c923,	0x93a62f57,	0x4450507e,	0x93c0faa3,
	0x447acd50,	0x93dbd6a0,	0x44a53f93,	0x93f6c34a,	0x44cfa740,	0x9411c09e,	0x44fa0450,	0x942cce96,
	0x452456bd,	0x9447ed2f,	0x454e9e80,	0x94631c65,	0x4578db93,	0x947e5c33,	0x45a30df0,	0x9499ac95,
	0x45cd358f,	0x94b50d87,	0x45f7526b,	0x94d07f05,	0x4621647d,	0x94ec010b,	0x464b6bbe,	0x95079394,
	0x46756828,	0x9523369c,	0x469f59b4,	0x953eea1e,	0x46c9405c,	0x955aae17,	0x46f31c1a,	0x95768283,
	0x471cece7,	0x9592675c,	0x4746b2bc,	0x95ae5c9f,	0x47706d93,	0x95ca6247,	0x479a1d67,	0x95e67850,
	0x47c3c22f,	0x96029eb6,	0x47ed5be6,	0x961ed574,	0x4816ea86,	0x963b1c86,	0x48406e08,	0x965773e7,
	0x4869e665,	0x9673db94,	0x48935397,	0x96905388,	0x48bcb599,	0x96acdbbe,	0x48e60c62,	0x96c97432,
	0x490f57ee,	0x96e61ce0,	0x49389836,	0x9702d5c3,	0x4961cd33,	0x971f9ed7,	0x498af6df,	0x973c7817,
	0x49b41533,	0x9759617f,	0x49dd282a,	0x97765b0a,	0x4a062fbd,	0x979364b5,	0x4a2f2be6,	0x97b07e7a,
	0x4a581c9e,	0x97cda855,	0x4a8101de,	0x97eae242,	0x4aa9dba2,	0x98082c3b,	0x4ad2a9e2,	0x9825863d,
	0x4afb6c98,	0x9842f043,	0x4b2423be,	0x98606a49,	0x4b4ccf4d,	0x987df449,	0x4b756f40,	0x989b8e40,
	0x4b9e0390,	0x98b93828,	0x4bc68c36,	0x98d6f1fe,	0x4bef092d,	0x98f4bbbc,	0x4c177a6e,	0x9912955f,
	0x4c3fdff4,	0x99307ee0,	0x4c6839b7,	0x994e783d,	0x4c9087b1,	0x996c816f,	0x4cb8c9dd,	0x998a9a74,
	0x4ce10034,	0x99a8c345,	0x4d092ab0,	0x99c6fbde,	0x4d31494b,	0x99e5443b,	0x4d595bfe,	0x9a039c57,
	0x4d8162c4,	0x9a22042d,	0x4da95d96,	0x9a407bb9,	0x4dd14c6e,	0x9a5f02f5,	0x4df92f46,	0x9a7d99de,
	0x4e210617,	0x9a9c406e,	0x4e48d0dd,	0x9abaf6a1,	0x4e708f8f,	0x9ad9bc71,	0x4e984229,	0x9af891db,
	0x4ebfe8a5,	0x9b1776da,	0x4ee782fb,	0x9b366b68,	0x4f0f1126,	0x9b556f81,	0x4f369320,	0x9b748320,
	0x4f5e08e3,	0x9b93a641,	0x4f857269,	0x9bb2d8de,	0x4faccfab,	0x9bd21af3,	0x4fd420a4,	0x9bf16c7a,
	0x4ffb654d,	0x9c10cd70,	0x50229da1,	0x9c303dcf,	0x5049c999,	0x9c4fbd93,	0x5070e92f,	0x9c6f4cb6,
	0x5097fc5e,	0x9c8eeb34,	0x50bf031f,	0x9cae9907,	0x50e5fd6d,	0x9cce562c,	0x510ceb40,	0x9cee229c,
	0x5133cc94,	0x9d0dfe54,	0x515aa162,	0x9d2de94d,	0x518169a5,	0x9d4de385,	0x51a82555,	0x9d6decf4,
	0x51ced46e,	0x9d8e0597,	0x51f576ea,	0x9dae2d68,	0x521c0cc2,	0x9dce6463,	0x524295f0,	0x9deeaa82,
	0x5269126e,	0x9e0effc1,	0x528f8238,	0x9e2f641b,	0x52b5e546,	0x9e4fd78a,	0x52dc3b92,	0x9e705a09,
	0x53028518,	0x9e90eb94,	0x5328c1d0,	0x9eb18c26,	0x534ef1b5,	0x9ed23bb9,	0x537514c2,	0x9ef2fa49,
	0x539b2af0,	0x9f13c7d0,	0x53c13439,	0x9f34a449,	0x53e73097,	0x9f558fb0,	0x540d2005,	0x9f7689ff,
	0x5433027d,	0x9f979331,	0x5458d7f9,	0x9fb8ab41,	0x547ea073,	0x9fd9d22a,	0x54a45be6,	0x9ffb07e7,
	0x54ca0a4b,	0xa01c4c73,	0x54efab9c,	0xa03d9fc8,	0x55153fd4,	0xa05f01e1,	0x553ac6ee,	0xa08072ba,
	0x556040e2,	0xa0a1f24d,	0x5585adad,	0xa0c38095,	0x55ab0d46,	0xa0e51d8c,	0x55d05faa,	0xa106c92f,
	0x55f5a4d2,	0xa1288376,	0x561adcb9,	0xa14a4c5e,	0x56400758,	0xa16c23e1,	0x566524aa,	0xa18e09fa,
	0x568a34a9,	0xa1affea3,	0x56af3750,	0xa1d201d7,	0x56d42c99,	0xa1f41392,	0x56f9147e,	0xa21633cd,
	0x571deefa,	0xa2386284,	0x5742bc06,	0xa25a9fb1,	0x57677b9d,	0xa27ceb4f,	0x578c2dba,	0xa29f4559,
	0x57b0d256,	0xa2c1adc9,	0x57d5696d,	0xa2e4249b,	0x57f9f2f8,	0xa306a9c8,	0x581e6ef1,	0xa3293d4b,
	0x5842dd54,	0xa34bdf20,	0x58673e1b,	0xa36e8f41,	0x588b9140,	0xa3914da8,	0x58afd6bd,	0xa3b41a50,
	0x58d40e8c,	0xa3d6f534,	0x58f838a9,	0xa3f9de4e,	0x591c550e,	0xa41cd599,	0x594063b5,	0xa43fdb10,
	0x59646498,	0xa462eeac,	0x598857b2,	0xa486106a,	0x59ac3cfd,	0xa4a94043,	0x59d01475,	0xa4cc7e32,
	0x59f3de12,	0xa4efca31,	0x5a1799d1,	0xa513243b,	0x5a3b47ab,	0xa5368c4b,	0x5a5ee79a,	0xa55a025b,
	0x5a82799a,	0xa57d8666,	0x5aa5fda5,	0xa5a11866,	0x5ac973b5,	0xa5c4b855,	0x5aecdbc5,	0xa5e8662f,
	0x5b1035cf,	0xa60c21ee,	0x5b3381ce,	0xa62feb8b,	0x5b56bfbd,	0xa653c303,	0x5b79ef96,	0xa677a84e,
	0x5b9d1154,	0xa69b9b68,	0x5bc024f0,	0xa6bf9c4b,	0x5be32a67,	0xa6e3aaf2,	0x5c0621b2,	0xa707c757,
	0x5c290acc,	0xa72bf174,	0x5c4be5b0,	0xa7502943,	0x5c6eb258,	0xa7746ec0,	0x5c9170bf,	0xa798c1e5,
	0x5cb420e0,	0xa7bd22ac,	0x5cd6c2b5,	0xa7e1910f,	0x5cf95638,	0xa8060d08,	0x5d1bdb65,	0xa82a9693,
	0x5d3e5237,	0xa84f2daa,	0x5d60baa7,	0xa873d246,	0x5d8314b1,	0xa8988463,	0x5da5604f,	0xa8bd43fa,
	0x5dc79d7c,	0xa8e21106,	0x5de9cc33,	0xa906eb82,	0x5e0bec6e,	0xa92bd367,	0x5e2dfe29,	0xa950c8b0,
	0x5e50015d,	0xa975cb57,	0x5e71f606,	0xa99adb56,	0x5e93dc1f,	0xa9bff8a8,	0x5eb5b3a2,	0xa9e52347,
	0x5ed77c8a,	0xaa0a5b2e,	0x5ef936d1,	0xaa2fa056,	0x5f1ae274,	0xaa54f2ba,	0x5f3c7f6b,	0xaa7a5253,
	0x5f5e0db3,	0xaa9fbf1e,	0x5f7f8d46,	0xaac53912,	0x5fa0fe1f,	0xaaeac02c,	0x5fc26038,	0xab105464,
	0x5fe3b38d,	0xab35f5b5,	0x6004f819,	0xab5ba41a,	0x60262dd6,	0xab815f8d,	0x604754bf,	0xaba72807,
	0x60686ccf,	0xabccfd83,	0x60897601,	0xabf2dffb,	0x60aa7050,	0xac18cf69,	0x60cb5bb7,	0xac3ecbc7,
	0x60ec3830,	0xac64d510,	0x610d05b7,	0xac8aeb3e,	0x612dc447,	0xacb10e4b,	0x614e73da,	0xacd73e30,
	0x616f146c,	0xacfd7ae8,	0x618fa5f7,	0xad23c46e,	0x61b02876,	0xad4a1aba,	0x61d09be5,	0xad707dc8,
	0x61f1003f,	0xad96ed92,	0x6211557e,	0xadbd6a10,	0x62319b9d,	0xade3f33e,	0x6251d298,	0xae0a8916,
	0x6271fa69,	0xae312b92,	0x6292130c,	0xae57daab,	0x62b21c7b,	0xae7e965b,	0x62d216b3,	0xaea55e9e,
	0x62f201ac,	0xaecc336c,	0x6311dd64,	0xaef314c0,	0x6331a9d4,	0xaf1a0293,	0x635166f9,	0xaf40fce1,
	0x637114cc,	0xaf6803a2,	0x6390b34a,	0xaf8f16d1,	0x63b0426d,	0xafb63667,	0x63cfc231,	0xafdd625f,
	0x63ef3290,	0xb0049ab3,	0x640e9386,	0xb02bdf5c,	0x642de50d,	0xb0533055,	0x644d2722,	0xb07a8d97,
	0x646c59bf,	0xb0a1f71d,	0x648b7ce0,	0xb0c96ce0,	0x64aa907f,	0xb0f0eeda,	0x64c99498,	0xb1187d05,
	0x64e88926,	0xb140175b,	0x65076e25,	0xb167bdd7,	0x6526438f,	0xb18f7071,	0x6545095f,	0xb1b72f23,
	0x6563bf92,	0xb1def9e9,	0x65826622,	0xb206d0ba,	0x65a0fd0b,	0xb22eb392,	0x65bf8447,	0xb256a26a,
	0x65ddfbd3,	0xb27e9d3c,	0x65fc63a9,	0xb2a6a402,	0x661abbc5,	0xb2ceb6b5,	0x66390422,	0xb2f6d550,
	0x66573cbb,	0xb31effcc,	0x6675658c,	0xb3473623,	0x66937e91,	0xb36f784f,	0x66b187c3,	0xb397c649,
	0x66cf8120,	0xb3c0200c,	0x66ed6aa1,	0xb3e88592,	0x670b4444,	0xb410f6d3,	0x67290e02,	0xb43973ca,
	0x6746c7d8,	0xb461fc70,	0x676471c0,	0xb48a90c0,	0x67820bb7,	0xb4b330b3,	0x679f95b7,	0xb4dbdc42,
	0x67bd0fbd,	0xb5049368,	0x67da79c3,	0xb52d561e,	0x67f7d3c5,	0xb556245e,	0x68151dbe,	0xb57efe22,
	0x683257ab,	0xb5a7e362,	0x684f8186,	0xb5d0d41a,	0x686c9b4b,	0xb5f9d043,	0x6889a4f6,	0xb622d7d6,
	0x68a69e81,	0xb64beacd,	0x68c387e9,	0xb6750921,	0x68e06129,	0xb69e32cd,	0x68fd2a3d,	0xb6c767ca,
	0x6919e320,	0xb6f0a812,	0x69368bce,	0xb719f39e,	0x69532442,	0xb7434a67,	0x696fac78,	0xb76cac69,
	0x698c246c,	0xb796199b,	0x69a88c19,	0xb7bf91f8,	0x69c4e37a,	0xb7e9157a,	0x69e12a8c,	0xb812a41a,
	0x69fd614a,	0xb83c3dd1,	0x6a1987b0,	0xb865e299,	0x6a359db9,	0xb88f926d,	0x6a51a361,	0xb8b94d44,
	0x6a6d98a4,	0xb8e31319,	0x6a897d7d,	0xb90ce3e6,	0x6aa551e9,	0xb936bfa4,	0x6ac115e2,	0xb960a64c,
	0x6adcc964,	0xb98a97d8,	0x6af86c6c,	0xb9b49442,	0x6b13fef5,	0xb9de9b83,	0x6b2f80fb,	0xba08ad95,
	0x6b4af279,	0xba32ca71,	0x6b66536b,	0xba5cf210,	0x6b81a3cd,	0xba87246d,	0x6b9ce39b,	0xbab16180,
	0x6bb812d1,	0xbadba943,	0x6bd3316a,	0xbb05fbb0,	0x6bee3f62,	0xbb3058c0,	0x6c093cb6,	0xbb5ac06d,
	0x6c242960,	0xbb8532b0,	0x6c3f055d,	0xbbafaf82,	0x6c59d0a9,	0xbbda36dd,	0x6c748b3f,	0xbc04c8ba,
	0x6c8f351c,	0xbc2f6513,	0x6ca9ce3b,	0xbc5a0be2,	0x6cc45698,	0xbc84bd1f,	0x6cdece2f,	0xbcaf78c4,
	0x6cf934fc,	0xbcda3ecb,	0x6d138afb,	0xbd050f2c,	0x6d2dd027,	0xbd2fe9e2,	0x6d48047e,	0xbd5acee5,
	0x6d6227fa,	0xbd85be30,	0x6d7c3a98,	0xbdb0b7bb,	0x6d963c54,	0xbddbbb7f,	0x6db02d29,	0xbe06c977,
	0x6dca0d14,	0xbe31e19b,	0x6de3dc11,	0xbe5d03e6,	0x6dfd9a1c,	0xbe88304f,	0x6e174730,	0xbeb366d1,
	0x6e30e34a,	0xbedea765,	0x6e4a6e66,	0xbf09f205,	0x6e63e87f,	0xbf3546a8,	0x6e7d5193,	0xbf60a54a,
	0x6e96a99d,	0xbf8c0de3,	0x6eaff099,	0xbfb7806c,	0x6ec92683,	0xbfe2fcdf,	0x6ee24b57,	0xc00e8336,
	0x6efb5f12,	0xc03a1368,	0x6f1461b0,	0xc065ad70,	0x6f2d532c,	0xc0915148,	0x6f463383,	0xc0bcfee7,
	0x6f5f02b2,	0xc0e8b648,	0x6f77c0b3,	0xc1147764,	0x6f906d84,	0xc1404233,	0x6fa90921,	0xc16c16b0,
	0x6fc19385,	0xc197f4d4,	0x6fda0cae,	0xc1c3dc97,	0x6ff27497,	0xc1efcdf3,	0x700acb3c,	0xc21bc8e1,
	0x7023109a,	0xc247cd5a,	0x703b44ad,	0xc273db58,	0x70536771,	0xc29ff2d4,	0x706b78e3,	0xc2cc13c7,
	0x708378ff,	0xc2f83e2a,	0x709b67c0,	0xc32471f7,	0x70b34525,	0xc350af26,	0x70cb1128,	0xc37cf5b0,
	0x70e2cbc6,	0xc3a94590,	0x70fa74fc,	0xc3d59ebe,	0x71120cc5,	0xc4020133,	0x7129931f,	0xc42e6ce8,
	0x71410805,	0xc45ae1d7,	0x71586b74,	0xc4875ff9,	0x716fbd68,	0xc4b3e746,	0x7186fdde,	0xc4e077b8,
	0x719e2cd2,	0xc50d1149,	0x71b54a41,	0xc539b3f1,	0x71cc5626,	0xc5665fa9,	0x71e35080,	0xc593146a,
	0x71fa3949,	0xc5bfd22e,	0x7211107e,	0xc5ec98ee,	0x7227d61c,	0xc61968a2,	0x723e8a20,	0xc6464144,
	0x72552c85,	0xc67322ce,	0x726bbd48,	0xc6a00d37,	0x72823c67,	0xc6cd0079,	0x7298a9dd,	0xc6f9fc8d,
	0x72af05a7,	0xc727016d,	0x72c54fc1,	0xc7540f11,	0x72db8828,	0xc7812572,	0x72f1aed9,	0xc7ae4489,
	0x7307c3d0,	0xc7db6c50,	0x731dc70a,	0xc8089cbf,	0x7333b883,	0xc835d5d0,	0x73499838,	0xc863177b,
	0x735f6626,	0xc89061ba,	0x73752249,	0xc8bdb485,	0x738acc9e,	0xc8eb0fd6,	0x73a06522,	0xc91873a5,
	0x73b5ebd1,	0xc945dfec,	0x73cb60a8,	0xc97354a4,	0x73e0c3a3,	0xc9a0d1c5,	0x73f614c0,	0xc9ce5748,
	0x740b53fb,	0xc9fbe527,	0x74208150,	0xca297b5a,	0x74359cbd,	0xca5719db,	0x744aa63f,	0xca84c0a3,
	0x745f9dd1,	0xcab26fa9,	0x74748371,	0xcae026e8,	0x7489571c,	0xcb0de658,	0x749e18cd,	0xcb3badf3,
	0x74b2c884,	0xcb697db0,	0x74c7663a,	0xcb97558a,	0x74dbf1ef,	0xcbc53579,	0x74f06b9e,	0xcbf31d75,
	0x7504d345,	0xcc210d79,	0x751928e0,	0xcc4f057c,	0x752d6c6c,	0xcc7d0578,	0x75419de7,	0xccab0d65,
	0x7555bd4c,	0xccd91d3d,	0x7569ca99,	0xcd0734f9,	0x757dc5ca,	0xcd355491,	0x7591aedd,	0xcd637bfe,
	0x75a585cf,	0xcd91ab39,	0x75b94a9c,	0xcdbfe23a,	0x75ccfd42,	0xcdee20fc,	0x75e09dbd,	0xce1c6777,
	0x75f42c0b,	0xce4ab5a2,	0x7607a828,	0xce790b79,	0x761b1211,	0xcea768f2,	0x762e69c4,	0xced5ce08,
	0x7641af3d,	0xcf043ab3,	0x7654e279,	0xcf32aeeb,	0x76680376,	0xcf612aaa,	0x767b1231,	0xcf8fade9,
	0x768e0ea6,	0xcfbe389f,	0x76a0f8d2,	0xcfeccac7,	0x76b3d0b4,	0xd01b6459,	0x76c69647,	0xd04a054e,
	0x76d94989,	0xd078ad9e,	0x76ebea77,	0xd0a75d42,	0x76fe790e,	0xd0d61434,	0x7710f54c,	0xd104d26b,
	0x77235f2d,	0xd13397e2,	0x7735b6af,	0xd1626490,	0x7747fbce,	0xd191386e,	0x775a2e89,	0xd1c01375,
	0x776c4edb,	0xd1eef59e,	0x777e5cc3,	0xd21ddee2,	0x7790583e,	0xd24ccf39,	0x77a24148,	0xd27bc69c,
	0x77b417df,	0xd2aac504,	0x77c5dc01,	0xd2d9ca6a,	0x77d78daa,	0xd308d6c7,	0x77e92cd9,	0xd337ea12,
	0x77fab989,	0xd3670446,	0x780c33b8,	0xd396255a,	0x781d9b65,	0xd3c54d47,	0x782ef08b,	0xd3f47c06,
	0x78403329,	0xd423b191,	0x7851633b,	0xd452eddf,	0x786280bf,	0xd48230e9,	0x78738bb3,	0xd4b17aa8,
	0x78848414,	0xd4e0cb15,	0x789569df,	0xd5102228,	0x78a63d11,	0xd53f7fda,	0x78b6fda8,	0xd56ee424,
	0x78c7aba2,	0xd59e4eff,	0x78d846fb,	0xd5cdc062,	0x78e8cfb2,	0xd5fd3848,	0x78f945c3,	0xd62cb6a8,
	0x7909a92d,	0xd65c3b7b,	0x7919f9ec,	0xd68bc6ba,	0x792a37fe,	0xd6bb585e,	0x793a6361,	0xd6eaf05f,
	0x794a7c12,	0xd71a8eb5,	0x795a820e,	0xd74a335b,	0x796a7554,	0xd779de47,	0x797a55e0,	0xd7a98f73,
	0x798a23b1,	0xd7d946d8,	0x7999dec4,	0xd809046e,	0x79a98715,	0xd838c82d,	0x79b91ca4,	0xd868920f,
	0x79c89f6e,	0xd898620c,	0x79d80f6f,	0xd8c8381d,	0x79e76ca7,	0xd8f81439,	0x79f6b711,	0xd927f65b,
	0x7a05eead,	0xd957de7a,	0x7a151378,	0xd987cc90,	0x7a24256f,	0xd9b7c094,	0x7a332490,	0xd9e7ba7f,
	0x7a4210d8,	0xda17ba4a,	0x7a50ea47,	0xda47bfee,	0x7a5fb0d8,	0xda77cb63,	0x7a6e648a,	0xdaa7dca1,
	0x7a7d055b,	0xdad7f3a2,	0x7a8b9348,	0xdb08105e,	0x7a9a0e50,	0xdb3832cd,	0x7aa8766f,	0xdb685ae9,
	0x7ab6cba4,	0xdb9888a8,	0x7ac50dec,	0xdbc8bc06,	0x7ad33d45,	0xdbf8f4f8,	0x7ae159ae,	0xdc293379,
	0x7aef6323,	0xdc597781,	0x7afd59a4,	0xdc89c109,	0x7b0b3d2c,	0xdcba1008,	0x7b190dbc,	0xdcea6478,
	0x7b26cb4f,	0xdd1abe51,	0x7b3475e5,	0xdd4b1d8c,	0x7b420d7a,	0xdd7b8220,	0x7b4f920e,	0xddabec08,
	0x7b5d039e,	0xdddc5b3b,	0x7b6a6227,	0xde0ccfb1,	0x7b77ada8,	0xde3d4964,	0x7b84e61f,	0xde6dc84b,
	0x7b920b89,	0xde9e4c60,	0x7b9f1de6,	0xdeced59b,	0x7bac1d31,	0xdeff63f4,	0x7bb9096b,	0xdf2ff764,
	0x7bc5e290,	0xdf608fe4,	0x7bd2a89e,	0xdf912d6b,	0x7bdf5b94,	0xdfc1cff3,	0x7bebfb70,	0xdff27773,
	0x7bf88830,	0xe02323e5,	0x7c0501d2,	0xe053d541,	0x7c116853,	0xe0848b7f,	0x7c1dbbb3,	0xe0b54698,
	0x7c29fbee,	0xe0e60685,	0x7c362904,	0xe116cb3d,	0x7c4242f2,	0xe14794ba,	0x7c4e49b7,	0xe17862f3,
	0x7c5a3d50,	0xe1a935e2,	0x7c661dbc,	0xe1da0d7e,	0x7c71eaf9,	0xe20ae9c1,	0x7c7da505,	0xe23bcaa2,
	0x7c894bde,	0xe26cb01b,	0x7c94df83,	0xe29d9a23,	0x7ca05ff1,	0xe2ce88b3,	0x7cabcd28,	0xe2ff7bc3,
	0x7cb72724,	0xe330734d,	0x7cc26de5,	0xe3616f48,	0x7ccda169,	0xe3926fad,	0x7cd8c1ae,	0xe3c37474,
	0x7ce3ceb2,	0xe3f47d96,	0x7ceec873,	0xe4258b0a,	0x7cf9aef0,	0xe4569ccb,	0x7d048228,	0xe487b2d0,
	0x7d0f4218,	0xe4b8cd11,	0x7d19eebf,	0xe4e9eb87,	0x7d24881b,	0xe51b0e2a,	0x7d2f0e2b,	0xe54c34f3,
	0x7d3980ec,	0xe57d5fda,	0x7d43e05e,	0xe5ae8ed8,	0x7d4e2c7f,	0xe5dfc1e5,	0x7d58654d,	0xe610f8f9,
	0x7d628ac6,	0xe642340d,	0x7d6c9ce9,	0xe6737319,	0x7d769bb5,	0xe6a4b616,	0x7d808728,	0xe6d5fcfc,
	0x7d8a5f40,	0xe70747c4,	0x7d9423fc,	0xe7389665,	0x7d9dd55a,	0xe769e8d8,	0x7da77359,	0xe79b3f16,
	0x7db0fdf8,	0xe7cc9917,	0x7dba7534,	0xe7fdf6d4,	0x7dc3d90d,	0xe82f5844,	0x7dcd2981,	0xe860bd61,
	0x7dd6668f,	0xe8922622,	0x7ddf9034,	0xe8c39280,	0x7de8a670,	0xe8f50273,	0x7df1a942,	0xe92675f4,
	0x7dfa98a8,	0xe957ecfb,	0x7e0374a0,	0xe9896781,	0x7e0c3d29,	0xe9bae57d,	0x7e14f242,	0xe9ec66e8,
	0x7e1d93ea,	0xea1debbb,	0x7e26221f,	0xea4f73ee,	0x7e2e9cdf,	0xea80ff7a,	0x7e37042a,	0xeab28e56,
	0x7e3f57ff,	0xeae4207a,	0x7e47985b,	0xeb15b5e1,	0x7e4fc53e,	0xeb474e81,	0x7e57dea7,	0xeb78ea52,
	0x7e5fe493,	0xebaa894f,	0x7e67d703,	0xebdc2b6e,	0x7e6fb5f4,	0xec0dd0a8,	0x7e778166,	0xec3f78f6,
	0x7e7f3957,	0xec71244f,	0x7e86ddc6,	0xeca2d2ad,	0x7e8e6eb2,	0xecd48407,	0x7e95ec1a,	0xed063856,
	0x7e9d55fc,	0xed37ef91,	0x7ea4ac58,	0xed69a9b3,	0x7eabef2c,	0xed9b66b2,	0x7eb31e78,	0xedcd2687,
	0x7eba3a39,	0xedfee92b,	0x7ec14270,	0xee30ae96,	0x7ec8371a,	0xee6276bf,	0x7ecf1837,	0xee9441a0,
	0x7ed5e5c6,	0xeec60f31,	0x7edc9fc6,	0xeef7df6a,	0x7ee34636,	0xef29b243,	0x7ee9d914,	0xef5b87b5,
	0x7ef05860,	0xef8d5fb8,	0x7ef6c418,	0xefbf3a45,	0x7efd1c3c,	0xeff11753,	0x7f0360cb,	0xf022f6da,
	0x7f0991c4,	0xf054d8d5,	0x7f0faf25,	0xf086bd39,	0x7f15b8ee,	0xf0b8a401,	0x7f1baf1e,	0xf0ea8d24,
	0x7f2191b4,	0xf11c789a,	0x7f2760af,	0xf14e665c,	0x7f2d1c0e,	0xf1805662,	0x7f32c3d1,	0xf1b248a5,
	0x7f3857f6,	0xf1e43d1c,	0x7f3dd87c,	0xf21633c0,	0x7f434563,	0xf2482c8a,	0x7f489eaa,	0xf27a2771,
	0x7f4de451,	0xf2ac246e,	0x7f531655,	0xf2de2379,	0x7f5834b7,	0xf310248a,	0x7f5d3f75,	0xf342279b,
	0x7f62368f,	0xf3742ca2,	0x7f671a05,	0xf3a63398,	0x7f6be9d4,	0xf3d83c77,	0x7f70a5fe,	0xf40a4735,
	0x7f754e80,	0xf43c53cb,	0x7f79e35a,	0xf46e6231,	0x7f7e648c,	0xf4a07261,	0x7f82d214,	0xf4d28451,
	0x7f872bf3,	0xf50497fb,	0x7f8b7227,	0xf536ad56,	0x7f8fa4b0,	0xf568c45b,	0x7f93c38c,	0xf59add02,
	0x7f97cebd,	0xf5ccf743,	0x7f9bc640,	0xf5ff1318,	0x7f9faa15,	0xf6313077,	0x7fa37a3c,	0xf6634f59,
	0x7fa736b4,	0xf6956fb7,	0x7faadf7c,	0xf6c79188,	0x7fae7495,	0xf6f9b4c6,	0x7fb1f5fc,	0xf72bd967,
	0x7fb563b3,	0xf75dff66,	0x7fb8bdb8,	0xf79026b9,	0x7fbc040a,	0xf7c24f59,	0x7fbf36aa,	0xf7f4793e,
	0x7fc25596,	0xf826a462,	0x7fc560cf,	0xf858d0bb,	0x7fc85854,	0xf88afe42,	0x7fcb3c23,	0xf8bd2cef,
	0x7fce0c3e,	0xf8ef5cbb,	0x7fd0c8a3,	0xf9218d9e,	0x7fd37153,	0xf953bf91,	0x7fd6064c,	0xf985f28a,
	0x7fd8878e,	0xf9b82684,	0x7fdaf519,	0xf9ea5b75,	0x7fdd4eec,	0xfa1c9157,	0x7fdf9508,	0xfa4ec821,
	0x7fe1c76b,	0xfa80ffcb,	0x7fe3e616,	0xfab3384f,	0x7fe5f108,	0xfae571a4,	0x7fe7e841,	0xfb17abc2,
	0x7fe9cbc0,	0xfb49e6a3,	0x7feb9b85,	0xfb7c223d,	0x7fed5791,	0xfbae5e89,	0x7feeffe1,	0xfbe09b80,
	0x7ff09478,	0xfc12d91a,	0x7ff21553,	0xfc45174e,	0x7ff38274,	0xfc775616,	0x7ff4dbd9,	0xfca9956a,
	0x7ff62182,	0xfcdbd541,	0x7ff75370,	0xfd0e1594,	0x7ff871a2,	0xfd40565c,	0x7ff97c18,	0xfd729790,
	0x7ffa72d1,	0xfda4d929,	0x7ffb55ce,	0xfdd71b1e,	0x7ffc250f,	0xfe095d69,	0x7ffce093,	0xfe3ba002,
	0x7ffd885a,	0xfe6de2e0,	0x7ffe1c65,	0xfea025fd,	0x7ffe9cb2,	0xfed2694f,	0x7fff0943,	0xff04acd0,
	0x7fff6216,	0xff36f078,	0x7fffa72c,	0xff69343f,	0x7fffd886,	0xff9b781d,	0x7ffff621,	0xffcdbc0b

};

/**   
*   
* @brief  Initialization function for the Q31 CFFT/CIFFT.  
* @param[in,out] *S             points to an instance of the Q31 CFFT/CIFFT structure.  
* @param[in]     fftLen         length of the FFT.  
* @param[in]     ifftFlag       flag that selects forward (ifftFlag=0) or inverse (ifftFlag=1) transform.  
* @param[in]     bitReverseFlag flag that enables (bitReverseFlag=1) or disables (bitReverseFlag=0) bit reversal of output.  
* @return        The function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLen</code> is not a supported value.  
*   
* \par Description:  
* \par   
* The parameter <code>ifftFlag</code> controls whether a forward or inverse transform is computed.   
* Set(=1) ifftFlag for calculation of CIFFT otherwise  CFFT is calculated  
* \par   
* The parameter <code>bitReverseFlag</code> controls whether output is in normal order or bit reversed order.   
* Set(=1) bitReverseFlag for output to be in normal order otherwise output is in bit reversed order.   
* \par   
* The parameter <code>fftLen</code>	Specifies length of CFFT/CIFFT process. Supported FFT Lengths are 16, 64, 256, 1024.   
* \par   
* This Function also initializes Twiddle factor table pointer and Bit reversal table pointer.   
*/

arm_status arm_cfft_radix4_init_q31(
  arm_cfft_radix4_instance_q31 * S,
  uint16_t fftLen,
  uint8_t ifftFlag,
  uint8_t bitReverseFlag)
{
  /*  Initialise the default arm status */
  arm_status status = ARM_MATH_SUCCESS;
  /*  Initialise the FFT length */
  S->fftLen = fftLen;
  /*  Initialise the Twiddle coefficient pointer */
  S->pTwiddle = (q31_t *) twiddleCoefQ31;
  /*  Initialise the Flag for selection of CFFT or CIFFT */
  S->ifftFlag = ifftFlag;
  /*  Initialise the Flag for calculation Bit reversal or not */
  S->bitReverseFlag = bitReverseFlag;

  /*  Initializations of Instance structure depending on the FFT length */
  switch (S->fftLen)
  {

  /*  Initializations of structure parameters for 4096 point FFT */
  case 4096u:
    /*  Initialise the twiddle coef modifier value */
    S->twidCoefModifier = 1u;
    /*  Initialise the bit reversal table modifier */
    S->bitRevFactor = 1u;
    /*  Initialise the bit reversal table pointer */
    S->pBitRevTable = armBitRevTable;
    break;

    /*  Initializations of structure parameters for 1024 point FFT */
  case 1024u:
    /*  Initialise the twiddle coef modifier value */
    S->twidCoefModifier = 4u;
    /*  Initialise the bit reversal table modifier */
    S->bitRevFactor = 4u;
    /*  Initialise the bit reversal table pointer */
    S->pBitRevTable = (uint16_t *) & armBitRevTable[3];
    break;

  case 256u:
    /*  Initializations of structure parameters for 256 point FFT */
    S->twidCoefModifier = 16u;
    S->bitRevFactor = 16u;
    S->pBitRevTable = (uint16_t *) & armBitRevTable[15];
    break;

  case 64u:
    /*  Initializations of structure parameters for 64 point FFT */
    S->twidCoefModifier = 64u;
    S->bitRevFactor = 64u;
    S->pBitRevTable = &armBitRevTable[63];
    break;

  case 16u:
    /*  Initializations of structure parameters for 16 point FFT */
    S->twidCoefModifier = 256u;
    S->bitRevFactor = 256u;
    S->pBitRevTable = &armBitRevTable[255];
    break;

  default:
    /*  Reporting argument error if fftSize is not valid value */
    status = ARM_MATH_ARGUMENT_ERROR;
    break;
  }

  return (status);
}

/**   
 * @} end of CFFT_CIFFT group   
 */
